#pragma once

#include <Window.h>
#include <View.h>
#include <Bitmap.h>
#include <PopUpMenu.h>
#include <MenuItem.h>
#include <Message.h>
#include <Messenger.h>
#include <Region.h>
#include <Screen.h>

#include <cmath>
#include <cstdio>
#include <string>
#include <algorithm>
#include <vector>
#include <OS.h>

extern "C" {
#include "fft.h"
}

#include "WinampSkin.h"
class SkinDock {
public:
    static constexpr int kSnapThresh = 13;  // pixels — same as qmmp

    void AddWindow(BWindow* w) {
        fWindows.push_back(w);
        fDocked.push_back(false);
        fDelta.push_back(BPoint(0, 0));
    }

    // Move window `mv` to proposed screen position `npos`.
    // Snaps against all other windows and the screen edge.
    // Returns the snapped position and moves all docked followers.
    BPoint Move(BWindow* mv, BPoint npos) {
        if (fWindows.empty()) return npos;

        if (mv == fWindows[0]) {
            // Main window: carry docked followers, snap against non-docked
            for (size_t i = 1; i < fWindows.size(); ++i) {
                if (!fDocked[i]) {
                    if (fWindows[i]->IsHidden()) continue;
                    npos = Snap(npos, mv, fWindows[i]);
                } else {
                    // Docked follower: compute candidate position, snap it
                    BPoint pos = npos + fDelta[i];
                    for (size_t j = 1; j < fWindows.size(); ++j) {
                        if (!fDocked[j] && !fWindows[j]->IsHidden())
                            pos = Snap(pos, fWindows[i], fWindows[j]);
                    }
                    npos = pos - fDelta[i];
                }
            }
            npos = SnapDesktop(npos, mv);
            // Move all docked followers
            for (size_t i = 1; i < fWindows.size(); ++i) {
                if (fDocked[i]) {
                    BPoint pos = SnapDesktop(npos + fDelta[i], fWindows[i]);
                    if (fWindows[i]->LockLooper()) {
                        fWindows[i]->MoveTo(pos);
                        fWindows[i]->UnlockLooper();
                    }
                    npos = pos - fDelta[i];
                }
            }
        } else {
            // Non-main window: snap against all others and screen
            for (size_t i = 0; i < fWindows.size(); ++i) {
                fDocked[i] = false;
                if (fWindows[i] != mv && !fWindows[i]->IsHidden()) {
                    npos = Snap(npos, mv, fWindows[i]);
                    npos = SnapDesktop(npos, mv);
                }
            }
        }
        return npos;
    }

    // Call after mouse release: recompute which windows are docked to main.
    void UpdateDock() {
        if (fWindows.size() < 2) return;
        BWindow* main = fWindows[0];
        for (size_t j = 1; j < fWindows.size(); ++j)
            fDocked[j] = IsDocked(main, fWindows[j]);

        // Transitive: if A docked to main, check if B docked to A
        for (size_t j = 1; j < fWindows.size(); ++j) {
            if (fDocked[j]) {
                for (size_t i = 1; i < fWindows.size(); ++i) {
                    if (!fDocked[i] && !fWindows[i]->IsHidden())
                        if (IsDocked(fWindows[j], fWindows[i]))
                            fDocked[i] = true;
                }
            }
        }

        // Record current offsets from main
        BPoint mainPos = fWindows[0]->Frame().LeftTop();
        for (size_t i = 1; i < fWindows.size(); ++i)
            fDelta[i] = fWindows[i]->Frame().LeftTop() - mainPos;
    }

    // Recompute deltas (call after programmatic repositioning)
    void RecalcDeltas() { UpdateDock(); }

private:
    BPoint SnapDesktop(BPoint npos, BWindow* mv) {
        BScreen screen;
        BRect sr = screen.Frame();
        BRect wf = mv->Frame();
        float w = wf.Width(), h = wf.Height();
        int t = kSnapThresh;
        if (std::abs(npos.x - sr.left)              < t) npos.x = sr.left;
        if (std::abs(npos.y - sr.top)               < t) npos.y = sr.top;
        if (std::abs(npos.x + w - sr.right)         < t) npos.x = sr.right  - w;
        if (std::abs(npos.y + h - sr.bottom)        < t) npos.y = sr.bottom - h;
        return npos;
    }

    // Snap moving window `mv` (at proposed pos `npos`) against stationary `st`.
    // Direct port of qmmp Dock::snap().
    BPoint Snap(BPoint npos, BWindow* mv, BWindow* st) {
        BRect sf = st->Frame();
        float mw = mv->Frame().Width(), mh = mv->Frame().Height();
        float sw = sf.Width(),          sh = sf.Height();
        float sx = sf.left,             sy = sf.top;
        int   t  = kSnapThresh;

        float nx = npos.x - sx;
        float ny = std::abs(npos.y - sy + mh);
        if (std::abs(nx) < t && ny < t)           npos.x = sx;
        if (ny < t && nx > -mw && nx < sw)        npos.y = sy - mh;
        nx = std::abs(npos.x + mw - sx - sw);
        if (nx < t && ny < t)                     npos.x = sx + sw - mw;

        nx = npos.x - sx;
        ny = std::abs(npos.y - sy - sh);
        if (std::abs(nx) < t && ny < t)           npos.x = sx;
        if (ny < t && nx > -mw && nx < sw)        npos.y = sy + sh;
        nx = std::abs(npos.x + mw - sx - sw);
        if (nx < t && ny < t)                     npos.x = sx + sw - mw;

        // ── left: mv right edge near st left ──────────────────────────────────
        nx = std::abs(npos.x - sx + mw);
        float ny2 = npos.y - sy;
        if (nx < t && std::abs(ny2) < t)          npos.y = sy;
        if (nx < t && ny2 > -mh && ny2 < sh)      npos.x = sx - mw;
        ny2 = std::abs(npos.y + mh - sy - sh);
        if (nx < t && ny2 < t)                    npos.y = sy + sh - mh;

        // ── right: mv left edge near st right ─────────────────────────────────
        nx = std::abs(npos.x - sx - sw);
        ny2 = npos.y - sy;
        if (nx < t && std::abs(ny2) < t)          npos.y = sy;
        if (nx < t && ny2 > -mh && ny2 < sh)      npos.x = sx + sw;
        ny2 = std::abs(npos.y + mh - sy - sh);
        if (nx < t && ny2 < t)                    npos.y = sy + sh - mh;

        return npos;
    }

    // Direct port of qmmp Dock::isDocked() Thank you!
    bool IsDocked(BWindow* mv, BWindow* st) {
        if (mv->IsHidden() || st->IsHidden()) return false;
        BRect mf = mv->Frame(), sf = st->Frame();
        float mw = mf.Width(), mh = mf.Height();
        float sw = sf.Width(), sh = sf.Height();
        float mx = mf.left, my = mf.top, sx = sf.left, sy = sf.top;

        // above
        float nx = mx - sx;
        float ny = std::abs(my - sy + mh);
        if (ny < 2 && nx > -mw && nx < sw) return true;
        // below
        ny = std::abs(my - sy - sh);
        if (ny < 2 && nx > -mw && nx < sw) return true;
        // left
        nx = std::abs(mx - sx + mw);
        float ny2 = my - sy;
        if (nx < 2 && ny2 > -mh && ny2 < sh) return true;
        // right
        nx = std::abs(mx - sx - sw);
        if (nx < 2 && ny2 > -mh && ny2 < sh) return true;
        return false;
    }

    std::vector<BWindow*> fWindows;
    std::vector<bool>     fDocked;
    std::vector<BPoint>   fDelta;
};

namespace WA {
    // Window sizes
    static constexpr int kMainW  = 275, kMainH  = 116;
    static constexpr int kPlW    = 275, kPlH    = 116;  // base; grows in 25×29 increments

    // Player titlebar (TITLEBAR.BMP)
    static constexpr int kTitleH  = 14;
    static constexpr int kMenuBX  = 6,  kMenuBY  = 3;   // BT_MENU button pos
    static constexpr int kMinBX   = 244,kMinBY   = 3;
    static constexpr int kCloseBX = 264,kCloseBY = 3;
    static constexpr int kBtnSz   = 9;                   // titlebar button size

    // Transport buttons (CBUTTONS.BMP)
    static constexpr int kBtnY   = 88;
    static constexpr int kPrevX  = 16, kPlayX  = 39, kPauseX = 62;
    static constexpr int kStopX  = 85, kNextX  = 108;
    static constexpr int kCBtnW  = 23, kCBtnH  = 18;

    // Position bar (POSBAR.BMP): bar=248px, knob=29px
    static constexpr int kPosBarX = 16, kPosBarY = 72;
    static constexpr int kPosBarW = 248, kPosKnobW = 29;

    // Volume bar (VOLUME.BMP) — qmmp SkinnedVolumeBar
    // Widget placed at (107, 57) in skin pixels
    static constexpr int kVolX = 107, kVolY = 57;
    // Balance bar: position (177,57), 38x13 strip, 25px knob travel (-100..+100)
    static constexpr int kBalX        = 177;   // same Y as volume (57)
    static constexpr int kBalStripW   = 38;    // strip width (px, 1x)
    static constexpr int kBalStripH   = 13;    // strip height
    static constexpr int kBalUsable   = 25;    // knob travel = 38-13
    static constexpr int kBalKnobY    = 422;   // y of knob sprite in balance.bmp
    static constexpr int kBalKnobW    = 14;    // knob width (same as vol knob)
    // Widget width = 68*scale (= kVolStripW*scale from WinampSkin.h)
    // Usable drag range = (68-18)*scale = 50*scale
    static constexpr int kVolWidgetW = 68;    // strip width in skin pixels
    static constexpr int kVolUsable  = 50;    // 68-18 — knob travel range
    static constexpr int kVolStripH  = 13;    // strip height (mirrors WinampSkin.h)

    // Time digits (NUMBERS.BMP): x=26, y=26
    static constexpr int kTimeX = 26, kTimeY = 26;

    // Song title area
    static constexpr int kSongX = 111, kSongY = 27, kSongW = 154, kSongH = 10;

    // EQ/PL toggle buttons (from SHUFREP.BMP row at y=58)
    // BT_EQ_ON/OFF: x=219, BT_PL_ON/OFF: x=242
    static constexpr int kEQBtnX = 219, kPLBtnX = 241, kTogBtnY = 58;
    static constexpr int kTogBtnW = 23, kTogBtnH = 12;

    // Playlist window
    static constexpr int kPlTitleH  = 20;   // titlebar height
    static constexpr int kPlListX   = 12,   kPlListY  = 20;
    static constexpr int kPlListW   = 243,  kPlListH  = 58;  // at base 275x116
    static constexpr int kPlBotY    = 78;   // bottom bar y
    static constexpr int kPlCloseBX = 264,  kPlCloseBY = 3;

    // Snap threshold (qmmp Dock: 13px)
    static constexpr int kSnapThresh = 13;
    // ─── EQMAIN.BMP sprite map (qmmp skin.cpp lines 605-644) ──────────────────────
    // All coordinates are in 1x skin pixels; DrawScaled multiplies by fScale.
    //
    // EQ_MAIN (275×116):
    static constexpr SkinRect kEQ_Main        = {   0,   0, 275, 116 };
    //
    // Titlebar (y=134 active, y=149 inactive — 275×14 each):
    static constexpr SkinRect kEQ_TitlebarA   = {   0, 134, 275,  14 };
    static constexpr SkinRect kEQ_TitlebarI   = {   0, 149, 275,  14 };
    //
    // Close button (from eqmain.bmp):
    static constexpr SkinRect kEQ_CloseN      = {   0, 116,   9,   9 };
    static constexpr SkinRect kEQ_CloseP      = {   0, 125,   9,   9 };
    // Shade button (from eqmain.bmp):
    static constexpr SkinRect kEQ_ShadeN      = { 254, 137,   9,   9 };
    //
    // ON/OFF toggle buttons:
    //   ON  normal=69,119  pressed=128,119  (28×12)
    //   OFF normal=10,119  pressed=187,119  (28×12)
    static constexpr SkinRect kEQ_OnN         = {  69, 119,  28,  12 };
    static constexpr SkinRect kEQ_OnP         = { 128, 119,  28,  12 };
    static constexpr SkinRect kEQ_OffN        = {  10, 119,  28,  12 };
    static constexpr SkinRect kEQ_OffP        = { 187, 119,  28,  12 };
    //
    // AUTO toggle buttons (33×12):
    static constexpr SkinRect kEQ_Auto1N      = {  94, 119,  33,  12 };
    static constexpr SkinRect kEQ_Auto1P      = { 153, 119,  33,  12 };
    static constexpr SkinRect kEQ_Auto0N      = {  35, 119,  33,  12 };
    static constexpr SkinRect kEQ_Auto0P      = { 212, 119,  33,  12 };
    //
    // PRESETS button (44×12):
    static constexpr SkinRect kEQ_PresetsN    = { 224, 164,  44,  12 };
    static constexpr SkinRect kEQ_PresetsP    = { 224, 176,  44,  12 };
    //
    // EQ slider strip (14×63 per frame, 28 frames, normal at y=164, pressed at y=229):
    //   frame i: x = 13 + i*15
    static constexpr int kEQ_SliderW  = 14;
    static constexpr int kEQ_SliderH  = 63;
    static constexpr int kEQ_SliderY0 = 164;   // normal
    static constexpr int kEQ_SliderY1 = 229;   // pressed (unused — we use N always)
    //
    // Slider knob buttons (11×11):
    static constexpr SkinRect kEQ_BarN        = {   0, 164,  11,  11 };
    static constexpr SkinRect kEQ_BarP        = {   0, 176,  11,  11 };
    //
    // EQ graph background (113×19, from eqmain.bmp at y=294):
    static constexpr SkinRect kEQ_Graph       = {   0, 294, 113,  19 };
    // EQ graph spline pixels (1×1 each, x=115, y=294..312):
    //   spline[i] is at (115, 294+i) for i=0..18
    //
    // Layout positions (in 1x skin pixels, multiply by scale):
    //   Titlebar:     (0, 0)
    //   On button:    (14, 18)
    //   Auto button:  (39, 18)
    //   Presets btn:  (217, 18)
    //   EQ graph:     (87, 17)
    //   Preamp sldr:  (21, 38)    — 14×63
    //   Band sliders: (78 + i*18, 38) for i=0..9
    //   Close btn:    (264, 3)    on titlebar
    //   Shade btn:    (254, 3)    on titlebar
    static constexpr int kEQ_W = 275, kEQ_H = 116;
    static constexpr int kEQ_TitleH = 14;
    static constexpr int kEQ_SliderX0 = 78;   // first band slider x
    static constexpr int kEQ_SliderStep = 18; // spacing between sliders
    static constexpr int kEQ_SliderY  = 38;   // y position for all sliders
    static constexpr int kEQ_PreampX  = 21;   // preamp slider x
    static constexpr int kEQ_GraphX   = 87, kEQ_GraphY = 17;
    static constexpr int kEQ_CloseBX  = 264, kEQ_CloseBY = 3;
    //   Close button: x=264*r + sx*25,  y=3*r   (we fix sx=0)
    //   Add button:   x=11*r,  y=86*r + 29*sy   (we fix sy=0)
    //   Sub button:   x=40*r,  y=86*r + 29*sy
    //   PL_Control:   x=128*r, y=100*r + 29*sy
    //   List widget:  x=12*r,  y=20*r,  w=243*r, h=58*r

}

// ─── Helpers ──────────────────────────────────────────────────────────────────
static inline bool InR(BPoint p, int x, int y, int w, int h) {
    return p.x >= x && p.x < x+w && p.y >= y && p.y < y+h;
}

// Scale a SkinRect destination (draw at scaled coords)
static inline BRect ScaledRect(int x, int y, int w, int h, int s) {
    return BRect((float)(x*s), (float)(y*s),
                 (float)(x*s + w*s - 1), (float)(y*s + h*s - 1));
}

//
// FFT: uses fft.h (Richard Boulton, GPL) — same file qmmp uses unchanged.
// Audio: calls VisAudioSnapshot() which snapshots the last 512 frames from gAudio.

// Declared in MainWindow.cpp — snapshots the last `count` mono float samples.
extern void VisAudioSnapshot(float* out, int count);

static constexpr uint32 kMsgVisTick = 'VisT';

// xscale tables from qmmp — frequency bin ranges for each analyzer bar
static const int kXscaleLong[77] = {  // 77 fence-post values for 75 bars
    0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,
    19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,
    35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,
    52,53,54,55,56,57,58,61,66,71,76,81,87,93,100,107,
    114,122,131,140,150,161,172,184,255
};
static const int kXscaleShort[21] = {
    0,1,2,3,4,5,6,7,8,11,15,20,27,36,47,62,82,107,141,184,255
};

class WinampVis : public BView {
public:
    enum Mode { kOff, kAnalyzer, kScope };

    WinampVis(BPoint skinPos, WinampSkin* skin, int scale)
        : BView(BRect(skinPos.x * scale, skinPos.y * scale,
                      skinPos.x * scale + 76 * scale - 1,
                      skinPos.y * scale + 16 * scale - 1),
                "WinampVis", B_FOLLOW_NONE, B_WILL_DRAW)
        , fSkin(skin), fScale(scale), fMode(kAnalyzer)
        , fFftState(nullptr), fRunner(nullptr)
    {
        memset(fAnalyzerData, 0, sizeof(fAnalyzerData));
        memset(fPeaks,        0, sizeof(fPeaks));
        memset(fScopeData,    0, sizeof(fScopeData));
        SetViewColor(B_TRANSPARENT_COLOR);
    }

    ~WinampVis() override {
        delete fRunner;
        if (fFftState) fft_close(fFftState);
    }

    void AttachedToWindow() override {
        // Start 25fps timer (qmmp default fps)
        BMessage tick(kMsgVisTick);
        fRunner = new BMessageRunner(BMessenger(this), &tick, 1000000LL / 25);
    }

    void ReloadSkin(int newScale) {
        fScale = newScale;
        float x = 24.0f * newScale, y = 43.0f * newScale;
        MoveTo(x, y);
        ResizeTo(76.0f * newScale - 1, 16.0f * newScale - 1);
        Invalidate();
    }

    void MessageReceived(BMessage* msg) override {
        if (msg->what == kMsgVisTick) {
            Tick();
        } else {
            BView::MessageReceived(msg);
        }
    }

    void MouseDown(BPoint) override {
        // Left-click cycles: Off → Analyzer → Scope → Off
        fMode = (Mode)((fMode + 1) % 3);
        if (fMode == kOff) {
            memset(fAnalyzerData, 0, sizeof(fAnalyzerData));
            memset(fPeaks,        0, sizeof(fPeaks));
            memset(fScopeData,    0, sizeof(fScopeData));
        }
        Invalidate();
    }

    void Draw(BRect) override {
        DrawBackground();
        if (fMode == kAnalyzer) DrawAnalyzer();
        else if (fMode == kScope) DrawScope();
    }

private:
    void Tick() {
        if (fMode == kOff) return;
        float buf[512];
        VisAudioSnapshot(buf, 512);

        if (fMode == kAnalyzer) {
            ProcessAnalyzer(buf);
        } else {
            ProcessScope(buf);
        }
        Invalidate();
    }

    // ── Background: alternating dot pattern ──────────────────────────────────
    // qmmp drawBackGround(): for x in 0..76*r step 2:
    //   col x+1 (odd): all viscolor[0]
    //   col x (even): even rows viscolor[0], odd rows viscolor[1]
    void DrawBackground() {
        const VisColors& vc = fSkin->VisCol();
        int s = fScale;
        int W = 76 * s, H = 16 * s;
        for (int x = 0; x < W; x += 2 * s) {
            // odd column (x+s): all background color 0
            SetHighColor(vc.r[0], vc.g[0], vc.b[0]);
            FillRect(BRect(x + s, 0, x + 2*s - 1, H - 1));
            // even column (x): stripe viscolor[0]/[1] every s rows
            for (int y = 0; y < H; y += 2 * s) {
                SetHighColor(vc.r[0], vc.g[0], vc.b[0]);
                FillRect(BRect(x, y,       x + s - 1, y + s - 1));
                SetHighColor(vc.r[1], vc.g[1], vc.b[1]);
                FillRect(BRect(x, y + s,   x + s - 1, y + 2*s - 1));
            }
        }
    }

    // ── Analyzer FFT processing ───────────────────────────────────────────────
    // qmmp Analyzer::process(): calc_freq → 75 or 19 bins, log scale 0..15
    void ProcessAnalyzer(float* buf) {
        if (!fFftState) fFftState = fft_init();
        short dest[256] = {};
        // calc_freq inline (from inlines.h)
        float tmp[257] = {};
        fft_perform(buf, tmp, fFftState);
        for (int i = 0; i < 256; i++)
            dest[i] = (short)(((int)sqrtf(tmp[i+1])) >> 8);

        const double y_scale = 3.60673760222; // 20.0/log(256)
        // Lines mode: 75 bars using kXscaleLong
        for (int i = 0; i < 75; i++) {
            int y = 0;
            for (int j = kXscaleLong[i]; j < kXscaleLong[i+1]; j++)
                if (dest[j] > y) y = dest[j];
            y >>= 7;
            int mag = (y > 0) ? (int)(log((double)y) * y_scale) : 0;
            mag = std::max(0, std::min(15, mag));

            fAnalyzerData[i] -= kAnalyzerFalloff;
            if (mag > fAnalyzerData[i]) fAnalyzerData[i] = (double)mag;

            fPeaks[i] -= kPeaksFalloff;
            if (mag > fPeaks[i]) fPeaks[i] = (double)mag;
        }
    }

    // ── Analyzer drawing ──────────────────────────────────────────────────────
    // qmmp Analyzer::draw() lines mode (r=1): for each of 75 bars, draw points
    // from bottom up, color from viscolor[18-i] (top bar brightest).
    // At scale s: each "pixel" is s×s, each bar occupies 1×s columns.
    void DrawAnalyzer() {
        const VisColors& vc = fSkin->VisCol();
        int s = fScale, H = 16 * s;
        for (int j = 0; j < 75; j++) {
            int height = (int)fAnalyzerData[j];
            for (int i = 0; i <= height; i++) {
                // viscolor[18-i]: index 18 = bottom (dim), decreasing toward top (bright)
                int ci = std::max(0, 18 - i);
                SetHighColor(vc.r[ci], vc.g[ci], vc.b[ci]);
                int y = H - s * (i + 1);
                FillRect(BRect(j * s, y, j * s + s - 1, y + s - 1));
            }
            // Peak dot: viscolor[23]
            if (fPeaks[j] > 0) {
                int pi = (int)fPeaks[j];
                SetHighColor(vc.r[23], vc.g[23], vc.b[23]);
                int y = H - s * (pi + 1);
                FillRect(BRect(j * s, y, j * s + s - 1, y + s - 1));
            }
        }
    }

    // qmmp Scope::process(): 76 evenly-spaced samples, scaled to ±4
    void ProcessScope(float* buf) {
        int step = (512 << 8) / 76;
        int pos = 0;
        for (int i = 0; i < 76; i++) {
            pos += step;
            int v = (int)(buf[pos >> 8] * 8.0f);
            fScopeData[i] = std::max(-4, std::min(4, v));
        }
    }

    // qmmp Scope::draw(): 75 lines from (i*r, h1*r) to ((i+1)*r, h2*r)
    // color = viscolor[18 + abs(8 - h2)]   (center = dim, extremes = bright)
    void DrawScope() {
        const VisColors& vc = fSkin->VisCol();
        int s = fScale;
        for (int i = 0; i < 75; i++) {
            int h1 = 8 - fScopeData[i];
            int h2 = 8 - fScopeData[i+1];
            if (h1 > h2) { int t = h1; h1 = h2; h2 = t; }
            int ci = std::min(23, 18 + std::abs(8 - h2));
            SetHighColor(vc.r[ci], vc.g[ci], vc.b[ci]);
            // Draw vertical segment from h1 to h2 at x=i*s
            StrokeLine(BPoint(i * s, h1 * s), BPoint((i+1) * s, h2 * s));
        }
        memset(fScopeData, 0, sizeof(fScopeData));
    }

    static constexpr double kAnalyzerFalloff = 2.2;
    static constexpr double kPeaksFalloff    = 0.2;

    WinampSkin*     fSkin;
    int             fScale;
    Mode            fMode;
    fft_state*      fFftState;
    BMessageRunner* fRunner;
    double          fAnalyzerData[75];
    double          fPeaks[75];
    int             fScopeData[76];
};

class SkinMainView : public BView {
public:
    SkinMainView(BRect frame, WinampSkin* skin, BMessenger target, int scale)
        : BView(frame, "skinMain", B_FOLLOW_ALL, B_WILL_DRAW)
        , fSkin(skin), fTarget(target), fScale(scale)
    {
        SetViewColor(B_TRANSPARENT_COLOR);
        // Visualizer: 76x16 at (24,43) — qmmp m_vis->move(r*24, r*43)
        fVis = new WinampVis(BPoint(24, 43), skin, scale);
        AddChild(fVis);
    }

    // State setters — only call from within this looper (LockLooper held)
    void SetTrackInfo(const char* song, const char* artist) {
        fSong   = song   ? song   : "";
        fArtist = artist ? artist : "";
        Invalidate();
    }
    void SetPosition(int elapsed, int duration) {
        fElapsed = elapsed; fDuration = duration; Invalidate();
    }
    void SetPlaying(bool playing, bool paused) {
        fPlaying = playing; fPaused = paused; Invalidate();
    }
    void SetVolume(int v)  { fVolume  = v; Invalidate(); }
    void SetBalance(int v) { fBalance = v; Invalidate(); }

    void Draw(BRect) override {
        // ── 1. Main background (MAIN.BMP, 275×116 crop) ──────────────────────
        if (BBitmap* bg = fSkin->Sheet("main.bmp")) {
            SkinRect r = {0, 0, WA::kMainW, WA::kMainH};
            DrawScaled(bg, r, 0, 0);
        } else {
            SetHighColor(0, 0, 0);
            FillRect(Bounds());
        }
        DrawTitlebar();
        DrawTransport();
        DrawVolume();
        DrawBalance();
        DrawPosBar();
        DrawTime();
        DrawSongTitle();
        DrawToggleButtons();
    }

    void MouseDown(BPoint pt) override {
        uint32 btns = 0;
        GetMouse(&pt, &btns, false);
        BPoint sp(pt.x / fScale, pt.y / fScale);   // skin-pixel space

        // Volume bar
        if (InR(sp, WA::kVolX, WA::kVolY, WA::kVolWidgetW, WA::kVolStripH + 2)) {
            fDraggingVol = true;
            fVolPressed  = true;
            UpdateVolFromPoint(sp);
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
            Invalidate(ScaledRect(WA::kVolX, WA::kVolY,
                                  WA::kVolWidgetW, WA::kVolStripH + 2, fScale));
            return;
        }
        // Balance bar
        if (InR(sp, WA::kBalX, WA::kVolY, WA::kBalStripW, WA::kBalStripH + 2)) {
            fDraggingBal = true;
            fBalPressed  = true;
            UpdateBalFromPoint(sp);
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
            Invalidate(ScaledRect(WA::kBalX, WA::kVolY,
                                  WA::kBalStripW, WA::kBalStripH + 2, fScale));
            return;
        }
        // Titlebar drag / button zone
        if (InR(sp, 0, 0, WA::kMainW, WA::kTitleH)) {
            // Right-click anywhere on titlebar → context menu
            if (btns & B_SECONDARY_MOUSE_BUTTON) { ShowMenu(pt); return; }
            // Menu button (BT_MENU)
            if (InR(sp, WA::kMenuBX, WA::kMenuBY, WA::kBtnSz, WA::kBtnSz))
                { ShowMenu(BPoint((float)(WA::kMenuBX * fScale), (float)(WA::kTitleH * fScale))); return; }
            // Minimize button
            if (InR(sp, WA::kMinBX, WA::kMinBY, WA::kBtnSz, WA::kBtnSz))
                { Window()->Minimize(true); return; }
            // Close button
            if (InR(sp, WA::kCloseBX, WA::kCloseBY, WA::kBtnSz, WA::kBtnSz))
                { Window()->Hide(); return; }
            // Drag
            fDragging   = true;
            fDragOrigin = ConvertToScreen(pt);
            fWinOrigin  = Window()->Frame().LeftTop();
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS | B_NO_POINTER_HISTORY);
            return;
        }
        // Position bar scrub
        if (InR(sp, WA::kPosBarX, WA::kPosBarY,
                WA::kPosBarW + WA::kPosKnobW, 12)) {
            fDraggingPos = true;
            UpdatePosFromPoint(sp);
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
            return;
        }
        // Transport / toggle buttons
        fPressedBtn = HitTestBtn(sp);
        if (fPressedBtn >= 0) {
            Invalidate();
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
        }
        // Right-click anywhere → context menu
        if (btns & B_SECONDARY_MOUSE_BUTTON) ShowMenu(pt);
    }

    void MouseMoved(BPoint pt, uint32, const BMessage*) override {
        BPoint sp(pt.x / fScale, pt.y / fScale);
        if (fDragging) {
            BPoint sc = ConvertToScreen(pt);
            BPoint npos = fWinOrigin + sc - fDragOrigin;
            if (fDock) npos = fDock->Move(Window(), npos);
            else       npos = SnapToScreen(npos);
            Window()->MoveTo(npos);
        } else if (fDraggingVol) {
            UpdateVolFromPoint(sp);
        } else if (fDraggingBal) {
            UpdateBalFromPoint(sp);
        } else if (fDraggingPos) {
            UpdatePosFromPoint(sp);
        }
    }

    void MouseUp(BPoint pt) override {
        BPoint sp(pt.x / fScale, pt.y / fScale);
        if (fDragging)    { fDragging = false; if (fDock) fDock->UpdateDock(); return; }
        if (fDraggingVol) {
            fDraggingVol = false; fVolPressed = false;
            Invalidate(ScaledRect(WA::kVolX, WA::kVolY,
                                  WA::kVolWidgetW, WA::kVolStripH + 2, fScale));
            return;
        }
        if (fDraggingBal) {
            fDraggingBal = false; fBalPressed = false;
            Invalidate(ScaledRect(WA::kBalX, WA::kVolY,
                                  WA::kBalStripW, WA::kBalStripH + 2, fScale));
            return;
        }
        if (fDraggingPos) { fDraggingPos = false; return; }

        int hit = HitTestBtn(sp);
        if (hit >= 0 && hit == fPressedBtn) FireButton(hit);
        fPressedBtn = -1;
        Invalidate();
    }

private:
    enum {
        BTN_NONE = -1,
        BTN_PREV, BTN_PLAY, BTN_PAUSE, BTN_STOP, BTN_NEXT,
        BTN_PL, BTN_EQ
    };

    WinampSkin* fSkin;
    BMessenger  fTarget;
    int         fScale;
    WinampVis*  fVis = nullptr;  // visualizer widget (child view)

    // Player state
    std::string fSong, fArtist;
    int  fElapsed = 0, fDuration = 0, fVolume = 80, fBalance = 0;
    bool fPlaying = false, fPaused = false;

    // Interaction state
    int    fPressedBtn  = BTN_NONE;
    SkinDock* fDock     = nullptr;
    bool   fDragging    = false;
    bool   fDraggingVol = false;
    bool   fVolPressed  = false;
    bool   fDraggingBal = false;
    bool   fBalPressed  = false;
    bool   fDraggingPos = false;
    BPoint fDragOrigin, fWinOrigin;

    void DrawScaled(BBitmap* src, SkinRect r, int dstX, int dstY) {
        if (!src) return;
        int s  = fScale;
        int sw = (int)src->Bounds().Width() + 1;
        int sh = (int)src->Bounds().Height() + 1;
        // Source rect in scaled-sheet pixels
        int sx1 = r.x * s,          sy1 = r.y * s;
        int sw2 = std::min(r.w * s, sw - sx1);
        int sh2 = std::min(r.h * s, sh - sy1);
        if (sx1 >= sw || sy1 >= sh || sw2 <= 0 || sh2 <= 0) return;
        BRect srcR((float)sx1, (float)sy1,
                   (float)(sx1 + sw2 - 1), (float)(sy1 + sh2 - 1));
        // Destination: same pixel dimensions, placed at (dstX*s, dstY*s)
        float dx = (float)(dstX * s), dy = (float)(dstY * s);
        DrawBitmap(src, srcR, BRect(dx, dy, dx + sw2 - 1, dy + sh2 - 1));
    }

    // Blit a sub-rect of a named sheet
    void DrawSheet(const char* file, SkinRect r, int dstX, int dstY) {
        DrawScaled(fSkin->Sheet(file), r, dstX, dstY);
    }

    void DrawTitlebar() {
        // Background: TITLEBAR_A = copy(27,0, 275,14)
        DrawSheet("titlebar.bmp", kTB_BgActive, 0, 0);
        // Buttons: normal state (no press state tracked for titlebar buttons)
        DrawSheet("titlebar.bmp", kTB_MenuN,  WA::kMenuBX,  WA::kMenuBY);
        DrawSheet("titlebar.bmp", kTB_MinN,   WA::kMinBX,   WA::kMinBY);
        DrawSheet("titlebar.bmp", kTB_CloseN, WA::kCloseBX, WA::kCloseBY);
    }

    void DrawTransport() {
        // qmmp: SkinnedButton uses BT_PREV_N/P from CBUTTONS.BMP
        // y=0 normal row, y=18 pressed row
        struct BtnDef { SkinRect up, dn; int x; int id; } btns[] = {
            { kCBtn_Prev_Up,  kCBtn_Prev_Dn,  WA::kPrevX,  BTN_PREV  },
            { kCBtn_Play_Up,  kCBtn_Play_Dn,  WA::kPlayX,  BTN_PLAY  },
            { kCBtn_Pause_Up, kCBtn_Pause_Dn, WA::kPauseX, BTN_PAUSE },
            { kCBtn_Stop_Up,  kCBtn_Stop_Dn,  WA::kStopX,  BTN_STOP  },
            { kCBtn_Next_Up,  kCBtn_Next_Dn,  WA::kNextX,  BTN_NEXT  },
        };
        for (auto& b : btns)
            DrawSheet("cbuttons.bmp",
                      (fPressedBtn == b.id) ? b.dn : b.up,
                      b.x, WA::kBtnY);
    }

    void DrawVolume() {
        // qmmp SkinnedVolumeBar::draw():

        int s = fScale;

        // Step 1: background strip (1x SkinRect, DrawScaled handles scaling)
        int stripIdx = fVolume * 27 / 100;
        SkinRect strip = {0, stripIdx * kVolStripStep, kVolStripW, kVolStripH};
        DrawSheet("volume.bmp", strip, WA::kVolX, WA::kVolY);

        // Step 2: knob at pixel offset p (in 1x skin pixels)
        int p_skin = (int)std::ceil((double)fVolume * WA::kVolUsable / 100.0);
        p_skin = std::max(0, std::min(p_skin, WA::kVolUsable));

        // Step 3: draw knob sprite
        // BT_VOL_N = copy(15,422,14,h-422) at 1x.
        // Sheet is pre-scaled, so actual knob is at (15*s, 422*s) in the stored bitmap.
        // We pass 1x coords to DrawScaled which multiplies by s internally.
        BBitmap* vol = fSkin->Sheet("volume.bmp");
        if (vol) {
            int volH = (int)vol->Bounds().Height() + 1;
            // qmmp: only draw knob if pixmap->height() > 425 (at 1x)
            if (volH / s > 425) {
                int kx1 = fVolPressed ? 0 : 15;  // 1x knob x
                SkinRect kr = {kx1, kVolKnobY, kVolKnobW, (volH / s) - kVolKnobY};
                DrawScaled(vol, kr, WA::kVolX + p_skin, WA::kVolY + 1);
            }
        }
    }

    // Dead zone: if |value| < 6, treat as 0 (snaps to center)
    void DrawBalance() {
        int s = fScale;
        // Apply center dead zone (qmmp: if(abs(m_value)<6) m_value=0)
        int val = (std::abs(fBalance) < 6) ? 0 : fBalance;
        val = std::max(-100, std::min(100, val));

        // Choose bitmap: balance.bmp first, fall back to volume.bmp
        BBitmap* bm = fSkin->Sheet("balance.bmp");
        if (!bm) bm = fSkin->Sheet("volume.bmp");
        if (!bm) return;

        int bmW = (int)bm->Bounds().Width()  + 1;
        int bmH = (int)bm->Bounds().Height() + 1;

        // ── Strip background ─────────────────────────────────────────────────
        // Frame index: abs(27 * val / 100), source at (9, frame*15, 38, 13)
        int frameIdx = std::abs(27 * val / 100);
        frameIdx = std::max(0, std::min(27, frameIdx));
        int srcX = 9 * s, srcY = frameIdx * 15 * s;
        int sw = WA::kBalStripW * s, sh = WA::kBalStripH * s;
        // Clamp to bitmap bounds
        sw = std::min(sw, bmW - srcX); sh = std::min(sh, bmH - srcY);
        if (sw > 0 && sh > 0) {
            BRect srcR((float)srcX, (float)srcY,
                       (float)(srcX+sw-1), (float)(srcY+sh-1));
            BRect dstR((float)(WA::kBalX*s), (float)(WA::kVolY*s),
                       (float)(WA::kBalX*s+sw-1), (float)(WA::kVolY*s+sh-1));
            DrawBitmap(bm, srcR, dstR);
        }

        // ── Knob ─────────────────────────────────────────────────────────────
        // p = ceil((val - (-100)) * kBalUsable / (100-(-100)))
        //   = ceil((val+100) * 25 / 200)
        int p = (int)std::ceil((double)(val + 100) * WA::kBalUsable / 200.0);
        p = std::max(0, std::min(p, WA::kBalUsable));

        // Knob sprite at (15*s, kBalKnobY*s) normal; (0, kBalKnobY*s) pressed
        int kx = (fBalPressed ? 0 : 15) * s;
        int ky = WA::kBalKnobY * s;
        int kw = WA::kBalKnobW * s;
        int kh = bmH - ky;   // knob height = rest of bitmap from y=422
        // qmmp: only draw knob if pixmap->height() > 427 (at 1x)
        if (kh > 0 && bmH / s > 427 && ky < bmH && kx + kw <= bmW) {
            BRect knobSrc((float)kx, (float)ky,
                          (float)(kx+kw-1), (float)(ky+kh-1));
            // Knob drawn at (kBalX + p, kVolY + 1) in skin pixels
            int dkx = (WA::kBalX + p) * s;
            int dky = (WA::kVolY + 1) * s;
            BRect knobDst((float)dkx, (float)dky,
                          (float)(dkx+kw-1), (float)(dky+kh-1));
            DrawBitmap(bm, knobSrc, knobDst);
        }
    }

    void DrawPosBar() {
        // POSBAR.BMP: bar = copy(0,0,248,min(h,10))
        //             knob = copy(248,0,29,h)  [only if width > 249]
        // (mirrors qmmp Skin::loadPosBar())
        BBitmap* pb = fSkin->Sheet("posbar.bmp");
        if (!pb) return;
        int s = fScale;
        int pbW = (int)pb->Bounds().Width() + 1;
        int pbH = (int)pb->Bounds().Height() + 1;

        // Bar background: x=0, y=0, w=248, h=min(rawH,10)
        int barH = std::min(pbH / s, 10);
        SkinRect barR = {0, 0, WA::kPosBarW, barH};
        DrawScaled(pb, barR, WA::kPosBarX, WA::kPosBarY);

        // Knob: only if bitmap is wide enough (qmmp: width > 249)
        int kx = WA::kPosBarX;
        if (fDuration > 0) {
            int travel = WA::kPosBarW - WA::kPosKnobW;  // 248-29=219
            kx += (int)((double)travel * fElapsed / fDuration);
        }
        if (pbW / s > 249) {
            // Full bitmap height for knob (qmmp uses pixmap->height())
            int knobH = pbH / s;
            SkinRect knobR = {WA::kPosBarW, 0, WA::kPosKnobW, knobH};
            DrawScaled(pb, knobR, kx, WA::kPosBarY);
        }
    }

    void DrawTime() {
        // PLAYPAUS.BMP indicator
        SkinRect ind = fPlaying ? (fPaused ? kPP_Pause : kPP_Play) : kPP_Stop;
        DrawSheet("playpaus.bmp", ind, 24, 28);

        // Digits: prefer NUMS_EX.BMP (wider set), fall back to NUMBERS.BMP
        // qmmp: getPixmap("nums_ex", "numbers") — same fallback logic
        BBitmap* numBm = fSkin->Sheet("nums_ex.bmp");
        if (!numBm) numBm = fSkin->Sheet("numbers.bmp");
        if (!numBm) return;

        // Use actual bitmap height for digit height (some skins differ from 13px)
        int digitH = ((int)numBm->Bounds().Height() + 1) / fScale;
        digitH = std::max(1, digitH);

        int t = fPlaying ? fElapsed : 0;
        char buf[5];
        int mm = std::abs(t / 60) % 100;
        int ss = std::abs(t % 60);
        snprintf(buf, sizeof(buf), "%02d%02d", mm, ss);

        int dx = WA::kTimeX;
        for (int i = 0; i < 4; i++) {
            if (i == 2) dx += 2;  // gap between MM and SS
            int col = (buf[i] >= '0' && buf[i] <= '9') ? (buf[i] - '0') : 10;
            SkinRect dr = {col * kDigitW, 0, kDigitW, digitH};
            DrawScaled(numBm, dr, dx, WA::kTimeY);
            dx += kDigitW + 1;
        }
    }

    void DrawSongTitle() {
        if (fSong.empty()) return;
        // Black out the title area first
        int x = WA::kSongX * fScale, y = WA::kSongY * fScale;
        int w = WA::kSongW * fScale, h = WA::kSongH * fScale + 4;
        SetHighColor(0, 0, 0);
        FillRect(BRect((float)x, (float)y, (float)(x+w-1), (float)(y+h-1)));

        std::string title = fSong + " - " + fArtist;
        size_t len = title.size();
        size_t offset = (fPlaying && len > 0)
                        ? (size_t)(fElapsed / 2) % len : 0;
        std::string disp = title.substr(offset) + "   " + title;

        SetHighColor(0, 200, 0);
        BFont f(be_fixed_font);
        f.SetSize(7.5f * (float)fScale);
        SetFont(&f);
        BRegion clip(BRect((float)x, (float)y, (float)(x+w-1), (float)(y+h-1)));
        ConstrainClippingRegion(&clip);
        DrawString(disp.c_str(), BPoint((float)x + 2.f, (float)(y + h) - 2.f));
        ConstrainClippingRegion(nullptr);
    }

    void DrawToggleButtons() {
        // EQ and PL toggles from SHUFREP.BMP
        // BT_EQ_OFF_N = copy(0, 61, 23, 12)   BT_PL_OFF_N = copy(23, 61, 23, 12)
        // BT_EQ_ON_N  = copy(0, 73, 23, 12)   BT_PL_ON_N  = copy(23, 73, 23, 12)
        // We use the OFF state for EQ (no equalizer), ON state depends on PL visibility
        // EQ toggle button: on=0,73 off=0,61 in shufrep.bmp
        SkinRect eqState = fEqVisible ? SkinRect{0, 73, WA::kTogBtnW, WA::kTogBtnH}
                                       : SkinRect{0, 61, WA::kTogBtnW, WA::kTogBtnH};
        DrawSheet("shufrep.bmp", eqState, WA::kEQBtnX, WA::kTogBtnY);
        SkinRect plState = fPlVisible ? SkinRect{23, 73, WA::kTogBtnW, WA::kTogBtnH}
                                      : SkinRect{23, 61, WA::kTogBtnW, WA::kTogBtnH};
        DrawSheet("shufrep.bmp", plState, WA::kPLBtnX, WA::kTogBtnY);
    }

    //
    // qmmp createActions():
    //   Play, Pause, Stop, Previous, Next
    //   ── separator ──
    //   View > Show Playlist / Double Size / Antialiasing
    //   Audio > Volume Up / Volume Down
    //   ── separator ──
    //   Settings / Quit
    //
    // BT_MENU button or right-click → exec menu at click position.
    void ShowMenu(BPoint screenPt) {
        BPopUpMenu* m = new BPopUpMenu("main", false, false);

        BMenuItem* ppItem = new BMenuItem(
            fPlaying ? (fPaused ? "Play" : "Pause") : "Play",
            new BMessage('PlPs'));
        m->AddItem(ppItem);
        m->AddItem(new BMenuItem("Next",  new BMessage('Next')));
        m->AddSeparatorItem();
        m->AddItem(new BMenuItem("Add Station…", new BMessage(MSG_ADD_STATION)));
        m->AddSeparatorItem();
        m->AddItem(new BMenuItem("Load Skin (.wsz)…", new BMessage(MSG_LOAD_SKIN)));
        m->AddItem(new BMenuItem("Exit Skin Mode",     new BMessage(MSG_CLEAR_SKIN)));
        m->AddSeparatorItem();

        BMenuItem* dblItem = new BMenuItem("Double Size (2×)",
                                            new BMessage(MSG_SKIN_DOUBLE_SZ));
        dblItem->SetMarked(fScale == 2);
        m->AddItem(dblItem);
        BMenuItem* smItem = new BMenuItem("Smooth Scaling",
                                           new BMessage(MSG_SKIN_SMOOTH));
        smItem->SetMarked(fSkin->CurrentSmooth());
        m->AddItem(smItem);

        m->AddSeparatorItem();
        m->AddItem(new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED)));

        // Route all messages to MainWindow
        m->SetTargetForItems(fTarget);
        BPoint local = ConvertFromScreen(screenPt);
        m->Go(ConvertToScreen(local), true, true, true);
        delete m;
    }

    int HitTestBtn(BPoint sp) const {
        if (InR(sp, WA::kPrevX,  WA::kBtnY, WA::kCBtnW, WA::kCBtnH)) return BTN_PREV;
        if (InR(sp, WA::kPlayX,  WA::kBtnY, WA::kCBtnW, WA::kCBtnH)) return BTN_PLAY;
        if (InR(sp, WA::kPauseX, WA::kBtnY, WA::kCBtnW, WA::kCBtnH)) return BTN_PAUSE;
        if (InR(sp, WA::kStopX,  WA::kBtnY, WA::kCBtnW, WA::kCBtnH)) return BTN_STOP;
        if (InR(sp, WA::kNextX,  WA::kBtnY, WA::kCBtnW, WA::kCBtnH)) return BTN_NEXT;
        if (InR(sp, WA::kPLBtnX, WA::kTogBtnY, WA::kTogBtnW, WA::kTogBtnH)) return BTN_PL;
        if (InR(sp, WA::kEQBtnX, WA::kTogBtnY, WA::kTogBtnW, WA::kTogBtnH)) return BTN_EQ;
        return BTN_NONE;
    }

    void FireButton(int btn) {
        switch (btn) {
            case BTN_PREV:  { BMessage m('Prev'); fTarget.SendMessage(&m); break; }
            case BTN_PLAY:
            case BTN_PAUSE:
            case BTN_STOP:  { BMessage m('PlPs'); fTarget.SendMessage(&m); break; }
            case BTN_NEXT:  { BMessage m('Next'); fTarget.SendMessage(&m); break; }
            case BTN_PL:    { BMessage m('SkTg'); fTarget.SendMessage(&m); break; }
            case BTN_EQ:    { BMessage m('EqTg'); fTarget.SendMessage(&m); break; }
            default: break;
        }
    }

    void UpdateVolFromPoint(BPoint sp) {
        // p_skin = clamp(sp.x - kVolX, 0, kVolUsable)
        int p = (int)(sp.x) - WA::kVolX;
        p = std::max(0, std::min(p, WA::kVolUsable));
        fVolume = (int)std::ceil((double)p * 100.0 / WA::kVolUsable);
        fVolume = std::max(0, std::min(100, fVolume));
        BMessage m('Volm');
        m.AddInt32("vol", fVolume);
        fTarget.SendMessage(&m);
        Invalidate(ScaledRect(WA::kVolX, WA::kVolY,
                              WA::kVolWidgetW, kVolStripH + 4, fScale));
    }

    // qmmp SkinnedBalanceBar::convert() and sliderMoved logic:
    //   value = ceil((m_max-m_min) * po / travel + m_min)
    //         = ceil(200 * po / 25 + (-100))
    //   then clamp, apply dead zone, send 'Blnc' message
    void UpdateBalFromPoint(BPoint sp) {
        // po = sp.x - kBalX, clamped to [0, kBalUsable]
        int po = (int)(sp.x) - WA::kBalX;
        po = std::max(0, std::min(po, WA::kBalUsable));
        int val = (int)std::ceil(200.0 * po / WA::kBalUsable - 100.0);
        val = std::max(-100, std::min(100, val));
        // Apply center dead zone
        if (std::abs(val) < 6) val = 0;
        fBalance = val;
        BMessage m('Blnc');
        m.AddInt32("bal", fBalance);
        fTarget.SendMessage(&m);
        Invalidate(ScaledRect(WA::kBalX, WA::kVolY,
                              WA::kBalStripW, WA::kBalStripH + 4, fScale));
    }

    void UpdatePosFromPoint(BPoint sp) {
        if (fDuration <= 0) return;
        int travel = WA::kPosBarW - WA::kPosKnobW;
        int px = (int)(sp.x) - WA::kPosBarX;
        px = std::max(0, std::min(px, travel));
        fElapsed = (int)((double)px * fDuration / travel);
        BMessage m('Seek');
        m.AddInt32("sec", fElapsed);
        fTarget.SendMessage(&m);
        Invalidate(ScaledRect(WA::kPosBarX, WA::kPosBarY,
                              WA::kPosBarW + WA::kPosKnobW, 12, fScale));
    }

    BPoint SnapToScreen(BPoint pos) const {
        BScreen screen;
        BRect sr = screen.Frame();
        BRect wf = Window()->Frame();
        int t = WA::kSnapThresh;
        if (std::abs(pos.x - sr.left)  < t) pos.x = sr.left;
        if (std::abs(pos.y - sr.top)   < t) pos.y = sr.top;
        if (std::abs(pos.x + wf.Width()  - sr.right)  < t) pos.x = sr.right  - wf.Width();
        if (std::abs(pos.y + wf.Height() - sr.bottom) < t) pos.y = sr.bottom - wf.Height();
        return pos;
    }

    bool fPlVisible = true;
    bool fEqVisible = true;
    friend class SkinWindow;
};

// B_NO_BORDER_WINDOW_LOOK + B_NORMAL_WINDOW_FEEL = Qt::FramelessWindowHint
// B_WILL_ACCEPT_FIRST_CLICK prevents the click-to-focus dead-click problem.
class SkinWindow : public BWindow {
public:
    SkinMainView* fView = nullptr;

    SkinWindow(BPoint pos, WinampSkin* skin, BMessenger target, int scale)
        : BWindow(BRect(pos.x, pos.y,
                        pos.x + WA::kMainW * scale - 1,
                        pos.y + WA::kMainH * scale - 1),
                  "PandAmp Player",
                  B_NO_BORDER_WINDOW_LOOK,
                  B_NORMAL_WINDOW_FEEL,
                  B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_WILL_ACCEPT_FIRST_CLICK)
    {
        fView = new SkinMainView(Bounds(), skin, target, scale);
        AddChild(fView);
    }

    void SetDock(SkinDock* d) {
        if (LockLooper()) { fView->fDock = d; UnlockLooper(); }
    }

    void SetPlVisible(bool v) {
        if (LockLooper()) {
            fView->fPlVisible = v;
            fView->Invalidate(ScaledRect(WA::kPLBtnX, WA::kTogBtnY,
                                         WA::kTogBtnW, WA::kTogBtnH,
                                         fView->fScale));
            UnlockLooper();
        }
    }

    void SetEqVisible(bool v) {
        if (LockLooper()) {
            fView->fEqVisible = v;
            fView->Invalidate(ScaledRect(WA::kEQBtnX, WA::kTogBtnY,
                                         WA::kTogBtnW, WA::kTogBtnH,
                                         fView->fScale));
            UnlockLooper();
        }
    }

    // ReloadSkin: mirrors qmmp updateSkin() + setFixedSize(r*275, r*116).
    // Call after WinampSkin::SetScale() -- no window destroy/recreate needed.
    void ReloadSkin(int newScale) {
        if (LockLooper()) {
            fView->fScale = newScale;
            float newW = (float)(WA::kMainW * newScale) - 1;
            float newH = (float)(WA::kMainH * newScale) - 1;
            ResizeTo(newW, newH);
            fView->ResizeTo(newW, newH);
            if (fView->fVis) fView->fVis->ReloadSkin(newScale);
            fView->Invalidate();
            UnlockLooper();
        }
    }

    bool QuitRequested() override { Hide(); return false; }
};


class SkinListView : public BView {
public:
    SkinListView(BRect frame, WinampSkin* skin, int scale, BMessenger target)
        : BView(frame, "skinList", B_FOLLOW_ALL, B_WILL_DRAW | B_FRAME_EVENTS)
        , fSkin(skin), fScale(scale), fTarget(target)
    {
        SetViewColor(B_TRANSPARENT_COLOR);
        ComputeMetrics();
    }

    void AddItem(const char* label) {
        fItems.push_back(label);
        ComputeMetrics();
        Invalidate();
    }

    void MakeEmpty() {
        fItems.clear();
        fSelected = -1;
        fCurrent  = -1;
        fScrollTop = 0;
        ComputeMetrics();
        Invalidate();
    }

    int CountItems() const { return (int)fItems.size(); }
    int CurrentSelection() const { return fSelected; }
    int ScrollTop()   const { return fScrollTop; }
    int TotalItems()  const { return (int)fItems.size(); }
    int VisibleRowCount() const { return VisibleRows(); }
    void ScrollBy(int delta) {
        int maxTop = std::max(0, (int)fItems.size() - VisibleRows());
        fScrollTop = std::max(0, std::min(fScrollTop + delta, maxTop));
        Invalidate();
        InvalidateParentScrollStrip();
    }
    void ScrollToFraction(float frac) {
        int maxTop = std::max(0, (int)fItems.size() - VisibleRows());
        fScrollTop = std::max(0, std::min((int)(frac * maxTop + 0.5f), maxTop));
        Invalidate();
        InvalidateParentScrollStrip();
    }
    void SetCurrent(int idx) {
        fCurrent = idx;
        ScrollToVisible(idx);
        Invalidate();
    }

    void ReloadSkin(WinampSkin* skin, int scale) {
        fSkin = skin;
        fScale = scale;
        ComputeMetrics();
        Invalidate();
    }

    void Draw(BRect /*updateRect*/) override {
        const PlEditColors& plc = fSkin->PlColors();
        BRect bounds = Bounds();
        float listW = bounds.Width() + 1;
        int   rows  = VisibleRows();

        // Background fill (qmmp fillBackground)
        SetHighColor(plc.normBgR, plc.normBgG, plc.normBgB);
        FillRect(bounds);

        // Draw rows
        for (int i = 0; i < rows; i++) {
            int idx = fScrollTop + i;
            if (idx >= (int)fItems.size()) break;
            DrawRow(i, idx, listW, plc);
        }
    }

    void FrameResized(float, float) override {
        ComputeMetrics();
        ScrollToVisible(fCurrent >= 0 ? fCurrent : fSelected);
        Invalidate();
    }

    void MouseDown(BPoint pt) override {
        uint32 btns = 0; BPoint p = pt;
        GetMouse(&p, &btns, false);

        // Row click
        int row = RowAt(p.y);
        if (row < 0 || row >= (int)fItems.size()) return;

        int clicks = 1;
        // BView doesn't give click count natively; detect double-click by time
        bigtime_t now = system_time();
        if (row == fSelected && (now - fLastClickTime) < 400000LL)
            clicks = 2;
        fLastClickTime = now;

        fSelected = row;
        Invalidate();

        if (clicks == 2) {
            BMessage msg('Stnx');  // MSG_STATION_SELECT
            msg.AddInt32("index", fSelected);
            fTarget.SendMessage(&msg);
        }
    }

    void MouseMoved(BPoint, uint32, const BMessage*) override {}

    void MouseUp(BPoint) override {}

    void MessageReceived(BMessage* msg) override {
        if (msg->what == B_MOUSE_WHEEL_CHANGED) {
            float dy = 0;
            msg->FindFloat("be:wheel_delta_y", &dy);
            int maxTop = std::max(0, (int)fItems.size() - VisibleRows());
            fScrollTop = std::max(0, std::min(fScrollTop + (int)(dy * 3), maxTop));
            Invalidate();
            InvalidateParentScrollStrip();
        } else {
            BView::MessageReceived(msg);
        }
    }

    void InvalidateParentScrollStrip() {
        BView* parent = Parent();
        if (!parent) return;
        int s = fScale;
        // Scroll strip: x=255*s..274*s, y=kPlListY*s..(kPlBotY-1)*s
        parent->Invalidate(BRect(
            (float)(255 * s), (float)(WA::kPlListY * s),
            (float)(275 * s - 1), (float)((WA::kPlBotY - 1) * s)));
    }

private:
    void ComputeMetrics() {
        BFont font(be_plain_font);
        font.SetSize(9.0f * fScale);
        font_height fh;
        font.GetHeight(&fh);
        fRowH      = (int)(fh.ascent + fh.descent + fh.leading + 1.5f);
        fAscent    = fh.ascent;

        float gw = font.StringWidth("9");
        fPadding   = std::max(2, (int)(gw * 0.5f));

        // Number column: width of widest number (same as qmmp calculateNumberWidth)
        int digits = 1;
        int n = (int)fItems.size();
        while (n >= 10) { n /= 10; digits++; }
        fNumColW   = (int)(gw * digits) + 2 * fPadding;
        if (fItems.empty()) fNumColW = (int)(gw) + 2 * fPadding;

    }

    int VisibleRows() const {
        if (fRowH <= 0) return 0;
        return (int)(Bounds().Height() + 1) / fRowH;
    }

    int RowAt(float y) const {
        if (fRowH <= 0) return -1;
        return fScrollTop + (int)(y / fRowH);
    }

    void ScrollToVisible(int idx) {
        if (idx < 0) return;
        int rows = VisibleRows();
        if (idx < fScrollTop) fScrollTop = idx;
        else if (idx >= fScrollTop + rows) fScrollTop = idx - rows + 1;
        fScrollTop = std::max(0, fScrollTop);
    }



    // ── Row drawing (mirrors qmmp drawBackground + drawTrack, single-column) ──
    void DrawRow(int visRow, int idx, float listW, const PlEditColors& plc) {
        bool isCurrent  = (idx == fCurrent);
        bool isSelected = (idx == fSelected);

        float y = (float)(visRow * fRowH);
        BRect rowR(0, y, listW - 1, y + fRowH - 1);

        // Background (qmmp drawBackground)
        if (isSelected)
            SetHighColor(plc.selBgR, plc.selBgG, plc.selBgB);
        else
            SetHighColor(plc.normBgR, plc.normBgG, plc.normBgB);
        FillRect(rowR);

        // Text color (qmmp drawTrack: current→current, selected→normal, normal→normal)
        if (isCurrent)
            SetHighColor(plc.curR, plc.curG, plc.curB);
        else
            SetHighColor(plc.normalR, plc.normalG, plc.normalB);

        BFont font(be_plain_font);
        font.SetSize(9.0f * fScale);
        SetFont(&font);

        float baseline = y + fAscent + 1;

        // Number column: right-aligned (qmmp: drawText(sx - padding - metrics->advance(number), sy, number))
        std::string numStr = std::to_string(idx + 1) + ".";
        float numW = font.StringWidth(numStr.c_str());
        float numX = (float)fNumColW - (float)fPadding - numW;
        DrawString(numStr.c_str(), BPoint(numX, baseline));

        // Splitter line between number and text (qmmp: m_splitter color = m_normal)
        SetHighColor(plc.normalR, plc.normalG, plc.normalB);
        float sx = (float)fNumColW;
        StrokeLine(BPoint(sx, y), BPoint(sx, y + fRowH - 1));

        // Restore text color
        if (isCurrent)
            SetHighColor(plc.curR, plc.curG, plc.curB);
        else
            SetHighColor(plc.normalR, plc.normalG, plc.normalB);

        // Title: clipped to available width, truncated with ellipsis
        float textX    = sx + fPadding;
        float availW   = listW - sx - fPadding * 2 - 1;
        std::string title = fItems[idx];
        // Truncate with ellipsis if needed (qmmp: metrics->elidedText)
        while (!title.empty() && font.StringWidth(title.c_str()) > availW) {
            title.pop_back();
        }
        if (title.size() < fItems[idx].size()) {
            while (!title.empty() && font.StringWidth((title + "…").c_str()) > availW)
                title.pop_back();
            title += "…";
        }
        DrawString(title.c_str(), BPoint(textX, baseline));
    }



    WinampSkin*              fSkin;
    int                      fScale;
    BMessenger               fTarget;
    std::vector<std::string> fItems;
    int    fSelected       = -1;
    int    fCurrent        = -1;
    int    fScrollTop      = 0;
    int    fRowH           = 14;
    int    fNumColW        = 20;
    int    fPadding        = 3;
    float  fAscent         = 11.0f;
    bigtime_t fLastClickTime = 0;
};

class PlaylistView : public BView {
public:
    SkinListView* fListView = nullptr;
    SkinDock*     fDock     = nullptr;

    PlaylistView(BRect frame, WinampSkin* skin, BMessenger target, int scale)
        : BView(frame, "skinPlaylist", B_FOLLOW_ALL, B_WILL_DRAW)
        , fSkin(skin), fScale(scale), fTarget(target)
    {
        SetViewColor(B_TRANSPARENT_COLOR);

        float lx = (float)(WA::kPlListX * fScale);
        float ly = (float)(WA::kPlTitleH * fScale);
        float lw = (float)(WA::kPlListW * fScale);
        float lh = (float)(WA::kPlListH * fScale);

        fListView = new SkinListView(
            BRect(lx, ly, lx + lw - 1, ly + lh - 1),
            skin, scale, target);
        AddChild(fListView);
    }

    void Draw(BRect) override {
        DrawTitlebar();
        DrawBody();
    }

    void MouseDown(BPoint pt) override {
        uint32 btns = 0;
        GetMouse(&pt, &btns, false);
        BPoint sp(pt.x / fScale, pt.y / fScale);

        if (InR(sp, WA::kPlCloseBX, WA::kPlCloseBY, WA::kBtnSz, WA::kBtnSz)) {
            Window()->Hide(); return;
        }
        // ADD button (11,86, 25×18) — open station search window
        if (InR(sp, 11, 86, 25, 18)) {
            fAddBtnPressed = true;
            Invalidate();
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
            return;
        }
        // Scroll strip: x=255..274 skin px, y=kPlListY..kPlBotY-1
        if (InR(sp, 255, WA::kPlListY, 20, WA::kPlListH) && fListView) {
            fScrollDragging = true;
            fScrollDragOriginY = pt.y;
            fScrollDragTopAtPress = fListView->ScrollTop();
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
            return;
        }
        if (sp.y < WA::kPlTitleH) {
            if (btns & B_SECONDARY_MOUSE_BUTTON) { ShowMenu(pt); return; }
            fDragging   = true;
            fDragOrigin = ConvertToScreen(pt);
            fWinOrigin  = Window()->Frame().LeftTop();
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS | B_NO_POINTER_HISTORY);
        }
    }

    void MouseMoved(BPoint pt, uint32, const BMessage*) override {
        if (fScrollDragging && fListView) {
            // Convert drag distance to scroll fraction
            int total  = fListView->TotalItems();
            int vis    = fListView->VisibleRowCount();
            int maxTop = total - vis;
            if (maxTop > 0) {
                int s      = fScale;
                int listH  = WA::kPlListH * s;
                int knobH  = 18 * s;
                int trackH = listH - knobH;
                if (trackH > 0) {
                    float dy   = pt.y - fScrollDragOriginY;
                    int newTop = fScrollDragTopAtPress + (int)(dy * maxTop / trackH);
                    fListView->ScrollToFraction((float)newTop / maxTop);
                }
            }
            Invalidate(BRect((float)(255 * fScale), (float)(WA::kPlListY * fScale),
                             (float)(275 * fScale - 1), (float)(WA::kPlBotY * fScale - 1)));
            return;
        }
        if (!fDragging) return;
        BPoint sc  = ConvertToScreen(pt);
        BPoint pos = fWinOrigin + sc - fDragOrigin;
        if (fDock) pos = fDock->Move(Window(), pos);
        else {
            BScreen screen; BRect sr = screen.Frame(); BRect wf = Window()->Frame();
            int t = WA::kSnapThresh;
            if (std::abs(pos.x - sr.left)  < t) pos.x = sr.left;
            if (std::abs(pos.y - sr.top)   < t) pos.y = sr.top;
            if (std::abs(pos.x + wf.Width()  - sr.right)  < t) pos.x = sr.right  - wf.Width();
            if (std::abs(pos.y + wf.Height() - sr.bottom) < t) pos.y = sr.bottom - wf.Height();
        }
        Window()->MoveTo(pos);
    }

    void MouseUp(BPoint pt) override {
        if (fAddBtnPressed) {
            fAddBtnPressed = false;
            Invalidate();
            BPoint sp(pt.x / fScale, pt.y / fScale);
            if (InR(sp, 11, 86, 25, 18))
                fTarget.SendMessage(MSG_ADD_STATION);
            return;
        }
        if (fDragging && fDock) fDock->UpdateDock();
        fDragging = false;
        fScrollDragging = false;
    }

public:
    WinampSkin* fSkin;  // accessed by PlaylistWindow::ReloadSkin
public:
    int         fScale;
private:
    BMessenger  fTarget;
    bool   fDragging          = false;
    bool   fScrollDragging    = false;
    bool   fAddBtnPressed     = false;
    float  fScrollDragOriginY = 0;
    int    fScrollDragTopAtPress = 0;
    BPoint fDragOrigin, fWinOrigin;

    void DrawScaled(BBitmap* src, SkinRect r, int dstX, int dstY) {
        if (!src) return;
        int s  = fScale;
        int sw = (int)src->Bounds().Width() + 1;
        int sh = (int)src->Bounds().Height() + 1;
        int sx1 = r.x * s, sy1 = r.y * s;
        int sw2 = std::min(r.w * s, sw - sx1);
        int sh2 = std::min(r.h * s, sh - sy1);
        if (sx1 >= sw || sy1 >= sh || sw2 <= 0 || sh2 <= 0) return;
        BRect srcR((float)sx1, (float)sy1, (float)(sx1+sw2-1), (float)(sy1+sh2-1));
        float dx = (float)(dstX * s), dy = (float)(dstY * s);
        DrawBitmap(src, srcR, BRect(dx, dy, dx+sw2-1, dy+sh2-1));
    }

    void DrawSheet(const char* file, SkinRect r, int dstX, int dstY) {
        DrawScaled(fSkin->Sheet(file), r, dstX, dstY);
    }

    void DrawTitlebar() {
        BBitmap* pl = fSkin->Sheet("pledit.bmp");
        if (!pl) {
            SetHighColor(30, 30, 60);
            FillRect(BRect(0, 0, (float)(WA::kPlW*fScale-1),
                                 (float)(WA::kPlTitleH*fScale-1)));

            return;
        }
        DrawScaled(pl, kPL_CornerUL_A, 0, 0);
        for (int i = 1; i < 10; i++)
            DrawScaled(pl, kPL_TFill1A, 25*i, 0);
        DrawScaled(pl, kPL_TitleA,    88, 0);
        DrawScaled(pl, kPL_CornerUR_A, 250, 0);
        DrawScaled(pl, kPL_CloseN, WA::kPlCloseBX, WA::kPlCloseBY);


    }

    void DrawBody() {
        BBitmap* pl = fSkin->Sheet("pledit.bmp");
        if (!pl) {
            const PlEditColors& plc = fSkin->PlColors();
            SetHighColor(plc.normBgR, plc.normBgG, plc.normBgB);
            FillRect(BRect(0, (float)(WA::kPlTitleH*fScale),
                           (float)(WA::kPlW*fScale-1),
                           (float)(WA::kPlH*fScale-1)));
            return;
        }

        // sx=0 (fixed width), sy=0 (fixed height), r=fScale
        int r  = fScale;
        int sy = 0;  // extra 29px height increments beyond base

        // Left edge: PL_LFILL tiles
        DrawScaled(pl, kPL_LFill, 0, WA::kPlListY);
        for (int i = 1; i < sy + 2 * r; i++)
            DrawScaled(pl, kPL_LFill, 0, WA::kPlListY + 29 * i);

        // Right scroll strip: PL_RFILL tiles at x=255 (qmmp: slider at 255*r, 20*r)
        DrawScaled(pl, kPL_RFill, 255, WA::kPlListY);
        for (int i = 1; i < sy + 2 * r; i++)
            DrawScaled(pl, kPL_RFill, 255, WA::kPlListY + 29 * i);

        // Scroll knob on right strip (qmmp PL_BT_SCROLL_N/P)
        DrawScrollKnob(pl);

        // Bottom bar
        DrawScaled(pl, kPL_LSBar, 0,   WA::kPlBotY);
        DrawScaled(pl, kPL_RSBar, 125, WA::kPlBotY);

        // Bottom-bar buttons: ADD at (11,86), SUB at (40,86)
        DrawScaled(pl, fAddBtnPressed ? kPL_AddBtnP : kPL_AddBtn, 11, 86);
    }

    // DrawScaled takes 1x skin-pixel coords and multiplies by fScale internally.
    void DrawScrollKnob(BBitmap* pl) {
        if (!fListView) return;
        int total = fListView->TotalItems();
        int vis   = fListView->VisibleRowCount();
        if (total <= 0 || vis >= total) return;

        // All arithmetic in skin pixels (×fScale only at draw time via DrawScaled)
        int listH  = WA::kPlListH;         // 58 skin px — list area height
        int knobH  = 18;                   // PL_BT_SCROLL_N natural height in skin px
        int trackH = listH - knobH;        // 40 skin px travel range

        int scrollTop = fListView->ScrollTop();
        int maxTop    = total - vis;
        float frac    = (maxTop > 0) ? (float)scrollTop / maxTop : 0.0f;
        int p         = (int)(frac * trackH + 0.5f);  // 0..40 skin px

        // x=255+5=260 skin px from window left; y=kPlListY+p skin px from top
        DrawScaled(pl, kPL_ScrollN, 260, WA::kPlListY + p);
    }

    void ShowMenu(BPoint) {
        BPopUpMenu* m = new BPopUpMenu("plMenu", false, false);
        m->AddItem(new BMenuItem("Load Skin (.wsz)…", new BMessage(MSG_LOAD_SKIN)));
        m->AddItem(new BMenuItem("Exit Skin Mode",     new BMessage(MSG_CLEAR_SKIN)));
        m->AddSeparatorItem();
        BMenuItem* dbl = new BMenuItem("Double Size (2×)", new BMessage(MSG_SKIN_DOUBLE_SZ));
        dbl->SetMarked(fScale == 2);
        m->AddItem(dbl);
        BMenuItem* sm2 = new BMenuItem("Smooth Scaling", new BMessage(MSG_SKIN_SMOOTH));
        sm2->SetMarked(fSkin->CurrentSmooth());
        m->AddItem(sm2);
        m->SetTargetForItems(fTarget);
        BPoint gp = ConvertToScreen(BPoint(WA::kPlCloseBX*fScale, WA::kPlTitleH*fScale));
        m->Go(gp, true, true, true);
        delete m;
    }
};


class EqView : public BView {
public:
    static constexpr int kBands = 10;

    EqView(BRect frame, WinampSkin* skin, BMessenger target, int scale)
        : BView(frame, "eqView", B_FOLLOW_ALL, B_WILL_DRAW)
        , fSkin(skin), fScale(scale), fTarget(target)
    {
        SetViewColor(B_TRANSPARENT_COLOR);
        // Band values: 0.0 = flat (neutral)
        for (int i = 0; i < kBands; i++) fBands[i] = 0.0f;
        fPreamp  = 0.0f;
        fEnabled = false;
        fAutoMode = false;
    }

    void ReloadSkin(WinampSkin* skin, int scale) {
        fSkin  = skin;
        fScale = scale;
        Invalidate();
    }

    void Draw(BRect) override {
        DrawBackground();
        DrawTitlebar();
        DrawGraph();
        DrawSlider(WA::kEQ_PreampX, fPreamp, fSliderDrag == kPreampDrag && fPressed);
        for (int i = 0; i < kBands; i++)
            DrawSlider(WA::kEQ_SliderX0 + i * WA::kEQ_SliderStep, fBands[i],
                       fSliderDrag == i && fPressed);
        DrawButtons();
    }

    static constexpr int kNoDrag   = -2;
    static constexpr int kPreampDrag = -1;

    void MouseDown(BPoint pt) override {
        uint32 btns = 0; GetMouse(&pt, &btns, false);
        BPoint sp(pt.x / fScale, pt.y / fScale);

        // Close button
        if (InR(sp, WA::kEQ_CloseBX, WA::kEQ_CloseBY, 9, 9)) {
            Window()->Hide(); return;
        }
        // ON toggle
        if (InR(sp, 14, 18, 28, 12)) {
            fEnabled = !fEnabled;
            Invalidate(); return;
        }
        // AUTO toggle
        if (InR(sp, 39, 18, 33, 12)) {
            fAutoMode = !fAutoMode;
            Invalidate(); return;
        }
        // Titlebar drag — capture screen-space origin once
        if (sp.y < WA::kEQ_TitleH) {
            if (btns & B_SECONDARY_MOUSE_BUTTON) { ShowMenu(pt); return; }
            fWinDragging = true;
            fDragOrigin  = ConvertToScreen(pt);   // screen coords, captured once
            fWinOrigin   = Window()->Frame().LeftTop();
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS | B_NO_POINTER_HISTORY);
            return;
        }
        // Preamp slider
        if (HitSlider(sp, WA::kEQ_PreampX)) {
            fSliderDrag = kPreampDrag; fPressed = true;
            StartSliderDrag(pt.y, fPreamp);
            SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
            return;
        }
        // Band sliders
        for (int i = 0; i < kBands; i++) {
            if (HitSlider(sp, WA::kEQ_SliderX0 + i * WA::kEQ_SliderStep)) {
                fSliderDrag = i; fPressed = true;
                StartSliderDrag(pt.y, fBands[i]);
                SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
                return;
            }
        }
    }

    void MouseMoved(BPoint pt, uint32, const BMessage*) override {
        if (fWinDragging) {
            // Window drag: convert current mouse to screen, compute new window pos
            BPoint sc  = ConvertToScreen(pt);
            BPoint pos = fWinOrigin + sc - fDragOrigin;
            if (fDock) pos = fDock->Move(Window(), pos);
            else {
                BScreen screen; BRect sr = screen.Frame(); BRect wf = Window()->Frame();
                int t = WA::kSnapThresh;
                if (std::abs(pos.x)                            < t) pos.x = sr.left;
                if (std::abs(pos.y)                            < t) pos.y = sr.top;
                if (std::abs(pos.x + wf.Width()  - sr.right)  < t) pos.x = sr.right  - wf.Width();
                if (std::abs(pos.y + wf.Height() - sr.bottom) < t) pos.y = sr.bottom - wf.Height();
            }
            Window()->MoveTo(pos);
            return;
        }
        if (fSliderDrag == kNoDrag || !fPressed) return;
        // qmmp: po = e->y() - press_pos; value = convert(po)
        float po     = pt.y - fPressOffset;
        int   sliderH = WA::kEQ_SliderH * fScale;
        float travel  = float(sliderH - 12 * fScale);
        po = std::max(0.0f, std::min(po, travel));
        float m_value = (20.0f - (-20.0f)) * po / travel + (-20.0f); // = 40*po/travel - 20
        float newGain = -m_value;  // qmmp emits -m_value as gain
        newGain = std::max(-20.0f, std::min(20.0f, newGain));
        if (fSliderDrag == kPreampDrag) fPreamp              = newGain;
        else                            fBands[fSliderDrag]  = newGain;
        Invalidate();
    }

    void MouseUp(BPoint) override {
        if (fWinDragging && fDock) fDock->UpdateDock();
        fWinDragging = false;
        fSliderDrag  = kNoDrag;
        fPressed     = false;
    }

    void MessageReceived(BMessage* msg) override {
        if (msg->what == B_MOUSE_WHEEL_CHANGED) {
            // Not handled at view level; parent gets it
        }
        BView::MessageReceived(msg);
    }

private:
    void StartSliderDrag(float pressY, float currentGain) {
        int   s       = fScale;
        int   sliderH = WA::kEQ_SliderH * s;
        float travel  = float(sliderH - 12 * s);
        float m_value = -currentGain;
        float pF = (m_value - (-20.0f)) * travel / 40.0f;
        int   kp = (int)std::ceil(pF);
        kp = std::max(0, std::min((int)(sliderH - 11*s), kp));
        float localY = pressY - WA::kEQ_SliderY * s;
        if (kp < localY && localY < kp + 11 * s) {
            fPressOffset = pressY - kp;
        } else {
            // Clicked outside knob — jump to cursor center
            float po = localY - 6.0f * s;
            po = std::max(0.0f, std::min(po, travel));
            float m_val = 40.0f * po / travel - 20.0f;
            float newGain = -m_val;
            if (fSliderDrag == kPreampDrag) fPreamp             = newGain;
            else if (fSliderDrag >= 0)      fBands[fSliderDrag] = newGain;
            fPressOffset = pressY - po;  // press_pos = 6*r equivalent
        }
        Invalidate();
    }

    // Returns true if skin-px point sp hits the slider column at skinX
    bool HitSlider(BPoint sp, int skinX) {
        return sp.x >= skinX && sp.x < skinX + WA::kEQ_SliderW &&
               sp.y >= WA::kEQ_SliderY && sp.y < WA::kEQ_SliderY + WA::kEQ_SliderH;
    }

    // Draw the EQMAIN.BMP background (full 275×116)
    void DrawBackground() {
        BBitmap* bm = fSkin->Sheet("eqmain.bmp");
        if (!bm) {
            SetHighColor(20, 20, 40);
            FillRect(Bounds());
            return;
        }
        int s = fScale;
        SkinRect r = WA::kEQ_Main;
        BRect srcR((float)(r.x*s), (float)(r.y*s),
                   (float)((r.x+r.w)*s-1), (float)((r.y+r.h)*s-1));
        BRect dstR(0, 0, (float)(r.w*s-1), (float)(r.h*s-1));
        DrawBitmap(bm, srcR, dstR);
    }

    // Draw titlebar (EQ_TITLEBAR_A at y=134 in eqmain.bmp)
    void DrawTitlebar() {
        BBitmap* bm = fSkin->Sheet("eqmain.bmp");
        if (!bm) return;
        DrawScaled(bm, WA::kEQ_TitlebarA, 0, 0);
        // Close button on titlebar
        DrawScaled(bm, WA::kEQ_CloseN, WA::kEQ_CloseBX, WA::kEQ_CloseBY);
        // Shade button
        DrawScaled(bm, WA::kEQ_ShadeN, 254, WA::kEQ_CloseBY);
    }

    // Draw ON/OFF, AUTO, PRESETS buttons
    void DrawButtons() {
        BBitmap* bm = fSkin->Sheet("eqmain.bmp");
        if (!bm) return;
        // ON/OFF toggle — qmmp: SkinnedToggleButton ON_N/OFF_N
        DrawScaled(bm, fEnabled ? WA::kEQ_OnN : WA::kEQ_OffN, 14, 18);
        // AUTO toggle
        DrawScaled(bm, fAutoMode ? WA::kEQ_Auto1N : WA::kEQ_Auto0N, 39, 18);
        // PRESETS button
        DrawScaled(bm, WA::kEQ_PresetsN, 217, 18);
    }

    void DrawSlider(int skinX, float gainDb, bool pressed) {
        BBitmap* bm = fSkin->Sheet("eqmain.bmp");
        if (!bm) return;
        int s = fScale;

        // Convert gain to qmmp internal value
        float m_value = -gainDb;
        const float m_min = -20.0f, m_max = 20.0f;
        m_value = std::max(m_min, std::min(m_max, m_value));

        // idx = 27 - 27*(m_value-m_min)/(m_max-m_min), clamped 0..27
        float idxF   = 27.0f - 27.0f * (m_value - m_min) / (m_max - m_min);
        int   idx    = (int)std::ceil(idxF);
        idx = std::max(0, std::min(27, idx));

        // Map idx to source row (0..13 = normal y=164, 14..27 = alt y=229)
        int   frame  = idx % 14;
        int   srcY   = (idx < 14) ? WA::kEQ_SliderY0 : WA::kEQ_SliderY1;
        int   srcX   = 13 + frame * 15;  // skin px

        // Blit strip
        int sw = WA::kEQ_SliderW * s, sh = WA::kEQ_SliderH * s;
        BRect srcStrip((float)(srcX*s),        (float)(srcY*s),
                       (float)(srcX*s + sw-1), (float)(srcY*s + sh-1));
        BRect dstStrip((float)(skinX*s),                         (float)(WA::kEQ_SliderY*s),
                       (float)(skinX*s + sw-1),                  (float)(WA::kEQ_SliderY*s + sh-1));
        DrawBitmap(bm, srcStrip, dstStrip);

        // p = ceil((m_value - m_min) * (sliderH - 12*s) / (m_max - m_min))
        float pF = (m_value - m_min) * float(sh - 12*s) / (m_max - m_min);
        int   p  = (int)std::ceil(pF);
        p = std::max(0, std::min(sh - 11*s, p));

        // Knob sprite: EQ_BT_BAR_N at (0,164) normal, EQ_BT_BAR_P at (0,176) pressed
        SkinRect knobRect = pressed ? WA::kEQ_BarP : WA::kEQ_BarN;
        int kw = knobRect.w * s, kh = knobRect.h * s;
        BRect knobSrc((float)(knobRect.x*s),      (float)(knobRect.y*s),
                      (float)(knobRect.x*s+kw-1), (float)(knobRect.y*s+kh-1));
        // x = skinX*s + 1*s  (qmmp: paint.drawPixmap(1, p, knob))
        BRect knobDst((float)(skinX*s + s),        (float)(WA::kEQ_SliderY*s + p),
                      (float)(skinX*s + s + kw-1), (float)(WA::kEQ_SliderY*s + p + kh-1));
        DrawBitmap(bm, knobSrc, knobDst);
    }

    void DrawGraph() {
        BBitmap* bm = fSkin->Sheet("eqmain.bmp");
        if (!bm) return;
        int s = fScale;
        // Draw graph background (EQ_GRAPH at 0,294 in eqmain.bmp)
        SkinRect gr = WA::kEQ_Graph;
        int gx = WA::kEQ_GraphX * s, gy = WA::kEQ_GraphY * s;
        int gw = gr.w * s, gh = gr.h * s;
        BRect srcG((float)(gr.x*s), (float)(gr.y*s),
                   (float)((gr.x+gr.w)*s-1), (float)((gr.y+gr.h)*s-1));
        BRect dstG((float)gx, (float)gy, (float)(gx+gw-1), (float)(gy+gh-1));
        DrawBitmap(bm, srcG, dstG);

        static const int kBandX[10] = {0,11,23,35,47,59,71,83,97,109};
        float rowY[10];
        for (int i = 0; i < 10; i++) {
            float row = 9.0f - fBands[i] * 9.0f / 20.0f;
            rowY[i] = std::max(0.0f, std::min(18.0f, row));
        }

        // Draw polyline through band positions using spline colour at that row
        SetDrawingMode(B_OP_OVER);
        for (int i = 0; i < 9; i++) {
            // Get colour from eqmain.bmp spline pixel at (115, 294+row)
            // Use the colour at the midpoint row
            int midRow = (int)((rowY[i] + rowY[i+1]) * 0.5f + 0.5f);
            int splinePixX = 115 * s;
            int splinePixY = (294 + midRow) * s;
            // Sample colour from bitmap
            rgb_color col = SampleColor(bm, splinePixX, splinePixY);
            SetHighColor(col);

            float x1 = gx + kBandX[i]   * gw / 113.0f;
            float x2 = gx + kBandX[i+1] * gw / 113.0f;
            float y1 = gy + rowY[i]   * gh / 19.0f;
            float y2 = gy + rowY[i+1] * gh / 19.0f;
            StrokeLine(BPoint(x1, y1), BPoint(x2, y2));
        }
        SetDrawingMode(B_OP_COPY);
    }

    // Sample a pixel colour from a BBitmap at screen coords (px, py)
    rgb_color SampleColor(BBitmap* bm, int px, int py) {
        if (!bm) return {100, 180, 100, 255};
        int w = (int)bm->Bounds().Width() + 1;
        int h = (int)bm->Bounds().Height() + 1;
        px = std::max(0, std::min(px, w-1));
        py = std::max(0, std::min(py, h-1));
        color_space cs = bm->ColorSpace();
        const uint8* bits = (const uint8*)bm->Bits();
        int bpr = bm->BytesPerRow();
        if (cs == B_RGB32 || cs == B_RGBA32) {
            const uint8* p = bits + py * bpr + px * 4;
            return {p[2], p[1], p[0], 255};
        }
        return {100, 180, 100, 255};
    }

    // Blit a SkinRect from eqmain.bmp to (dstX*s, dstY*s)
    void DrawScaled(BBitmap* src, SkinRect r, int dstX, int dstY) {
        if (!src) return;
        int s = fScale;
        int sx1 = r.x*s, sy1 = r.y*s;
        int sw  = r.w*s, sh  = r.h*s;
        int bmW = (int)src->Bounds().Width()+1, bmH = (int)src->Bounds().Height()+1;
        sw = std::min(sw, bmW - sx1); sh = std::min(sh, bmH - sy1);
        if (sw <= 0 || sh <= 0) return;
        BRect srcR((float)sx1, (float)sy1, (float)(sx1+sw-1), (float)(sy1+sh-1));
        float dx = (float)(dstX*s), dy = (float)(dstY*s);
        DrawBitmap(src, srcR, BRect(dx, dy, dx+sw-1, dy+sh-1));
    }

    void ShowMenu(BPoint) {
        BPopUpMenu* m = new BPopUpMenu("eqMenu", false, false);
        m->AddItem(new BMenuItem("Load Skin (.wsz)…", new BMessage(MSG_LOAD_SKIN)));
        m->AddItem(new BMenuItem("Exit Skin Mode",     new BMessage(MSG_CLEAR_SKIN)));
        m->AddSeparatorItem();
        BMenuItem* dbl = new BMenuItem("Double Size (2×)", new BMessage(MSG_SKIN_DOUBLE_SZ));
        dbl->SetMarked(fScale == 2);
        m->AddItem(dbl);
        BMenuItem* sm = new BMenuItem("Smooth Scaling", new BMessage(MSG_SKIN_SMOOTH));
        sm->SetMarked(fSkin->CurrentSmooth());
        m->AddItem(sm);
        BMessenger target(nullptr, Window()->Looper());
        // Find the app messenger for menu dispatch
        m->SetTargetForItems(fTarget);
        BPoint gp = ConvertToScreen(BPoint(WA::kEQ_CloseBX * fScale,
                                            WA::kEQ_TitleH  * fScale));
        m->Go(gp, true, true, true);
        delete m;
    }

public:
    WinampSkin* fSkin;
    int         fScale;
    SkinDock*   fDock = nullptr;
private:
    BMessenger  fTarget;
    float       fBands[kBands];
    float       fPreamp    = 0.0f;
    bool        fEnabled   = false;
    bool        fAutoMode  = false;
    bool        fWinDragging  = false;   // titlebar window drag
    int         fSliderDrag   = kNoDrag; // kNoDrag, kPreampDrag, or 0..9
    bool        fPressed      = false;
    float       fPressOffset  = 0;       // qmmp press_pos: offset within knob
    BPoint      fDragOrigin, fWinOrigin;
};

class EqWindow : public BWindow {
public:
    EqView* fView = nullptr;

    EqWindow(BPoint pos, WinampSkin* skin, BMessenger target, int scale)
        : BWindow(BRect(pos.x, pos.y,
                        pos.x + WA::kEQ_W * scale - 1,
                        pos.y + WA::kEQ_H * scale - 1),
                  "PandAmp EQ",
                  B_NO_BORDER_WINDOW_LOOK,
                  B_NORMAL_WINDOW_FEEL,
                  B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_WILL_ACCEPT_FIRST_CLICK)
    {
        fView = new EqView(Bounds(), skin, target, scale);
        AddChild(fView);
    }

    void SetDock(SkinDock* d) {
        if (LockLooper()) { fView->fDock = d; UnlockLooper(); }
    }

    void ReloadSkin(WinampSkin* skin, int newScale) {
        if (LockLooper()) {
            fView->fSkin  = skin;
            fView->fScale = newScale;
            float nw = (float)(WA::kEQ_W * newScale) - 1;
            float nh = (float)(WA::kEQ_H * newScale) - 1;
            ResizeTo(nw, nh);
            fView->ResizeTo(nw, nh);
            fView->Invalidate();
            UnlockLooper();
        }
    }

    bool QuitRequested() override { Hide(); return false; }
};

class PlaylistWindow : public BWindow {
public:
    PlaylistView* fView = nullptr;

    PlaylistWindow(BPoint pos, WinampSkin* skin, BMessenger target, int scale)
        : BWindow(BRect(pos.x, pos.y,
                        pos.x + WA::kPlW  * scale - 1,
                        pos.y + WA::kPlH  * scale - 1),
                  "PandAmp Stations",
                  B_NO_BORDER_WINDOW_LOOK,
                  B_NORMAL_WINDOW_FEEL,
                  B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_WILL_ACCEPT_FIRST_CLICK)
    {
        fView = new PlaylistView(Bounds(), skin, target, scale);
        AddChild(fView);
    }

    SkinListView* fList() { return fView ? fView->fListView : nullptr; }
    void SetDock(SkinDock* d) {
        if (LockLooper()) { fView->fDock = d; UnlockLooper(); }
    }

    void ReloadSkin(int newScale) {
        if (LockLooper()) {
            fView->fScale = newScale;
            float newW = (float)(WA::kPlW * newScale) - 1;
            float newH = (float)(WA::kPlH * newScale) - 1;
            ResizeTo(newW, newH);
            fView->ResizeTo(newW, newH);
            // Reposition + reload the custom list widget
            if (fView->fListView) {
                float lx = (float)(WA::kPlListX * newScale);
                float ly = (float)(WA::kPlTitleH * newScale);
                float lw = (float)(WA::kPlListW * newScale);
                float lh = (float)(WA::kPlListH * newScale);
                fView->fListView->MoveTo(lx, ly);
                fView->fListView->ResizeTo(lw - 1, lh - 1);
                fView->fListView->ReloadSkin(fView->fSkin, newScale);
            }
            fView->Invalidate();
            UnlockLooper();
        }
    }

    bool QuitRequested() override { Hide(); return false; }
};
