#pragma once
#include <string>
#include <vector>
#include <cstdint>
#include <functional>

namespace PandoraEncryption {
    // Official pianobar/pandora keys (publicly documented)
    static constexpr const char* DECRYPT_KEY = "R=U!LH$O2B#";
    static constexpr const char* ENCRYPT_KEY = "6#26FRL$ZWD";
    static constexpr const char* PARTNER_USER = "android";
    static constexpr const char* PARTNER_PASSWORD = "AC7IBG09A3DTSYM4R41UJWL07VLN8JI7";
    static constexpr const char* DEVICE_MODEL = "android-generic";
    static constexpr const char* PARTNER_VERSION = "5";
}

struct PandoraTrack {
    std::string songName;
    std::string artistName;
    std::string albumName;
    std::string albumArtUrl;
    std::string audioUrl;
    std::string trackToken;
    std::string stationId;
    int         rating;       // 0=none, 1=thumbUp, -1=thumbDown
    int         durationSecs;
    bool        allowSkip;
};

struct PandoraStation {
    std::string stationId;
    std::string stationToken;
    std::string stationName;
    bool        isQuickMix;
};

// Async callback types
using AuthCallback    = std::function<void(bool ok, const std::string& error)>;
using StationsCallback = std::function<void(bool ok, const std::vector<PandoraStation>&, const std::string& error)>;
using PlaylistCallback = std::function<void(bool ok, const std::vector<PandoraTrack>&, const std::string& error)>;
using FeedbackCallback = std::function<void(bool ok, const std::string& error)>;

struct SearchResult {
    std::string name;        // artist or song name
    std::string musicToken;  // token to pass to CreateStation
    bool        isArtist;    // true = artist, false = song
};

using SearchCallback        = std::function<void(bool ok, const std::vector<SearchResult>&, const std::string& error)>;
using CreateStationCallback = std::function<void(bool ok, const PandoraStation&, const std::string& error)>;

class PandoraAPI {
public:
    PandoraAPI();
    ~PandoraAPI();

    // Authentication
    void Login(const std::string& email, const std::string& password, AuthCallback cb);
    void Logout();
    bool IsLoggedIn() const { return fLoggedIn; }

    // Station management
    void GetStations(StationsCallback cb);
    void GetPlaylist(const std::string& stationToken, PlaylistCallback cb);

    // Playback feedback
    void ThumbsUp(const std::string& trackToken, FeedbackCallback cb);
    void ThumbsDown(const std::string& trackToken, FeedbackCallback cb);
    void TiredOfSong(const std::string& trackToken, FeedbackCallback cb);

    // Station search & creation
    void SearchMusic(const std::string& query, SearchCallback cb);
    void CreateStation(const std::string& musicToken, CreateStationCallback cb);

    // Save credentials to disk
    void SaveCredentials(const std::string& email, const std::string& password);
    bool LoadCredentials(std::string& email, std::string& password);

private:
    // Internal network
    std::string HttpPost(const std::string& url,
                         const std::string& body,
                         bool encrypted = false,
                         const std::string& contentType = "application/json");

    // Pandora protocol helpers
    std::string BlowfishEncrypt(const std::string& data);
    std::string BlowfishDecrypt(const std::string& hex);
    std::string HexEncode(const std::string& bytes);
    std::string HexDecode(const std::string& hex);

    // Partner login (step 1 of 2-step auth)
    bool DoPartnerLogin();

    // JSON helpers
    std::string JsonGetString(const std::string& json, const std::string& key);
    std::string JsonGetStringInResult(const std::string& json, const std::string& key);
    std::vector<std::string> JsonGetArray(const std::string& json, const std::string& arrayKey);

    // Build API request URL
    std::string BuildUrl(const std::string& method);

    // Session state
    bool        fLoggedIn;
    std::string fPartnerId;
    std::string fPartnerAuthToken;
    std::string fUserId;
    std::string fUserAuthToken;
    uint64_t    fSyncTimeOffset;  // server time - local time offset

    // Credential storage path
    std::string fCredentialPath;
};
