#include "PandoraAPI.h"
#include "Blowfish.h"

#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

#include <ctime>
#include <cstring>
#include <sstream>
#include <algorithm>
#include <fstream>
#include <stdexcept>

static std::string HttpsPost(const std::string& host,
                              const std::string& path,
                              const std::string& body,
                              const std::string& contentType = "application/json")
{
    SSL_CTX* ctx = SSL_CTX_new(TLS_client_method());
    if (!ctx) return "";
    SSL_CTX_set_default_verify_paths(ctx);

    struct addrinfo hints{}, *res = nullptr;
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    getaddrinfo(host.c_str(), "443", &hints, &res);

    int sock = -1;
    for (auto* r = res; r; r = r->ai_next) {
        sock = socket(r->ai_family, r->ai_socktype, r->ai_protocol);
        if (sock < 0) continue;
        if (connect(sock, r->ai_addr, r->ai_addrlen) == 0) break;
        close(sock); sock = -1;
    }
    freeaddrinfo(res);
    if (sock < 0) { SSL_CTX_free(ctx); return ""; }

    SSL* ssl = SSL_new(ctx);
    SSL_set_fd(ssl, sock);
    SSL_set_tlsext_host_name(ssl, host.c_str());
    if (SSL_connect(ssl) <= 0) {
        SSL_free(ssl); close(sock); SSL_CTX_free(ctx); return "";
    }

    std::ostringstream req;
    req << "POST " << path << " HTTP/1.1\r\n"
        << "Host: " << host << "\r\n"
        << "Content-Type: " << contentType << "\r\n"
        << "Content-Length: " << body.size() << "\r\n"
        << "Connection: close\r\n\r\n"
        << body;
    std::string reqStr = req.str();
    SSL_write(ssl, reqStr.c_str(), reqStr.size());

    std::string response;
    char buf[4096];
    int n;
    while ((n = SSL_read(ssl, buf, sizeof(buf))) > 0)
        response.append(buf, n);

    SSL_shutdown(ssl); SSL_free(ssl);
    close(sock); SSL_CTX_free(ctx);

    auto pos = response.find("\r\n\r\n");
    if (pos != std::string::npos)
        response = response.substr(pos + 4);
    // Handle chunked transfer
    if (response.find("Transfer-Encoding: chunked") != std::string::npos ||
        response[0] == '\r' || (response.size() > 1 && isxdigit(response[0]))) {
        // Simple dechunk
        std::string dechunked;
        size_t i = 0;
        while (i < response.size()) {
            size_t nl = response.find("\r\n", i);
            if (nl == std::string::npos) break;
            int chunkLen = strtol(response.c_str()+i, nullptr, 16);
            if (chunkLen == 0) break;
            i = nl + 2;
            dechunked += response.substr(i, chunkLen);
            i += chunkLen + 2;
        }
        if (!dechunked.empty()) response = dechunked;
    }
    return response;
}
//JSSSSONNNNNNNN! JSSSSONNNNN!
static std::string JsonStr(const std::string& json, const std::string& key) {
    std::string search = "\"" + key + "\"";
    auto pos = json.find(search);
    if (pos == std::string::npos) return "";
    pos = json.find(":", pos + search.size());
    if (pos == std::string::npos) return "";
    pos = json.find("\"", pos + 1);
    if (pos == std::string::npos) return "";
    auto end = json.find("\"", pos + 1);
    while (end != std::string::npos && json[end-1] == '\\') end = json.find("\"", end+1);
    if (end == std::string::npos) return "";
    return json.substr(pos+1, end-pos-1);
}

static std::string JsonNum(const std::string& json, const std::string& key) {
    std::string search = "\"" + key + "\"";
    auto pos = json.find(search);
    if (pos == std::string::npos) return "";
    pos = json.find(":", pos + search.size());
    if (pos == std::string::npos) return "";
    while (pos < json.size() && (json[pos] == ':' || json[pos] == ' ')) pos++;
    auto end = pos;
    while (end < json.size() && (isdigit(json[end]) || json[end]=='-')) end++;
    return json.substr(pos, end-pos);
}

static std::vector<std::string> JsonArrayObjects(const std::string& json, const std::string& key) {
    std::vector<std::string> result;
    std::string search = "\"" + key + "\"";
    auto pos = json.find(search);
    if (pos == std::string::npos) return result;
    pos = json.find("[", pos);
    if (pos == std::string::npos) return result;
    int depth = 0;
    size_t objStart = 0;
    for (size_t i = pos; i < json.size(); i++) {
        if (json[i] == '{') {
            if (depth == 0) objStart = i;
            depth++;
        } else if (json[i] == '}') {
            depth--;
            if (depth == 0) result.push_back(json.substr(objStart, i-objStart+1));
        } else if (json[i] == ']' && depth == 0) break;
    }
    return result;
}

PandoraAPI::PandoraAPI()
    : fLoggedIn(false), fSyncTimeOffset(0)
{
    const char* home = getenv("HOME");
    fCredentialPath = home ? std::string(home) + "/config/settings/PandAmp/credentials"
                           : "/tmp/pandamp_creds";
}

PandoraAPI::~PandoraAPI() {}

std::string PandoraAPI::BuildUrl(const std::string& method) {
    std::ostringstream url;
    url << "/services/json/?method=" << method;
    if (!fPartnerId.empty())   url << "&partner_id=" << fPartnerId;
    if (!fUserId.empty())      url << "&user_id=" << fUserId;
    if (!fUserAuthToken.empty()) url << "&auth_token=" << fUserAuthToken;
    return url.str();
}

bool PandoraAPI::DoPartnerLogin() {
    std::string body = R"({"username":")" + std::string(PandoraEncryption::PARTNER_USER) +
                       R"(","password":")" + std::string(PandoraEncryption::PARTNER_PASSWORD) +
                       R"(","deviceModel":")" + std::string(PandoraEncryption::DEVICE_MODEL) +
                       R"(","version":"5","includeUrls":true})";

    std::string resp = HttpsPost("tuner.pandora.com",
                                 "/services/json/?method=auth.partnerLogin",
                                 body);
    if (resp.empty() || resp.find("\"ok\"") == std::string::npos) return false;

    fPartnerId        = JsonStr(resp, "partnerId");
    fPartnerAuthToken = JsonStr(resp, "partnerAuthToken");
    std::string syncTimeHex = JsonStr(resp, "syncTime");

    Blowfish bf;
    std::string decKey(PandoraEncryption::DECRYPT_KEY);
    bf.Init((const uint8_t*)decKey.c_str(), decKey.size());
    std::string decrypted = bf.DecryptHex(syncTimeHex);
    if (decrypted.size() > 4) {
        std::string timeStr = decrypted.substr(4);
        uint64_t serverTime = 0;
        for (char c : timeStr) {
            if (isdigit(c)) serverTime = serverTime*10 + (c-'0');
            else break;
        }
        fSyncTimeOffset = serverTime - (uint64_t)time(nullptr);
    }
    return true;
}

void PandoraAPI::Login(const std::string& email, const std::string& password, AuthCallback cb) {
    if (!DoPartnerLogin()) { cb(false, "Partner login failed"); return; }

    uint64_t syncTime = (uint64_t)time(nullptr) + fSyncTimeOffset;
    std::ostringstream body;
    body << R"({"loginType":"user","username":")" << email
         << R"(","password":")" << password
         << R"(","partnerAuthToken":")" << fPartnerAuthToken
         << R"(","syncTime":)" << syncTime
         << R"(,"includePandoraOneInfo":true,"includeSubscriptionExpiration":true,"includeAdAttributes":true,"includeStationArtUrl":true,"returnStationList":true,"includeStationSeeds":true,"returnGenreStations":true,"includeShuffleInsteadOfQuickMix":true})";

    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());

    std::string path = "/services/json/?method=auth.userLogin&partner_id=" + fPartnerId +
                       "&auth_token=" + fPartnerAuthToken;
    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted, "application/json");

    if (resp.empty() || resp.find("\"ok\"") == std::string::npos) {
        std::string msg = JsonStr(resp, "message");
        cb(false, msg.empty() ? "Login failed" : msg);
        return;
    }

    fUserId        = JsonStr(resp, "userId");
    fUserAuthToken = JsonStr(resp, "userAuthToken");
    fLoggedIn = true;
    cb(true, "");
}

void PandoraAPI::Logout() {
    fLoggedIn = false;
    fUserId.clear();
    fUserAuthToken.clear();
    fPartnerAuthToken.clear();
    fPartnerId.clear();
}

void PandoraAPI::GetStations(StationsCallback cb) {
    if (!fLoggedIn) { cb(false, {}, "Not logged in"); return; }

    uint64_t syncTime = (uint64_t)time(nullptr) + fSyncTimeOffset;
    std::ostringstream body;
    body << R"({"userAuthToken":")" << fUserAuthToken
         << R"(","syncTime":)" << syncTime << "}";

    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());

    std::string path = "/services/json/?method=user.getStationList"
                       "&partner_id=" + fPartnerId +
                       "&user_id=" + fUserId +
                       "&auth_token=" + fUserAuthToken;

    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted);
    if (resp.find("\"ok\"") == std::string::npos) {
        cb(false, {}, JsonStr(resp, "message"));
        return;
    }

    auto stationObjs = JsonArrayObjects(resp, "stations");
    std::vector<PandoraStation> stations;
    for (auto& obj : stationObjs) {
        PandoraStation s;
        s.stationId    = JsonStr(obj, "stationId");
        s.stationToken = JsonStr(obj, "stationToken");
        s.stationName  = JsonStr(obj, "stationName");
        std::string qm = JsonStr(obj, "isQuickMix");
        s.isQuickMix = (qm == "true");
        if (!s.stationId.empty())
            stations.push_back(s);
    }
    cb(true, stations, "");
}

void PandoraAPI::GetPlaylist(const std::string& stationToken, PlaylistCallback cb) {
    if (!fLoggedIn) { cb(false, {}, "Not logged in"); return; }

    uint64_t syncTime = (uint64_t)time(nullptr) + fSyncTimeOffset;
    std::ostringstream body;
    body << R"({"stationToken":")" << stationToken
         << R"(","userAuthToken":")" << fUserAuthToken
         << R"(","syncTime":)" << syncTime
         << R"(,"additionalAudioUrl":"HTTP_128_MP3","audioAdPodcastSupport":false})";

    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());

    std::string path = "/services/json/?method=station.getPlaylist"
                       "&partner_id=" + fPartnerId +
                       "&user_id=" + fUserId +
                       "&auth_token=" + fUserAuthToken;

    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted);
    if (resp.find("\"ok\"") == std::string::npos) {
        cb(false, {}, JsonStr(resp, "message"));
        return;
    }

    auto items = JsonArrayObjects(resp, "items");
    std::vector<PandoraTrack> tracks;
    for (auto& obj : items) {
        if (obj.find("songName") == std::string::npos) continue;
        PandoraTrack t;
        t.songName    = JsonStr(obj, "songName");
        t.artistName  = JsonStr(obj, "artistName");
        t.albumName   = JsonStr(obj, "albumName");
        t.albumArtUrl = JsonStr(obj, "albumArtUrl");
        t.trackToken  = JsonStr(obj, "trackToken");
        t.audioUrl    = JsonStr(obj, "audioUrl");
        std::string au2 = JsonStr(obj, "additionalAudioUrl");
        if (!au2.empty()) t.audioUrl = au2;
        std::string skip = JsonStr(obj, "allowSkip");
        t.allowSkip = (skip != "false");
        std::string dur = JsonNum(obj, "trackLength");
        t.durationSecs = dur.empty() ? 0 : std::stoi(dur);
        std::string rating = JsonNum(obj, "songRating");
        t.rating = rating.empty() ? 0 : std::stoi(rating);
        if (!t.songName.empty())
            tracks.push_back(t);
    }
    cb(true, tracks, "");
}

static void SendFeedback(const std::string& trackToken, bool isPositive,
                          const std::string& partnerId, const std::string& userId,
                          const std::string& userAuthToken, uint64_t syncTimeOffset,
                          FeedbackCallback cb) {
    uint64_t syncTime = (uint64_t)time(nullptr) + syncTimeOffset;
    std::ostringstream body;
    body << R"({"trackToken":")" << trackToken
         << R"(","isPositive":)" << (isPositive ? "true" : "false")
         << R"(,"userAuthToken":")" << userAuthToken
         << R"(","syncTime":)" << syncTime << "}";

    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());

    std::string path = "/services/json/?method=station.addFeedback"
                       "&partner_id=" + partnerId +
                       "&user_id=" + userId +
                       "&auth_token=" + userAuthToken;

    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted);
    bool ok = resp.find("\"ok\"") != std::string::npos;
    cb(ok, ok ? "" : JsonStr(resp, "message"));
}

void PandoraAPI::ThumbsUp(const std::string& trackToken, FeedbackCallback cb) {
    SendFeedback(trackToken, true, fPartnerId, fUserId, fUserAuthToken, fSyncTimeOffset, cb);
}

void PandoraAPI::ThumbsDown(const std::string& trackToken, FeedbackCallback cb) {
    SendFeedback(trackToken, false, fPartnerId, fUserId, fUserAuthToken, fSyncTimeOffset, cb);
}

void PandoraAPI::TiredOfSong(const std::string& trackToken, FeedbackCallback cb) {
    if (!fLoggedIn) { cb(false, "Not logged in"); return; }
    uint64_t syncTime = (uint64_t)time(nullptr) + fSyncTimeOffset;
    std::ostringstream body;
    body << R"({"trackToken":")" << trackToken
         << R"(","userAuthToken":")" << fUserAuthToken
         << R"(","syncTime":)" << syncTime << "}";
    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());
    std::string path = "/services/json/?method=user.sleepSong&partner_id=" + fPartnerId +
                       "&user_id=" + fUserId + "&auth_token=" + fUserAuthToken;
    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted);
    bool ok = resp.find("\"ok\"") != std::string::npos;
    cb(ok, ok ? "" : JsonStr(resp, "message"));
}

void PandoraAPI::SearchMusic(const std::string& query, SearchCallback cb) {
    if (!fLoggedIn) { cb(false, {}, "Not logged in"); return; }

    // URL-encode the query (basic: replace spaces with +, encode specials)
    std::string encoded;
    for (unsigned char ch : query) {
        if (std::isalnum(ch) || ch == '-' || ch == '_' || ch == '.' || ch == '~')
            encoded += ch;
        else if (ch == ' ')
            encoded += '+';
        else {
            char buf[4];
            snprintf(buf, sizeof(buf), "%%%02X", ch);
            encoded += buf;
        }
    }

    uint64_t syncTime = (uint64_t)time(nullptr) + fSyncTimeOffset;
    std::ostringstream body;
    body << R"({"searchText":")" << query
         << R"(","userAuthToken":")" << fUserAuthToken
         << R"(","syncTime":)" << syncTime << "}";

    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());

    std::string path = "/services/json/?method=music.search"
                       "&partner_id=" + fPartnerId +
                       "&user_id="    + fUserId +
                       "&auth_token=" + fUserAuthToken;

    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted);
    if (resp.find("\"ok\"") == std::string::npos) {
        cb(false, {}, JsonStr(resp, "message"));
        return;
    }

    std::vector<SearchResult> results;

    // Artists
    auto artists = JsonArrayObjects(resp, "artists");
    for (auto& obj : artists) {
        SearchResult r;
        r.name       = JsonStr(obj, "artistName");
        r.musicToken = JsonStr(obj, "musicToken");
        r.isArtist   = true;
        if (!r.musicToken.empty()) results.push_back(r);
    }

    // Songs
    auto songs = JsonArrayObjects(resp, "songs");
    for (auto& obj : songs) {
        SearchResult r;
        r.name       = JsonStr(obj, "songName") + " — " + JsonStr(obj, "artistName");
        r.musicToken = JsonStr(obj, "musicToken");
        r.isArtist   = false;
        if (!r.musicToken.empty()) results.push_back(r);
    }

    cb(true, results, "");
}

void PandoraAPI::CreateStation(const std::string& musicToken, CreateStationCallback cb) {
    if (!fLoggedIn) { cb(false, {}, "Not logged in"); return; }

    uint64_t syncTime = (uint64_t)time(nullptr) + fSyncTimeOffset;
    std::ostringstream body;
    body << R"({"musicToken":")" << musicToken
         << R"(","userAuthToken":")" << fUserAuthToken
         << R"(","syncTime":)" << syncTime << "}";

    Blowfish bf;
    std::string encKey(PandoraEncryption::ENCRYPT_KEY);
    bf.Init((const uint8_t*)encKey.c_str(), encKey.size());
    std::string encrypted = bf.EncryptToHex(body.str());

    std::string path = "/services/json/?method=station.createStation"
                       "&partner_id=" + fPartnerId +
                       "&user_id="    + fUserId +
                       "&auth_token=" + fUserAuthToken;

    std::string resp = HttpsPost("tuner.pandora.com", path, encrypted);
    if (resp.find("\"ok\"") == std::string::npos) {
        cb(false, {}, JsonStr(resp, "message"));
        return;
    }

    PandoraStation s;
    s.stationId    = JsonStr(resp, "stationId");
    s.stationToken = JsonStr(resp, "stationToken");
    s.stationName  = JsonStr(resp, "stationName");
    s.isQuickMix   = false;
    cb(true, s, "");
}

void PandoraAPI::SaveCredentials(const std::string& email, const std::string& password) {
    // Create directory
    std::string dir = fCredentialPath.substr(0, fCredentialPath.rfind('/'));
    std::string cmd = "mkdir -p " + dir;
    system(cmd.c_str());
    std::ofstream f(fCredentialPath);
    if (f.is_open()) {
        f << email << "\n" << password << "\n";
    }
}

bool PandoraAPI::LoadCredentials(std::string& email, std::string& password) {
    std::ifstream f(fCredentialPath);
    if (!f.is_open()) return false;
    std::getline(f, email);
    std::getline(f, password);
    return !email.empty() && !password.empty();
}
