#include "MainWindow.h"
#include "SkinWindow.h"

// minimp3: public-domain single-header MP3 decoder by lieff
// Get it here! It's boss : https://github.com/lieff/minimp3/blob/master/minimp3.h
#define MINIMP3_IMPLEMENTATION
#include "minimp3.h"

#include <Alert.h>
#include <Application.h>
#include <Bitmap.h>
#include <Box.h>
#include <FilePanel.h>
#include <Font.h>
#include <LayoutBuilder.h>
#include <MenuBar.h>
#include <Menu.h>
#include <MenuItem.h>
#include <Path.h>
#include <File.h>
#include <media/MediaDefs.h>
#include <Screen.h>
#include <ScrollView.h>
#include <StringItem.h>
#include <StringView.h>
#include <translation/TranslationUtils.h>
#include <storage/FindDirectory.h>

#include <arpa/inet.h>
#include <netdb.h>
#include <openssl/ssl.h>
#include <sys/socket.h>
#include <unistd.h>

#include <algorithm>
#include <cstdio>
#include <cstring>
#include <thread>

// Global PCM audio buffer (decode thread writes, BSoundPlayer callback reads)

namespace {
    struct AudioBuf {
        std::vector<int16_t> pcm;
        size_t               pos      = 0;
        std::atomic<bool>    ready{false};
        std::atomic<bool>    done{false};
        int                  hz       = 44100;
        int                  ch       = 2;
        std::mutex           mtx;
    } gAudio;

    // Balance: -100 (full left) to +100 (full right), 0 = center
    std::atomic<int> gBalance{0};

    void FillAudio(void*, void* buf, size_t size, const media_raw_audio_format&)
    {
        int16_t* dst   = static_cast<int16_t*>(buf);
        size_t   count = size / sizeof(int16_t);
        if (!gAudio.ready.load()) { memset(dst, 0, size); return; }
        std::lock_guard<std::mutex> lk(gAudio.mtx);
        size_t avail  = gAudio.pcm.size() - gAudio.pos;
        size_t toCopy = std::min(count, avail);
        memcpy(dst, gAudio.pcm.data() + gAudio.pos, toCopy * sizeof(int16_t));
        if (toCopy < count) memset(dst + toCopy, 0, (count - toCopy) * sizeof(int16_t));
        gAudio.pos += toCopy;
        if (gAudio.pos >= gAudio.pcm.size()) gAudio.done = true;
        // Apply balance: scale L or R channel down proportionally
        int bal = gBalance.load();
        if (bal != 0 && gAudio.ch == 2) {
            // bal > 0 → attenuate left; bal < 0 → attenuate right
            float lGain = (bal > 0) ? 1.0f - bal / 100.0f : 1.0f;
            float rGain = (bal < 0) ? 1.0f + bal / 100.0f : 1.0f;
            size_t frames = count / 2;
            for (size_t i = 0; i < frames; ++i) {
                dst[i*2  ] = (int16_t)(dst[i*2  ] * lGain);
                dst[i*2+1] = (int16_t)(dst[i*2+1] * rGain);
            }
        }
    }
}

void VisAudioSnapshot(float* out, int count) {
    if (!out || count <= 0) return;
    if (!gAudio.ready.load()) { memset(out, 0, count * sizeof(float)); return; }
    std::lock_guard<std::mutex> lk(gAudio.mtx);
    int channels = gAudio.ch;
    size_t posFrame = gAudio.pos / (size_t)channels;
    size_t startFrame = (posFrame > (size_t)count) ? posFrame - (size_t)count : 0;
    size_t totalFrames = gAudio.pcm.size() / (size_t)channels;
    for (int i = 0; i < count; i++) {
        size_t f = startFrame + (size_t)i;
        if (f >= totalFrames) { out[i] = 0.0f; continue; }
        float s = 0.0f;
        for (int ch = 0; ch < channels; ch++)
            s += (float)gAudio.pcm[f * (size_t)channels + (size_t)ch];
        out[i] = s / (32768.0f * (float)channels);
    }
}

static std::vector<uint8_t> Fetch(const std::string& url)
{
    bool        tls = false;
    std::string host, path, port;

    if (url.rfind("https://", 0) == 0)      { tls = true; host = url.substr(8); }
    else if (url.rfind("http://", 0) == 0)  { host = url.substr(7); }
    else return {};

    size_t sl = host.find('/');
    path = (sl != std::string::npos) ? host.substr(sl) : "/";
    if (sl != std::string::npos) host = host.substr(0, sl);

    size_t co = host.find(':');
    if (co != std::string::npos) { port = host.substr(co+1); host = host.substr(0, co); }
    else port = tls ? "443" : "80";

    struct addrinfo hints{}, *res = nullptr;
    hints.ai_family   = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    if (getaddrinfo(host.c_str(), port.c_str(), &hints, &res) != 0) return {};

    int sock = -1;
    for (auto* r = res; r; r = r->ai_next) {
        sock = socket(r->ai_family, r->ai_socktype, r->ai_protocol);
        if (sock < 0) continue;
        if (connect(sock, r->ai_addr, r->ai_addrlen) == 0) break;
        close(sock); sock = -1;
    }
    freeaddrinfo(res);
    if (sock < 0) return {};

    std::string req = "GET " + path + " HTTP/1.1\r\nHost: " + host +
                      "\r\nUser-Agent: PandAmp/1.0\r\nAccept-Encoding: identity\r\nConnection: close\r\n\r\n";
    std::vector<uint8_t> raw;

    if (tls) {
        SSL_CTX* ctx = SSL_CTX_new(TLS_client_method());
        SSL_CTX_set_default_verify_paths(ctx);
        SSL* ssl = SSL_new(ctx);
        SSL_set_fd(ssl, sock);
        SSL_set_tlsext_host_name(ssl, host.c_str());
        if (SSL_connect(ssl) <= 0) {
            SSL_free(ssl); close(sock); SSL_CTX_free(ctx); return {};
        }
        SSL_write(ssl, req.c_str(), (int)req.size());
        char b[32768]; int n;
        while ((n = SSL_read(ssl, b, sizeof(b))) > 0)
            raw.insert(raw.end(), b, b+n);
        SSL_shutdown(ssl); SSL_free(ssl); SSL_CTX_free(ctx);
    } else {
        send(sock, req.c_str(), req.size(), 0);
        char b[32768]; int n;
        while ((n = (int)recv(sock, b, sizeof(b), 0)) > 0)
            raw.insert(raw.end(), b, b+n);
    }
    close(sock);

    {
        size_t dumpLen = std::min(raw.size(), (size_t)256);
        fprintf(stderr, "[DEBUG] raw[0..%zu]: ", dumpLen);
        for (size_t i = 0; i < dumpLen; i++) {
            unsigned char c = raw[i];
            if (c >= 32 && c < 127) fputc(c, stderr);
            else fprintf(stderr, "\\x%02x", c);
        }
        fprintf(stderr, "\n");
    }
    std::string header(raw.begin(), raw.begin() + std::min(raw.size(), (size_t)4096));
    int code = 0;
    {
        size_t sp = header.find(' ');
        if (sp != std::string::npos) code = std::atoi(header.c_str() + sp + 1);
    }

    if (code == 301 || code == 302 || code == 303 || code == 307 || code == 308) {
        size_t loc = header.find("Location: ");
        if (loc == std::string::npos) loc = header.find("location: ");
        if (loc != std::string::npos) {
            loc += 10;
            size_t eol = header.find("\r\n", loc);
            std::string newUrl = header.substr(loc, eol - loc);
            return Fetch(newUrl);
        }
    }

    // Strip headers and handel chunks
    const uint8_t sep[] = {'\r','\n','\r','\n'};
    auto it = std::search(raw.begin(), raw.end(), sep, sep+4);
    if (it == raw.end()) return {};
    raw.erase(raw.begin(), it+4);

    if (header.find("Transfer-Encoding: chunked") != std::string::npos ||
        header.find("transfer-encoding: chunked") != std::string::npos) {
        std::vector<uint8_t> out;
        size_t pos = 0;
        while (pos < raw.size()) {
            size_t eol = pos;
            while (eol + 1 < raw.size() && !(raw[eol] == '\r' && raw[eol+1] == '\n')) eol++;
            if (eol + 1 >= raw.size()) break;
            char szBuf[32] = {};
            size_t szLen = std::min(eol - pos, (size_t)16);
            memcpy(szBuf, raw.data() + pos, szLen);
            size_t chunkSz = std::strtoul(szBuf, nullptr, 16);
            if (chunkSz == 0) break;
            pos = eol + 2;
            if (pos + chunkSz > raw.size()) { out.insert(out.end(), raw.begin()+pos, raw.end()); break; }
            out.insert(out.end(), raw.begin()+pos, raw.begin()+pos+chunkSz);
            pos += chunkSz + 2;
        }
        return out;
    }

    return raw;
}

class LoginView : public BView {
public:
    BTextControl* fEmail;
    BTextControl* fPass;
    BButton*      fSignInBtn;
    BCheckBox*    fRemember;
    BStringView*  fError;

    LoginView(BRect r)
        : BView(r, "loginView", B_FOLLOW_ALL, B_WILL_DRAW | B_SUPPORTS_LAYOUT)
    {
        SetViewUIColor(B_PANEL_BACKGROUND_COLOR);

        auto* title = new BStringView("t", "PandAmp");
        BFont f; title->GetFont(&f); f.SetSize(20); f.SetFace(B_BOLD_FACE);
        title->SetFont(&f);
        title->SetAlignment(B_ALIGN_CENTER);

        fEmail = new BTextControl("Email", "", nullptr);
        fPass  = new BTextControl("Password", "", nullptr);
        fPass->TextView()->HideTyping(true);

        fRemember  = new BCheckBox("r", "Remember me", nullptr);
        fRemember->SetValue(B_CONTROL_ON);

        fSignInBtn = new BButton("b", "Sign In", new BMessage(MSG_LOGIN));
        fSignInBtn->MakeDefault(true);

        fError = new BStringView("e", "");
        fError->SetHighUIColor(B_FAILURE_COLOR);
        fError->SetAlignment(B_ALIGN_CENTER);

        BLayoutBuilder::Group<>(this, B_VERTICAL, 6)
            .AddGlue()
            .Add(title)
            .AddStrut(6)
            .Add(fEmail).Add(fPass).Add(fRemember)
            .AddStrut(2)
            .Add(fSignInBtn).Add(fError)
            .AddGlue()
            .SetInsets(50, 0, 50, 0);
    }

    void SetError(const char* s) { fError->SetText(s); }
    void ClearError()            { fError->SetText(""); }
};

class NowPlayingView : public BView {
public:
    BBitmap* fArt    = nullptr;
    BString  fSong, fArtist, fAlbum;
    int      fRating = 0;

    NowPlayingView(BRect r)
        : BView(r, "np", B_FOLLOW_LEFT_RIGHT|B_FOLLOW_TOP, B_WILL_DRAW)
    { SetViewUIColor(B_PANEL_BACKGROUND_COLOR); }

    ~NowPlayingView() override { delete fArt; }

    void SetArt(BBitmap* bm) { delete fArt; fArt = bm; Invalidate(); }

    void SetInfo(const char* song, const char* artist,
                 const char* album, int rating)
    {
        fSong = song; fArtist = artist; fAlbum = album; fRating = rating;
        Invalidate();
    }

    void Clear() { delete fArt; fArt = nullptr;
                   fSong = fArtist = fAlbum = ""; fRating = 0; Invalidate(); }

    void Draw(BRect) override
    {
        BRect b     = Bounds();
        float artSz = b.Height();

        BRect ar(0, 0, artSz-1, artSz-1);
        if (fArt) {
            DrawBitmap(fArt, fArt->Bounds(), ar, B_FILTER_BITMAP_BILINEAR);
        } else {
            SetHighColor(55, 55, 75);
            FillRoundRect(ar.InsetByCopy(2,2), 8, 8);
            SetHighColor(160, 160, 200);
            BFont big; big.SetSize(30); SetFont(&big);
            DrawString("♪", BPoint(artSz/2 - 10, artSz/2 + 10));
        }

        float tx = artSz + 14, ty = 16;
        BFont bold; bold.SetFace(B_BOLD_FACE); bold.SetSize(14); SetFont(&bold);
        SetHighUIColor(B_PANEL_TEXT_COLOR);
        DrawString(fSong.IsEmpty() ? "No track loaded" : fSong.String(), BPoint(tx, ty));

        BFont norm; norm.SetSize(12); SetFont(&norm);
        SetHighColor(130, 130, 150);
        DrawString(fArtist.String(), BPoint(tx, ty+20));
        DrawString(fAlbum.String(),  BPoint(tx, ty+38));

        if      (fRating ==  1) { SetHighColor(50,190,80);  DrawString("▲ Liked",   BPoint(tx,ty+58)); }
        else if (fRating == -1) { SetHighColor(210,60,60);  DrawString("▼ Banned",  BPoint(tx,ty+58)); }
    }
};

// ─────────────────────────────────────────────────────────────────────────────
// MainWindow
// ─────────────────────────────────────────────────────────────────────────────
MainWindow::MainWindow()
    : BWindow(BRect(200, 200, 960, 440), "PandAmp",
               B_TITLED_WINDOW, B_QUIT_ON_WINDOW_CLOSE)
    , fLoginView(nullptr), fPlayerView(nullptr), fNowPlaying(nullptr)
    , fStationList(nullptr), fStatusBar(nullptr)
    , fPlayPauseBtn(nullptr), fNextBtn(nullptr)
    , fThumbUpBtn(nullptr), fThumbDownBtn(nullptr), fTiredBtn(nullptr)
    , fProgress(nullptr), fVolume(nullptr), fTimeLabel(nullptr)
    , fSoundPlayer(nullptr)
    , fDecodeRunning(false), fPlaying(false), fPaused(false), fElapsed(0)
    , fTickRunner(nullptr)
    , fSkinWindow(nullptr), fPlaylistWindow(nullptr), fEqWindow(nullptr), fSkinPanel(nullptr)
{
    BuildUI();

    // Auto-login from saved credentials
    std::string email, pass;
    if (fAPI.LoadCredentials(email, pass)) {
        fLoginView->fEmail->SetText(email.c_str());
        fLoginView->fPass->SetText(pass.c_str());
        AsyncLogin(email, pass);
    }

    CenterOnScreen();
    Show();
}

MainWindow::~MainWindow()
{
    fDecodeRunning = false;
    fPlaying       = false;
    if (fDecodeThread.joinable()) fDecodeThread.join();
    delete fTickRunner;
    if (fSoundPlayer) { fSoundPlayer->Stop(); delete fSoundPlayer; }
}

bool MainWindow::QuitRequested()
{
    fDecodeRunning = false;
    fPlaying       = false;
    if (fSoundPlayer) fSoundPlayer->Stop();
    be_app->PostMessage(B_QUIT_REQUESTED);
    return true;
}

// ─── BuildUI ─────────────────────────────────────────────────────────────────
void MainWindow::BuildUI()
{
    float W = Bounds().Width(), H = Bounds().Height();

    // ── Menu bar ─────────────────────────────────────────────────────────────
    BMenuBar* menuBar = new BMenuBar(BRect(0,0,W,20), "menu");
    BMenu* skinMenu = new BMenu("Skin");
    skinMenu->AddItem(new BMenuItem("Load Skin (.wsz)…",
                                    new BMessage(MSG_LOAD_SKIN), 'K'));
    skinMenu->AddItem(new BMenuItem("Exit Skin Mode",
                                    new BMessage(MSG_CLEAR_SKIN)));
    skinMenu->AddSeparatorItem();
    BMenuItem* dblItem = new BMenuItem("Double Size (2×)",
                                        new BMessage(MSG_SKIN_DOUBLE_SZ));
    dblItem->SetMarked(fSkinScale == 2);
    skinMenu->AddItem(dblItem);
    BMenuItem* smItem = new BMenuItem("Smooth Scaling (bilinear)",
                                       new BMessage(MSG_SKIN_SMOOTH));
    smItem->SetMarked(fSkinSmooth);
    skinMenu->AddItem(smItem);
    menuBar->AddItem(skinMenu);
    AddChild(menuBar);
    SetKeyMenuBar(menuBar);
    float menuH = menuBar->Bounds().Height() + 1;

    BView* root = new BView(BRect(0, menuH, W, H), "root", B_FOLLOW_ALL, B_WILL_DRAW);
    root->SetViewUIColor(B_PANEL_BACKGROUND_COLOR);
    AddChild(root);

    float rootH = H - menuH;

    // Login overlay
    fLoginView = new LoginView(BRect(0, 0, W, rootH));
    root->AddChild(fLoginView);

    // Player view (hidden until logged in)
    fPlayerView = new BView(BRect(0, 0, W, rootH), "player", B_FOLLOW_ALL, B_WILL_DRAW);
    fPlayerView->SetViewUIColor(B_PANEL_BACKGROUND_COLOR);
    fPlayerView->Hide();
    root->AddChild(fPlayerView);

    const float kSideW = 224;

    // ── Left: station list ───────────────────────────────────────────────────
    // Height matches right panel content (NP=88 + progress=14+10 + buttons=26+8 + vol=16+8 + pad)
    const float kPanelH = 192;
    BBox* sbox = new BBox(BRect(0, 0, kSideW, kPanelH), "sb",
                          B_FOLLOW_LEFT|B_FOLLOW_TOP);
    sbox->SetLabel("My Stations");
    fPlayerView->AddChild(sbox);

    // Leave 28px at bottom for "+ Add Station" button
    fStationList = new BListView(BRect(6, 18, kSideW-18, kPanelH-36),
                                 "sl", B_SINGLE_SELECTION_LIST, B_FOLLOW_ALL);
    fStationList->SetSelectionMessage(new BMessage(MSG_STATION_SELECT));
    sbox->AddChild(new BScrollView("ss", fStationList, B_FOLLOW_ALL, 0, false, true));

    BButton* addStBtn = new BButton(BRect(6, kPanelH-30, kSideW-12, kPanelH-10),
                                    "addst", "+ Add Station",
                                    new BMessage(MSG_ADD_STATION));
    sbox->AddChild(addStBtn);

    // ── Right: player ────────────────────────────────────────────────────────
    float rx = kSideW + 4;
    float rw = W - rx;
    BView* rp = new BView(BRect(rx, 0, W, kPanelH), "rp", B_FOLLOW_LEFT_RIGHT|B_FOLLOW_TOP, B_WILL_DRAW);
    rp->SetViewUIColor(B_PANEL_BACKGROUND_COLOR);
    fPlayerView->AddChild(rp);

    // Now-playing strip (88px tall)
    const float kNpH = 88;
    fNowPlaying = new NowPlayingView(BRect(0, 4, rw, kNpH));
    rp->AddChild(fNowPlaying);

    // Progress
    float py = kNpH + 10;
    fProgress = new BSlider(BRect(0, py, rw-84, py+14), "pg",
                             nullptr, nullptr, 0, 1000, B_HORIZONTAL);
    fProgress->SetEnabled(false);
    rp->AddChild(fProgress);
    fTimeLabel = new BStringView(BRect(rw-80, py, rw, py+14), "tl", "--:-- / --:--");
    rp->AddChild(fTimeLabel);

    // Buttons
    float by = py + 22;
    float bx = 0;
    auto btn = [&](const char* lbl, uint32_t msg_) -> BButton* {
        auto* b = new BButton(BRect(bx, by, bx+80, by+26), "", lbl, new BMessage(msg_));
        rp->AddChild(b); bx += 86; return b;
    };
    fPlayPauseBtn = btn("▶  Play",    MSG_PLAY_PAUSE);
    fNextBtn      = btn("⏭  Next",    MSG_NEXT);
    fThumbUpBtn   = btn("▲  Like",    MSG_THUMB_UP);
    fThumbDownBtn = btn("▼  Dislike", MSG_THUMB_DOWN);
    fTiredBtn     = btn("😴  Tired",  MSG_TIRED);

    // Sign out (right-aligned)
    rp->AddChild(new BButton(BRect(rw-88, by, rw, by+26), "so",
                              "Sign Out", new BMessage(MSG_LOGOUT)));

    // Volume
    float vy = by + 34;
    rp->AddChild(new BStringView(BRect(0, vy, 58, vy+16), "vl", "Volume:"));
    fVolume = new BSlider(BRect(60, vy, rw, vy+16), "vol",
                          nullptr, new BMessage(MSG_VOLUME), 0, 100, B_HORIZONTAL);
    fVolume->SetValue(80);
    rp->AddChild(fVolume);

    // Status bar
    fStatusBar = new BStringView(BRect(0, kPanelH+4, W, kPanelH+20), "sb", "Ready");
    fStatusBar->SetFont(be_fixed_font);
    fPlayerView->AddChild(fStatusBar);
}

void MainWindow::ShowLogin() {
    ResizeTo(420 - 1, 230 - 1);
    fLoginView->Show();
    fPlayerView->Hide();
}
void MainWindow::ShowPlayer() {
    ResizeTo(760 - 1, 240 - 1);
    fLoginView->Hide();
    fPlayerView->Show();
}

void MainWindow::SetStatus(const char* s)
{
    if (Lock()) { if (fStatusBar) fStatusBar->SetText(s); Unlock(); }
}

void MainWindow::MessageReceived(BMessage* msg)
{
    switch (msg->what) {

    case MSG_LOGIN: {
        const char* em = fLoginView->fEmail->Text();
        const char* pw = fLoginView->fPass->Text();
        if (!em || em[0]==0 || !pw || pw[0]==0) {
            fLoginView->SetError("Email and password are required.");
            break;
        }
        fLoginView->fSignInBtn->SetEnabled(false);
        fLoginView->ClearError();
        SetStatus("Signing in…");
        AsyncLogin(em, pw);
        break;
    }

    case MSG_AUTH_DONE: {
        bool ok = false; BString err;
        msg->FindBool("ok", &ok);
        msg->FindString("err", &err);
        if (ok) {
            ShowPlayer();
            SetStatus("Loading stations…");
            AsyncGetStations();
        } else {
            fLoginView->SetError(err.String());
            fLoginView->fSignInBtn->SetEnabled(true);
        }
        break;
    }

    case MSG_LOGOUT:
        StopPlayback();
        fAPI.Logout();
        fStationList->MakeEmpty();
        fNowPlaying->Clear();
        fCurrentStation.clear();
        { std::lock_guard<std::mutex> lk(fQueueMtx); fQueue.clear(); }
        ShowLogin();
        SetStatus("Signed out.");
        break;

    case MSG_STATIONS_LOADED: {
        std::vector<PandoraStation> st;
        { std::lock_guard<std::mutex> lk(fQueueMtx); st = fStations; }
        fStationList->MakeEmpty();
        for (auto& s : st) {
            BString n; if (s.isQuickMix) n = "⚡ ";
            n << s.stationName.c_str();
            fStationList->AddItem(new BStringItem(n.String()));
        }
        // Also update skin playlist window if open
        if (fPlaylistWindow && fPlaylistWindow->fList()) {
            if (fPlaylistWindow->LockLooper()) {
                fPlaylistWindow->fList()->MakeEmpty();
                for (auto& s : st) {
                    BString n; if (s.isQuickMix) n = "⚡ ";
                    n << s.stationName.c_str();
                    fPlaylistWindow->fList()->AddItem(n.String());
                }
                fPlaylistWindow->UnlockLooper();
            }
        }
        SetStatus("Select a station.");
        break;
    }

    case MSG_STATION_SELECT: {
        int32 idx = -1;
        if (msg->FindInt32("index", &idx) != B_OK)
            idx = fStationList->CurrentSelection();
        std::string tok;
        { std::lock_guard<std::mutex> lk(fQueueMtx);
          if (idx >= 0 && idx < (int32)fStations.size()) tok = fStations[idx].stationToken; }
        if (!tok.empty()) {
            StopPlayback();
            { std::lock_guard<std::mutex> lk(fQueueMtx); fQueue.clear(); }
            fCurrentStation = tok;
            fNowPlaying->Clear();
            SetStatus("Fetching playlist…");
            AsyncGetPlaylist(tok);
            if (fPlaylistWindow && fPlaylistWindow->fList() && idx >= 0) {
                if (fPlaylistWindow->LockLooper()) {
                    fPlaylistWindow->fList()->SetCurrent(idx);
                    fPlaylistWindow->UnlockLooper();
                }
            }
        }
        break;
    }

    case MSG_PLAYLIST_LOADED:
        if (!fPlaying.load()) StartNextTrack();
        break;

    case MSG_DECODE_DONE:
        BeginPlayback();
        break;

    case MSG_PLAY_PAUSE:
        if (!fPlaying.load()) break;
        if (fPaused.load()) {
            fPaused = false;
            if (fSoundPlayer) fSoundPlayer->Start();
            fPlayPauseBtn->SetLabel("⏸  Pause");
            if (!fTickRunner) {
                BMessage t(MSG_TICK);
                fTickRunner = new BMessageRunner(BMessenger(this), &t, 1000000LL);
            }
        } else {
            fPaused = true;
            if (fSoundPlayer) fSoundPlayer->Stop();
            fPlayPauseBtn->SetLabel("▶  Play");
            delete fTickRunner; fTickRunner = nullptr;
        }
        break;

    case MSG_NEXT:
        if (!fCurrent.allowSkip) { SetStatus("Skip limit reached."); break; }
        StopPlayback();
        { std::lock_guard<std::mutex> lk(fQueueMtx);
          if (!fQueue.empty()) fQueue.erase(fQueue.begin()); }
        StartNextTrack();
        break;

    case MSG_THUMB_UP:
        if (fCurrent.trackToken.empty()) break;
        fCurrent.rating = 1; fNowPlaying->fRating = 1; fNowPlaying->Invalidate();
        AsyncFeedback(fCurrent.trackToken, true, false);
        break;

    case MSG_THUMB_DOWN:
        if (fCurrent.trackToken.empty()) break;
        AsyncFeedback(fCurrent.trackToken, false, false);
        StopPlayback();
        { std::lock_guard<std::mutex> lk(fQueueMtx);
          if (!fQueue.empty()) fQueue.erase(fQueue.begin()); }
        StartNextTrack();
        break;

    case MSG_TIRED:
        if (fCurrent.trackToken.empty()) break;
        AsyncFeedback(fCurrent.trackToken, false, true);
        StopPlayback();
        { std::lock_guard<std::mutex> lk(fQueueMtx);
          if (!fQueue.empty()) fQueue.erase(fQueue.begin()); }
        StartNextTrack();
        break;

    case MSG_VOLUME: {
        // Can come from native BSlider (no "vol" field) or from skin (has "vol" field)
        int32 vol = -1;
        if (msg->FindInt32("vol", &vol) == B_OK) {
            // From skin — update native slider and sound player
            if (fVolume) fVolume->SetValue(vol);
            if (fSoundPlayer) fSoundPlayer->SetVolume(vol / 100.0f);
        } else {
            // From native slider
            if (fSoundPlayer && fVolume)
                fSoundPlayer->SetVolume(fVolume->Value() / 100.0f);
        }
        break;
    }

    case 'Blnc': {
        int32 bal = 0;
        if (msg->FindInt32("bal", &bal) == B_OK)
            gBalance.store((int)bal);
        break;
    }

    case 'EqTg':  // toggle EQ window
        if (fEqWindow) {
            bool nowHidden;
            if (fEqWindow->IsHidden()) { fEqWindow->Show(); nowHidden = false; }
            else                       { fEqWindow->Hide(); nowHidden = true;  }
            fSkinWindow->SetEqVisible(!nowHidden);
        }
        break;

    case 'SkTg':  // toggle playlist window
        if (fPlaylistWindow) {
            bool nowHidden;
            if (fPlaylistWindow->IsHidden()) { fPlaylistWindow->Show(); nowHidden = false; }
            else                             { fPlaylistWindow->Hide(); nowHidden = true;  }
            if (fSkinWindow) fSkinWindow->SetPlVisible(!nowHidden);
        }
        break;

    case 'Prev':  // restart current track
        fElapsed = 0;
        {
            std::lock_guard<std::mutex> lk(gAudio.mtx);
            gAudio.pos  = 0;
            gAudio.done = false;
        }
        SyncSkinState();
        break;

    case 'Seek': { // seek to sample position
        int32 sec = 0;
        msg->FindInt32("sec", &sec);
        fElapsed = sec;
        {
            std::lock_guard<std::mutex> lk(gAudio.mtx);
            size_t target = (size_t)sec * (size_t)gAudio.hz * (size_t)gAudio.ch;
            if (gAudio.ch > 0) target -= target % (size_t)gAudio.ch;
            gAudio.pos  = std::min(target, gAudio.pcm.size());
            gAudio.done = false;
        }
        SyncSkinState();
        break;
    }

    case MSG_TICK: {
        if (!fPlaying.load() || fPaused.load()) break;
        ++fElapsed;
        int el = fElapsed.load(), dur = fCurrent.durationSecs;
        if (dur > 0) {
            fProgress->SetValue((int32)(el * 1000.0 / dur));
            char buf[32];
            snprintf(buf, sizeof(buf), "%d:%02d / %d:%02d", el/60, el%60, dur/60, dur%60);
            fTimeLabel->SetText(buf);
        }
        // Update skin view if active
        if (fSkinWindow && fSkin.IsLoaded()) {
            SyncSkinState();
        }
        // Prefetch
        if (dur > 0 && dur - el <= 30 && !fCurrentStation.empty()) {
            bool need; { std::lock_guard<std::mutex> lk(fQueueMtx); need = fQueue.size() <= 1; }
            if (need) AsyncGetPlaylist(fCurrentStation);
        }
        // Track finished
        if (gAudio.done.load()) {
            delete fTickRunner; fTickRunner = nullptr; fPlaying = false;
            { std::lock_guard<std::mutex> lk(fQueueMtx);
              if (!fQueue.empty()) fQueue.erase(fQueue.begin()); }
            StartNextTrack();
        }
        break;
    }

    case MSG_ART_LOADED: {
        BBitmap* bm = nullptr;
        msg->FindPointer("bm", (void**)&bm);
        fNowPlaying->SetArt(bm);
        break;
    }

    case MSG_ERROR: {
        BString e; msg->FindString("e", &e);
        SetStatus(e.String());
        break;
    }

    case MSG_LOAD_SKIN: {
        // If this came from the file panel, it has a ref
        entry_ref ref;
        if (msg->FindRef("refs", &ref) == B_OK) {
            BEntry entry(&ref);
            BPath path;
            entry.GetPath(&path);
            LoadSkin(path.Path());
        } else {
            // Open file panel
            if (!fSkinPanel) {
                BMessenger target(this);
                BMessage openMsg(MSG_LOAD_SKIN);
                fSkinPanel = new BFilePanel(B_OPEN_PANEL, &target,
                                            nullptr, B_FILE_NODE, false,
                                            &openMsg);
            }
            fSkinPanel->Show();
        }
        break;
    }

    case MSG_SKIN_DOUBLE_SZ: {
        fSkinScale = (fSkinScale == 1) ? 2 : 1;
        BMenuBar* mb = KeyMenuBar();
        if (mb) {
            BMenuItem* it = mb->FindItem(MSG_SKIN_DOUBLE_SZ);
            if (it) it->SetMarked(fSkinScale == 2);
        }
        if (fSkin.IsLoaded()) {
            fSkin.SetScale(fSkinScale, fSkinSmooth);
            if (fSkinWindow) {
                fSkinWindow->ReloadSkin(fSkinScale);
                BPoint playerPos = fSkinWindow->Frame().LeftTop();
                if (fPlaylistWindow) {
                    BPoint plPos = playerPos;
                    plPos.y += WA::kMainH * fSkinScale + 2;
                    fPlaylistWindow->MoveTo(plPos);
                    fPlaylistWindow->ReloadSkin(fSkinScale);
                }
                if (fEqWindow) {
                    BPoint eqPos = playerPos;
                    eqPos.y += (WA::kMainH + WA::kPlH) * fSkinScale + 4;
                    fEqWindow->MoveTo(eqPos);
                    fEqWindow->ReloadSkin(&fSkin, fSkinScale);
                }
            }
            SyncSkinState();
        }
        break;
    }

    case MSG_SKIN_SMOOTH: {
        fSkinSmooth = !fSkinSmooth;
        BMenuBar* mb = KeyMenuBar();
        if (mb) {
            BMenuItem* it = mb->FindItem(MSG_SKIN_SMOOTH);
            if (it) it->SetMarked(fSkinSmooth);
        }
        if (fSkin.IsLoaded()) {
            fSkin.SetScale(fSkinScale, fSkinSmooth);
            if (fSkinWindow)     fSkinWindow->ReloadSkin(fSkinScale);
            if (fPlaylistWindow) fPlaylistWindow->ReloadSkin(fSkinScale);
            if (fEqWindow)       fEqWindow->ReloadSkin(&fSkin, fSkinScale);
            SyncSkinState();
        }
        break;
    }

    case MSG_ADD_STATION: {
        StationSearchWindow* w = new StationSearchWindow(BMessenger(this), &fAPI);
        w->Show();
        break;
    }

    case MSG_STATION_CREATED: {
        const char* name  = nullptr;
        const char* token = nullptr;
        msg->FindString("name",  &name);
        msg->FindString("token", &token);
        if (name && token) {
            BString info("Station \"");
            info << name << "\" added! Refreshing…";
            SetStatus(info.String());
        }
        // Reload full station list from server so the new one appears
        AsyncGetStations();
        break;
    }

    case MSG_CLEAR_SKIN:
        ClearSkin();
        break;

    default:
        BWindow::MessageReceived(msg);
    }
}

void MainWindow::AsyncLogin(const std::string& email, const std::string& pass)
{
    bool remember = fLoginView->fRemember->Value() == B_CONTROL_ON;
    std::thread([this, email, pass, remember]() {
        fAPI.Login(email, pass, [this, email, pass, remember](bool ok, const std::string& err) {
            if (ok && remember) fAPI.SaveCredentials(email, pass);
            BMessage m(MSG_AUTH_DONE);
            m.AddBool("ok", ok);
            m.AddString("err", err.c_str());
            PostMessage(&m);
        });
    }).detach();
}

void MainWindow::AsyncGetStations()
{
    std::thread([this]() {
        fAPI.GetStations([this](bool ok, const std::vector<PandoraStation>& s,
                                const std::string& err) {
            if (ok) {
                { std::lock_guard<std::mutex> lk(fQueueMtx); fStations = s; }
                PostMessage(MSG_STATIONS_LOADED);
            } else {
                BMessage m(MSG_ERROR);
                m.AddString("e", (std::string("Stations: ")+err).c_str());
                PostMessage(&m);
            }
        });
    }).detach();
}

void MainWindow::AsyncGetPlaylist(const std::string& tok)
{
    std::thread([this, tok]() {
        fAPI.GetPlaylist(tok, [this](bool ok, const std::vector<PandoraTrack>& tracks,
                                     const std::string& err) {
            if (ok) {
                {
                    std::lock_guard<std::mutex> lk(fQueueMtx);
                    for (auto& t : tracks) {
                        TrackInfo ti;
                        ti.songName     = t.songName;
                        ti.artistName   = t.artistName;
                        ti.albumName    = t.albumName;
                        ti.albumArtUrl  = t.albumArtUrl;
                        ti.audioUrl     = t.audioUrl;
                        ti.trackToken   = t.trackToken;
                        ti.rating       = t.rating;
                        ti.durationSecs = t.durationSecs;
                        ti.allowSkip    = t.allowSkip;
                        fprintf(stderr, "[DEBUG] queued track: '%s' audioUrl='%s'\n",
                                ti.songName.c_str(), ti.audioUrl.c_str());
                        fQueue.push_back(ti);
                    }
                }
                PostMessage(MSG_PLAYLIST_LOADED);
            } else {
                BMessage m(MSG_ERROR);
                m.AddString("e", (std::string("Playlist: ")+err).c_str());
                PostMessage(&m);
            }
        });
    }).detach();
}

void MainWindow::AsyncFeedback(const std::string& tok, bool up, bool tired)
{
    std::thread([this, tok, up, tired]() {
        auto cb = [](bool, const std::string&){};
        if      (tired) fAPI.TiredOfSong(tok, cb);
        else if (up)    fAPI.ThumbsUp(tok, cb);
        else            fAPI.ThumbsDown(tok, cb);
    }).detach();
}

void MainWindow::AsyncLoadArt(const std::string& url)
{
    if (url.empty()) return;
    std::thread([this, url]() {
        auto data = Fetch(url);
        BBitmap* bm = nullptr;
        if (!data.empty()) {
            BPath tmp;
            find_directory(B_SYSTEM_TEMP_DIRECTORY, &tmp);
            tmp.Append("pandamp_art.jpg");
            BFile f(tmp.Path(), B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE);
            if (f.InitCheck() == B_OK) {
                f.Write(data.data(), data.size());
                f.Unset();
                bm = BTranslationUtils::GetBitmap(tmp.Path());
            }
        }
        BMessage m(MSG_ART_LOADED);
        m.AddPointer("bm", bm);
        PostMessage(&m);
    }).detach();
}

void MainWindow::StartNextTrack()
{
    TrackInfo t;
    {
        std::lock_guard<std::mutex> lk(fQueueMtx);
        if (fQueue.empty()) {
            SetStatus("Buffering…");
            if (!fCurrentStation.empty()) AsyncGetPlaylist(fCurrentStation);
            return;
        }
        t = fQueue.front();
        fQueue.erase(fQueue.begin());  // consume the track
    }

    fCurrent = t;
    fElapsed = 0;
    fProgress->SetValue(0);
    fTimeLabel->SetText("0:00 / …");
    fNowPlaying->SetInfo(t.songName.c_str(), t.artistName.c_str(),
                         t.albumName.c_str(), t.rating);
    AsyncLoadArt(t.albumArtUrl);
    SetStatus((std::string("Loading: ") + t.songName + " — " + t.artistName).c_str());

    // Launch decode thread
    fDecodeRunning = false;
    if (fDecodeThread.joinable()) fDecodeThread.join();
    fDecodeRunning = true;
    std::string audioUrl = t.audioUrl;

    fDecodeThread = std::thread([this, audioUrl]() {
        fprintf(stderr, "[DEBUG] audioUrl: '%s'\n", audioUrl.c_str());
        if (audioUrl.empty()) {
            fprintf(stderr, "[DEBUG] audioUrl is EMPTY - skipping track\n");
            PostMessage(MSG_NEXT);
            return;
        }
        auto mp3 = Fetch(audioUrl);
        fprintf(stderr, "[DEBUG] Fetch returned %zu bytes\n", mp3.size());
        // Dump first 32 bytes as hex
        fprintf(stderr, "[DEBUG] mp3[0..32]: ");
        for (size_t i = 0; i < std::min(mp3.size(), (size_t)32); i++)
            fprintf(stderr, "%02x ", mp3[i]);
        fprintf(stderr, "\n");
        if (mp3.empty() || !fDecodeRunning.load()) return;

        mp3dec_t dec; mp3dec_init(&dec);
        std::vector<int16_t> pcm;
        int sr = 44100, ch = 2;
        size_t off = 0;

        // Skip ID3v2 tag if present: "ID3" + 2 bytes version + 1 byte flags + 4 bytes syncsafe size
        if (mp3.size() > 10 &&
            mp3[0] == 'I' && mp3[1] == 'D' && mp3[2] == '3') {
            uint32_t id3Size = ((uint32_t)(mp3[6] & 0x7f) << 21) |
                               ((uint32_t)(mp3[7] & 0x7f) << 14) |
                               ((uint32_t)(mp3[8] & 0x7f) <<  7) |
                               ((uint32_t)(mp3[9] & 0x7f));
            // Check for footer flag (bit 4 of flags)
            if (mp3[5] & 0x10) id3Size += 10;
            size_t newOff = 10 + id3Size;
            if (newOff < mp3.size()) {
                off = newOff;
                fprintf(stderr, "[DEBUG] Skipped ID3 tag, off=%zu\n", off);
            } else {
                fprintf(stderr, "[DEBUG] ID3 size %u looks bogus (mp3.size=%zu), scanning for sync\n",
                        id3Size, mp3.size());
            }
        }

        // Scan forward to find the first MPEG sync word (0xFF 0xE0..0xFF)
        // in case there's ID3 or other junk we haven't fully skipped
        {
            size_t scanStart = off;
            for (size_t i = scanStart; i + 1 < mp3.size(); i++) {
                if (mp3[i] == 0xff && (mp3[i+1] & 0xe0) == 0xe0) {
                    if (i != off) fprintf(stderr, "[DEBUG] sync found at %zu (skipped %zu bytes)\n", i, i - off);
                    off = i;
                    break;
                }
            }
        }

        int zeroFrames = 0;
        while (off < mp3.size() && fDecodeRunning.load()) {
            mp3dec_frame_info_t info;
            int16_t frame[MINIMP3_MAX_SAMPLES_PER_FRAME];
            int s = mp3dec_decode_frame(&dec,
                        mp3.data()+off, (int)(mp3.size()-off), frame, &info);
            if (info.frame_bytes == 0) {
                // Can't advance — try skipping 1 byte to find next sync
                off++;
                if (++zeroFrames > 1000) break; // give up
                continue;
            }
            zeroFrames = 0;
            off += info.frame_bytes;
            if (s > 0) {
                sr = info.hz; ch = info.channels;
                pcm.insert(pcm.end(), frame, frame + s*ch);
            }
        }

        fprintf(stderr, "[DEBUG] Decoded %zu PCM samples at %d Hz, %d ch\n", pcm.size(), sr, ch);
        if (!fDecodeRunning.load()) return;
        if (pcm.empty()) {
            fprintf(stderr, "[DEBUG] PCM empty after decode — skipping track\n");
            PostMessage(MSG_NEXT);
            return;
        }

        {
            std::lock_guard<std::mutex> lk(gAudio.mtx);
            gAudio.pcm  = std::move(pcm);
            gAudio.pos  = 0;
            gAudio.done = false;
            gAudio.hz   = sr;
            gAudio.ch   = ch;
        }
        gAudio.ready = true;
        PostMessage(MSG_DECODE_DONE);
    });
}

void MainWindow::BeginPlayback()
{
    if (fSoundPlayer) { fSoundPlayer->Stop(); delete fSoundPlayer; fSoundPlayer = nullptr; }

    media_raw_audio_format fmt;
    fmt.frame_rate    = (float)gAudio.hz;
    fmt.channel_count = (uint32)gAudio.ch;
    fmt.format        = media_raw_audio_format::B_AUDIO_SHORT;
    fmt.byte_order    = B_MEDIA_HOST_ENDIAN;
    fmt.buffer_size   = 4096;

    fSoundPlayer = new BSoundPlayer(&fmt, "PandAmp", FillAudio);
    fSoundPlayer->SetVolume(fVolume->Value() / 100.0f);
    fSoundPlayer->Start();
    fPlaying = true; fPaused = false;
    fPlayPauseBtn->SetLabel("⏸  Pause");

    delete fTickRunner;
    BMessage t(MSG_TICK);
    fTickRunner = new BMessageRunner(BMessenger(this), &t, 1000000LL);

    std::string s = "▶  " + fCurrent.songName + "  —  " + fCurrent.artistName;
    SetStatus(s.c_str());
    SetTitle((std::string("PandAmp — ") + fCurrent.songName).c_str());

    // Sync skin window
    if (fSkinWindow && fSkin.IsLoaded()) SyncSkinState();
}

void MainWindow::StopPlayback()
{
    fPlaying = false; fPaused = false;
    delete fTickRunner; fTickRunner = nullptr;
    if (fSoundPlayer) { fSoundPlayer->Stop(); delete fSoundPlayer; fSoundPlayer = nullptr; }
    gAudio.ready = false; gAudio.done = false;
    fPlayPauseBtn->SetLabel("▶  Play");
    fProgress->SetValue(0);
    fTimeLabel->SetText("--:-- / --:--");
}

// ─── Skin support ─────────────────────────────────────────────────────────────

void MainWindow::LoadSkin(const char* path)
{
    if (!fSkin.Load(path)) {
        BAlert* a = new BAlert("Skin Error",
            "Could not load skin. Make sure it's a valid .wsz file.",
            "OK");
        a->Go();
        return;
    }
    UpdateSkinView();
    SetStatus((std::string("Skin loaded: ") + path).c_str());
    Hide();
}

void MainWindow::ClearSkin()
{
    // Detach dock before hiding so windows don't reference stale dock state
    if (fSkinWindow)     { fSkinWindow->SetDock(nullptr);     fSkinWindow->Hide(); }
    if (fPlaylistWindow) { fPlaylistWindow->SetDock(nullptr);  fPlaylistWindow->Hide(); }
    if (fEqWindow)       { fEqWindow->SetDock(nullptr);        fEqWindow->Hide(); }
    fDock = SkinDock();  // reset dock
    fSkin.Unload();

    // Restore native widgets
    if (fNowPlaying)   fNowPlaying->Show();
    if (fProgress)     fProgress->Show();
    if (fTimeLabel)    fTimeLabel->Show();
    if (fPlayPauseBtn) fPlayPauseBtn->Show();
    if (fNextBtn)      fNextBtn->Show();
    if (fThumbUpBtn)   fThumbUpBtn->Show();
    if (fThumbDownBtn) fThumbDownBtn->Show();
    if (fTiredBtn)     fTiredBtn->Show();
    if (fVolume)       fVolume->Show();

    SetStatus("Skin cleared — using default UI");
    Show();  // native player window comes back
}

void MainWindow::UpdateSkinView()
{
    if (!fSkin.IsLoaded()) { ClearSkin(); return; }

    fSkin.SetScale(fSkinScale, fSkinSmooth);

    // Hide native player widgets
    if (fNowPlaying)   fNowPlaying->Hide();
    if (fProgress)     fProgress->Hide();
    if (fTimeLabel)    fTimeLabel->Hide();
    if (fPlayPauseBtn) fPlayPauseBtn->Hide();
    if (fNextBtn)      fNextBtn->Hide();
    if (fThumbUpBtn)   fThumbUpBtn->Hide();
    if (fThumbDownBtn) fThumbDownBtn->Hide();
    if (fTiredBtn)     fTiredBtn->Hide();
    if (fVolume)       fVolume->Hide();

    // Place skin windows at the left side of screen, stacked vertically
    // (qmmp default: player at ~100,100; we use screen left with a small margin)
    BScreen _scr;
    BPoint origin(_scr.Frame().left + 20.0f, _scr.Frame().top + 60.0f);

    BMessenger me(this);

    if (!fSkinWindow) {
        fSkinWindow = new SkinWindow(origin, &fSkin, me, fSkinScale);
        fSkinWindow->Show();
    } else {
        if (fSkinWindow->IsHidden()) fSkinWindow->Show();
    }

    BPoint plOrigin = origin;
    plOrigin.y += WA::kMainH * fSkinScale + 2;

    if (!fPlaylistWindow) {
        fPlaylistWindow = new PlaylistWindow(plOrigin, &fSkin, me, fSkinScale);
        if (fPlaylistWindow->LockLooper()) {
            fPlaylistWindow->fList()->MakeEmpty();
            std::vector<PandoraStation> stations;
            { std::lock_guard<std::mutex> lk(fQueueMtx); stations = fStations; }
            for (auto& st : stations)
                fPlaylistWindow->fList()->AddItem(st.stationName.c_str());
            fPlaylistWindow->UnlockLooper();
        }
        fPlaylistWindow->Show();
    } else {
        fPlaylistWindow->MoveTo(plOrigin);
        fPlaylistWindow->ReloadSkin(fSkinScale);
        if (fPlaylistWindow->IsHidden()) fPlaylistWindow->Show();
    }

    BPoint eqOrigin = origin;
    eqOrigin.y += (WA::kMainH + WA::kPlH) * fSkinScale + 4;

    if (!fEqWindow) {
        fEqWindow = new EqWindow(eqOrigin, &fSkin, me, fSkinScale);
        fEqWindow->Show();
    } else {
        fEqWindow->MoveTo(eqOrigin);
        fEqWindow->ReloadSkin(&fSkin, fSkinScale);
        if (fEqWindow->IsHidden()) fEqWindow->Show();
    }

    {
        fDock = SkinDock();  // reset
        fDock.AddWindow(fSkinWindow);
        if (fPlaylistWindow) fDock.AddWindow(fPlaylistWindow);
        if (fEqWindow)       fDock.AddWindow(fEqWindow);
        fDock.UpdateDock();  // compute initial docked state
        if (fSkinWindow)     fSkinWindow->SetDock(&fDock);
        if (fPlaylistWindow) fPlaylistWindow->SetDock(&fDock);
        if (fEqWindow)       fEqWindow->SetDock(&fDock);
    }

    SyncSkinState();
}

void MainWindow::SyncSkinState()
{
    if (!fSkinWindow || !fSkinWindow->fView) return;
    SkinMainView* v = fSkinWindow->fView;
    if (fSkinWindow->LockLooper()) {
        v->SetTrackInfo(fCurrent.songName.c_str(),
                        fCurrent.artistName.c_str());
        v->SetPosition(fElapsed.load(), fCurrent.durationSecs);
        v->SetPlaying(fPlaying.load(), fPaused.load());
        v->SetVolume(fVolume ? fVolume->Value() : 80);
        fSkinWindow->SetPlVisible(fPlaylistWindow && !fPlaylistWindow->IsHidden());
        fSkinWindow->SetEqVisible(fEqWindow      && !fEqWindow->IsHidden());
        fSkinWindow->UnlockLooper();
    }
}
