// wsnote.cpp
// e.moon 18dec98
//
// displays current workspace number in small, draggable
// window

#define DEBUG 1
#include <Debug.h>
#include <InterfaceKit.h>
#include <Application.h>
#include <stdio.h>
#include <stdlib.h>

#include <string.h> // Added 1-24-99 by Tyler Riti fizzboy@mail.utexas.edu

#include "wsnote.h"

const char* g_appSignature = "application/x-vnd.emoon-wsnote-1.1";

// -------------------------------------------------- //
// view impl
// -------------------------------------------------- //

const rgb_color View::s_bgColor = {0,0,128,255};
const rgb_color View::s_bgColorHilight = {64,64,192,255};
const rgb_color View::s_textColor = {255,255,255,255};
const rgb_color View::s_borderColor = {255,255,255,255};
const rgb_color View::s_hilightColor = {255,0,0,255};

const uint32 View::s_nBorderSize = 4;
const uint32 View::s_nCornerSize = 5;

const uint32 View::s_nScreenSnapThreshold = 6;

const uint32 View::s_nMinWidth = 16;
const uint32 View::s_nMinHeight = 16;

// -------------------------------------------------- //

View::View(BRect frame, bool bSnap) :
	BView(frame, "wsnote-view", B_FOLLOW_ALL_SIDES,
		B_WILL_DRAW|B_FRAME_EVENTS|B_SUBPIXEL_PRECISE),
	m_font(be_bold_font),
	m_drag(DRAG_NONE),
	m_dragHilight(DRAG_NONE),
	m_bDragInDeskbar(false),
	m_prevBounds(Bounds()),
	m_bSnapEdge(bSnap) {

	// init colors
	SetViewColor(B_TRANSPARENT_COLOR);
	SetLowColor(s_bgColor);

	// init font settings
	m_font.GetHeight(&m_fontHeight);
	SetFont(&m_font);
	
	// init context menu
	m_pMenu = new BPopUpMenu("context", false, false);
	m_pMenu->AddItem(new BMenuItem("About wsnote...",
		new BMessage('Abou')));
	m_pMenu->AddItem(new BSeparatorItem());
	BMenuItem* pItem = new BMenuItem("Edge Snap",
		new BMessage('Snap'));
	pItem->SetMarked(m_bSnapEdge);
	m_pMenu->AddItem(pItem);
	pItem = new BMenuItem("Float", new BMessage('Floa'));
	m_pMenu->AddItem(pItem);
	m_pMenu->AddItem(new BMenuItem("Quit",
		new BMessage(B_QUIT_REQUESTED)));
	
}

void View::AllAttached() {
	// menu isn't a child, so it doesn't get targets set
	// automatically
	m_pMenu->SetTargetForItems(Window());

	// if window floats, set associated menu item
	BMenuItem* pFloatItem = m_pMenu->FindItem('Floa');
	if(pFloatItem)
		pFloatItem->SetMarked(((class Window*)Window())->isFloating());

	// init window-size limits
	float fMinW, fMaxW, fMinH, fMaxH;
	Window()->GetSizeLimits(&fMinW, &fMaxW, &fMinH, &fMaxH);
	fMinW = s_nMinWidth;
	fMinH = s_nMinHeight;
	Window()->SetSizeLimits(fMinW, fMaxW, fMinH, fMaxH);
}

void View::Draw(BRect updateRect) {
	int32 nWorkspace = current_workspace() + 1;

	BRect b = Bounds();
/*	if(m_bDragInDeskbar) {
		rgb_color red = {255,0,0,255};
		SetHighColor(red);
	}
	else */
	SetHighColor(m_dragHilight == DRAG_MOVE ? s_bgColorHilight : s_bgColor);

	BRect bBack = b;
	bBack.InsetBy(1.0,1.0);
	FillRect(bBack);

	SetHighColor(s_textColor);
	sprintf(buffer, "%ld", nWorkspace);
	float fWidth = m_font.StringWidth(buffer);
	float fHeight = m_fontHeight.ascent + m_fontHeight.descent - 2.0;
	BPoint p(
		(b.Width()-fWidth)/2 + 0.5,
		(b.Height()-fHeight)/2 + m_fontHeight.ascent);
	DrawString(buffer, p);

	SetHighColor(s_borderColor);
	StrokeRect(b);
	
	drawDragHilight(b);
	b.InsetBy(1.0, 1.0);
	drawDragHilight(b);
}

void View::drawDragHilight(BRect b) {
	// hilight draggable bits
	SetHighColor(s_hilightColor);
	bool bCorners = b.Height() > s_nCornerSize*2 &&
		b.Width() > s_nCornerSize*2;

	BPoint topleft = b.LeftTop();
	if(bCorners)
		topleft.x += s_nCornerSize;
	BPoint topright = b.RightTop();
	if(bCorners)
		topright.x -= s_nCornerSize;
	BPoint bottomleft = b.LeftBottom();
	if(bCorners)
		bottomleft.x += s_nCornerSize;
	BPoint bottomright = b.RightBottom();
	if(bCorners)
		bottomright.x -= s_nCornerSize;
	BPoint lefttop = b.LeftTop();
	if(bCorners)
		lefttop.y += s_nCornerSize;
	BPoint leftbottom = b.LeftBottom();
	if(bCorners)
		leftbottom.y -= s_nCornerSize;
	BPoint righttop = b.RightTop();
	if(bCorners)
		righttop.y += s_nCornerSize;
	BPoint rightbottom = b.RightBottom();
	if(bCorners)
		rightbottom.y -= s_nCornerSize;
			
	switch(m_dragHilight) {
		case DRAG_UL:
			StrokeLine(b.LeftTop(), lefttop);
			StrokeLine(b.LeftTop(), topleft);
			break;

		case DRAG_TOP:
			StrokeLine(topleft, topright);
			break;
				
		case DRAG_UR:
			StrokeLine(b.RightTop(), righttop);
			StrokeLine(b.RightTop(), topright);
			break;

		case DRAG_RIGHT:
			StrokeLine(righttop, rightbottom);
			break;

		case DRAG_LR:
			StrokeLine(rightbottom, b.RightBottom());
			StrokeLine(bottomright, b.RightBottom());
			break;

		case DRAG_BOTTOM:
			StrokeLine(bottomleft, bottomright);
			break;

		case DRAG_LL:
			StrokeLine(leftbottom, b.LeftBottom());
			StrokeLine(b.LeftBottom(), bottomleft);
			break;
				
		case DRAG_LEFT:
			StrokeLine(lefttop, leftbottom);
			break;
				
		default:
			break;
	}
}

void View::FrameResized(float width, float height) {
	BView::FrameResized(width, height);
	Invalidate();
	m_prevBounds = Bounds();
}
	
void View::MouseDown(BPoint p) {
	
	Window()->Activate(true);
	
	uint32 nButtons;
	GetMouse(&p, &nButtons);
	if(nButtons & B_PRIMARY_MOUSE_BUTTON) {
		// start drag op
		if(m_drag != DRAG_NONE) {
			(new BAlert("huh?", "i screwed up", "whatever"))->Go();
			return;
		}
		m_drag = dragType(p);
		m_dragInitPoint = ConvertToScreen(p);
		m_dragInitFrame = Window()->Frame();

		// figure deskbar (tray) frame; will be invalid rect
		// if not findable
		m_deskbarFrame = BRect();
		getDeskbarFrame(m_deskbarFrame);

		// old-school mouse tracking	
		while(true) {
			snooze(10000);
			GetMouse(&p, &nButtons);
			if(!(nButtons & B_PRIMARY_MOUSE_BUTTON))
				break;
			dragUpdate(p);
		}
		m_drag = DRAG_NONE;
	}
	else if(nButtons & B_SECONDARY_MOUSE_BUTTON) {
		// context menu
		m_pMenu->Go(ConvertToScreen(p), true);
	}
}
	
void View::MouseUp(BPoint p) {
	if(m_drag != DRAG_NONE) {
		m_drag = DRAG_NONE;
	}
}
	
void View::MouseMoved(BPoint p, uint32 nTransit,
	const BMessage* pDragMsg) {
	if(m_drag != DRAG_NONE)
		return; // dragging now handled the old-fashioned way
		
	if(nTransit == B_EXITED_VIEW)
		updateDragHilight(DRAG_NONE);
	else
		updateDragHilight(dragType(p));
}


// update drag operation
void View::dragUpdate(BPoint p) {
	ConvertToScreen(&p);
	BPoint delta = p - m_dragInitPoint;
	
	BRect newFrame = m_dragInitFrame;
	
	switch(m_drag) {
		case DRAG_MOVE: {
			newFrame.OffsetBy(delta);

			if(m_bSnapEdge) {
				// snap to screen border
				BPoint snapDelta(0.0, 0.0);
				snapToScreenEdge(newFrame, snapDelta);
				newFrame.OffsetBy(snapDelta);
			}

			setFrame(newFrame);
			break;
		}
			
		case DRAG_TOP:
			newFrame.top += delta.y;
			
			// constrain to s_nMinHeight
			if(newFrame.Height() < s_nMinHeight)
				newFrame.top -= s_nMinHeight-newFrame.Height();

			setFrame(newFrame);
			break;					

		case DRAG_BOTTOM:
			newFrame.bottom += delta.y;
			
			// constrain to s_nMinHeight
			if(newFrame.Height() < s_nMinHeight)
				newFrame.bottom += s_nMinHeight-newFrame.Height();

			setFrame(newFrame);
			break;					

		case DRAG_LEFT:
			newFrame.left += delta.x;
			
			// constrain to s_nMinWidth
			if(newFrame.Width() < s_nMinWidth)
				newFrame.left -= s_nMinWidth-newFrame.Width();

			setFrame(newFrame);
			break;					

		case DRAG_RIGHT:
			newFrame.right += delta.x;
			
			// constrain to s_nMinWidth
			if(newFrame.Width() < s_nMinWidth)
				newFrame.right += s_nMinWidth-newFrame.Width();

			setFrame(newFrame);
			break;

		case DRAG_UL:
			newFrame.left += delta.x;
			newFrame.top += delta.y;

			// constrain to s_nMinWidth
			if(newFrame.Width() < s_nMinWidth)
				newFrame.left -= s_nMinWidth-newFrame.Width();
			// constrain to s_nMinHeight
			if(newFrame.Height() < s_nMinHeight)
				newFrame.top -= s_nMinHeight-newFrame.Height();
				
			setFrame(newFrame);
			break;

		case DRAG_UR:
			newFrame.right += delta.x;
			newFrame.top += delta.y;

			// constrain to s_nMinWidth
			if(newFrame.Width() < s_nMinWidth)
				newFrame.right += s_nMinWidth-newFrame.Width();
			// constrain to s_nMinHeight
			if(newFrame.Height() < s_nMinHeight)
				newFrame.top -= s_nMinHeight-newFrame.Height();

			setFrame(newFrame);
			break;

		case DRAG_LL:
			newFrame.left += delta.x;
			newFrame.bottom += delta.y;

			// constrain to s_nMinWidth
			if(newFrame.Width() < s_nMinWidth)
				newFrame.left -= s_nMinWidth-newFrame.Width();
			// constrain to s_nMinHeight
			if(newFrame.Height() < s_nMinHeight)
				newFrame.bottom += s_nMinHeight-newFrame.Height();
				
			setFrame(newFrame);
			break;

		case DRAG_LR:
			newFrame.right += delta.x;
			newFrame.bottom += delta.y;

			// constrain to s_nMinWidth
			if(newFrame.Width() < s_nMinWidth)
				newFrame.right += s_nMinWidth-newFrame.Width();
			// constrain to s_nMinHeight
			if(newFrame.Height() < s_nMinHeight)
				newFrame.bottom += s_nMinHeight-newFrame.Height();

			setFrame(newFrame);
			break;
			
		default:
			break;
	}
}

// move & resize parent window to given frame rectangle, and
// save it to be written->attributes

void View::setFrame(BRect newFrame) {
	Window()->MoveTo(newFrame.left, newFrame.top);
	Window()->ResizeTo(newFrame.Width(), newFrame.Height());

	FrameResized(newFrame.Width(), newFrame.Height());
	((App*)be_app)->storeFrame(current_workspace(), Window()->Frame());
	
	if(!m_deskbarFrame.IsValid())
		return;

	m_bDragInDeskbar =
		newFrame.right >= m_deskbarFrame.left &&
		newFrame.left <= m_deskbarFrame.right &&
		newFrame.bottom >= m_deskbarFrame.top &&
		newFrame.top <= m_deskbarFrame.bottom;
}

// given hilight style, display a hint
void View::updateDragHilight(View::dragtype t) {
	m_dragHilight = t;
	Invalidate();
}
	
// given point, figure drag type
View::dragtype View::dragType(BPoint p) {
	BRect b = Bounds();
	bool bCorners = b.Height() > s_nCornerSize*2 &&
		b.Width() > s_nCornerSize*2;
	if(p.x < s_nBorderSize) {
		// resize, left
		if(!bCorners) {
			// not big enough for corners; dragging left side
			return DRAG_LEFT;
		}
		if(p.y < s_nCornerSize)
			return DRAG_UL;
		else if(p.y > b.bottom-s_nCornerSize)
			return DRAG_LL;
		else
			return DRAG_LEFT;
	}
	else if(p.x > b.right-s_nBorderSize) {
		// resize, right
		if(!bCorners)
			return DRAG_RIGHT;
		else if(p.y < s_nCornerSize)
			return DRAG_UR;
		else if(p.y > b.bottom-s_nCornerSize)
			return DRAG_LR;
		else
			return DRAG_RIGHT;
	}
	else if(p.y < s_nBorderSize) {
		// resize, top
		if(!bCorners)
			return DRAG_TOP;
		else if(p.x < s_nCornerSize)
			return DRAG_UL;
		else if(p.x > b.right-s_nCornerSize)
			return DRAG_UR;
		else
			return DRAG_TOP;
	}
	else if(p.y > b.bottom-s_nBorderSize) {
		// resize, bottom
		if(!bCorners)
			return DRAG_BOTTOM;
		else if(p.x < s_nCornerSize)
			return DRAG_LL;
		else if(p.x > b.right-s_nCornerSize)
			return DRAG_LR;
		else
			return DRAG_BOTTOM;
	}
	else
		// move
		return DRAG_MOVE;
}

void View::snapToScreenEdge(BRect frame, BPoint& oDelta) {

	// get screen bounds
	BRect sb = BScreen(Window()).Frame();
	
	// test top & left
	if(frame.top <= sb.top + s_nScreenSnapThreshold) {
		oDelta.y = sb.top - frame.top;
	}
	if(frame.left <= sb.left + s_nScreenSnapThreshold) {
		oDelta.x = sb.left - frame.left;
	}

		
	// test bottom & right
	if(frame.bottom >= sb.bottom - s_nScreenSnapThreshold) {
		oDelta.y = sb.bottom - frame.bottom;
	}
	if(frame.right >= sb.right - s_nScreenSnapThreshold) {
		oDelta.x = sb.right - frame.right;
	}
}

bool View::getDeskbarFrame(BRect& r) {

	// rather hardcoded at the moment:
	
	BMessage msgGetStatusFrame,
		msgGetBarViewFrame;

	msgGetStatusFrame.what = B_GET_PROPERTY;
	msgGetStatusFrame.AddSpecifier("Frame");
	msgGetStatusFrame.AddSpecifier("View", "Status");
	msgGetStatusFrame.AddSpecifier("View", "BarView");
	msgGetStatusFrame.AddSpecifier("Window", "Deskbar");
	
	msgGetBarViewFrame.what = B_GET_PROPERTY;
	msgGetBarViewFrame.AddSpecifier("Frame");
	msgGetBarViewFrame.AddSpecifier("View", "BarView");
	msgGetBarViewFrame.AddSpecifier("Window", "Deskbar");
	
	// fetch status view frame
	BMessenger msgr("application/x-vnd.Be-TSKB");

	if(!msgr.IsValid())
		return false;
	BMessage reply;
	if(msgr.SendMessage(&msgGetStatusFrame, &reply) != B_OK)
		return false;
	BRect statusFrame;
	if(reply.FindRect("result", &statusFrame) != B_OK)
		return false;
		
	// fetch parent view ('BarView') frame
	if(msgr.SendMessage(&msgGetBarViewFrame, &reply) != B_OK)
		return false;
	BRect barViewFrame;
	if(reply.FindRect("result", &barViewFrame) != B_OK)
		return false;
	
	// fetch window frame
	BRect deskbarFrame;
	if(get_deskbar_frame(&deskbarFrame) != B_OK)
		return false;
	
	// figure rect in screen coordinates
	r = statusFrame;
	r.OffsetBy(barViewFrame.LeftTop());
	r.OffsetBy(deskbarFrame.LeftTop());
	return true;
}

// -------------------------------------------------- //
// Window impl.
// -------------------------------------------------- //

void Window::MessageReceived(BMessage* pMsg) {
	switch(pMsg->what) {
		case 'Abou':
			doAbout();
			break;
			
		case 'Snap':
			toggleSnap(pMsg);
			break;
			
		case 'Floa':
			toggleFloat(pMsg);
			break;

		default:
			_inherited::MessageReceived(pMsg);
	}
}

bool Window::QuitRequested() {
	((App*)be_app)->storeSettings(m_pView->edgeSnap(),
		isFloating());
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void Window::FrameMoved(BPoint p) {
	_inherited::FrameMoved(p);
}

void Window::FrameResized(float width, float height) {
	_inherited::FrameResized(width, height);
}

void Window::WorkspaceActivated(int32 nWorkspace, bool bActive) {
	if(bActive) {
		BRect f = Frame();
		SERIAL_PRINT(("WorkspaceActivated(%d):\n"
			"Old frame (%.0f,%.0f)-(%.0f,%.0f)\n", nWorkspace,
			f.left, f.top, f.right, f.bottom));
		BRect newFrame = ((App*)be_app)->fetchFrame(nWorkspace);
		SERIAL_PRINT(("New frame (%.0f,%.0f)-(%.0f,%.0f)\n",
			newFrame.left, newFrame.top, newFrame.right, newFrame.bottom));
		MoveTo(newFrame.LeftTop());
		ResizeTo(newFrame.Width(), newFrame.Height());
		m_pView->Invalidate();
	}
}

void Window::doAbout() {
	(new BAlert("About wsnote",
		"wsnote: compact display of the current workspace\n"
		"by Eric Moon (eamoon@meadgroup.com)\n",
		"Close"))->Go();
}

void Window::toggleSnap(BMessage* pMsg) {
	BMenuItem* pSrc;
	if(pMsg->FindPointer("source", (void**)&pSrc) != B_OK)
		return;
	bool bSnap = !m_pView->edgeSnap();
	pSrc->SetMarked(bSnap);
	m_pView->setEdgeSnap(bSnap);
}

void Window::toggleFloat(BMessage* pMsg) {
	BMenuItem* pSrc;
	if(pMsg->FindPointer("source", (void**)&pSrc) != B_OK)
		return;
	bool bFloat = !isFloating();
	pSrc->SetMarked(bFloat);
	SetFeel(bFloat ? B_FLOATING_ALL_WINDOW_FEEL : B_NORMAL_WINDOW_FEEL);
}
	
bool Window::isFloating() {
	return Feel() == B_FLOATING_ALL_WINDOW_FEEL;
}

// -------------------------------------------------- //
// app class impl.
// -------------------------------------------------- //

const int32 App::s_nDefWndWidth = 24;
const int32 App::s_nDefWndHeight = 16;

App::App() :
	BApplication(g_appSignature),
	m_pFrames(NULL),
	m_nFrames(0) {
	
	bool bSnap, bFloat;
	fetchSettings(bSnap, bFloat);
	m_pWnd = new Window(fetchFrame(current_workspace()), bSnap, bFloat);
	m_pWnd->Show();
}

// be anal:
App::~App() {
	if(m_pFrames)
		delete [] m_pFrames;
}

void App::fetchSettings(bool& bSnap, bool& bFloat) {
	app_info ai;
	be_app->GetAppInfo(&ai);
	BFile file(&ai.ref, B_READ_WRITE);
	
	m_nFrames = count_workspaces();
	m_pFrames = new BRect[m_nFrames];
	
	char nameBuffer[80];
	for(int n = 0; n < m_nFrames; n++) {
		sprintf(nameBuffer, "frame-%d", n);

		file.ReadAttr(nameBuffer,
			B_RECT_TYPE, 0, &m_pFrames[n], sizeof(BRect));
	}
	
	ssize_t nRead = file.ReadAttr("edgeSnap", B_BOOL_TYPE, 0,
		&bSnap, sizeof(bool));
	if(nRead != sizeof(bool))
		bSnap = true;
		
	nRead = file.ReadAttr("float", B_BOOL_TYPE, 0,
		&bFloat, sizeof(bool));
	if(nRead != sizeof(bool))
		bFloat = true;
}

BRect App::fetchFrame(int nWorkspace) {
	BRect r;
	
	// workspace count increased? reallocate list:
	int32 nWorkspaces = count_workspaces();
	if(nWorkspaces > m_nFrames) {
		BRect* pFrames = new BRect[nWorkspaces];
		memcpy(pFrames, m_pFrames, sizeof(BRect)*m_nFrames);
		m_nFrames = nWorkspaces;
		delete [] m_pFrames;
		m_pFrames = pFrames;
	}
	
	// requested workspace in range? grab it:
	if(nWorkspace >= 0 && nWorkspace < m_nFrames)
		r = m_pFrames[nWorkspace];

	// make sure that
	// a) a valid BRect was read, and
	// b) the window isn't offscreen in this workspace:
	BRect sb = BScreen(m_pWnd).Frame();
	if(!r.IsValid() ||
		!sb.Contains(r.LeftTop())) {
		r.right = sb.right;
		r.bottom = sb.bottom;
		r.left = r.right - s_nDefWndWidth;
		r.top = r.bottom - s_nDefWndHeight;
	}

	return r;
}

void App::storeFrame(int nWorkspace, BRect r) {
	if(nWorkspace >= 0 && nWorkspace < m_nFrames)
		m_pFrames[nWorkspace] = r;
}	

void App::storeSettings(bool bSnap, bool bFloat) {
	app_info ai;
	be_app->GetAppInfo(&ai);
	BFile file(&ai.ref, B_READ_WRITE);

	char nameBuffer[80];
	for(int n = 0; n < m_nFrames; n++) {
		sprintf(nameBuffer, "frame-%d", n);
		file.WriteAttr(nameBuffer, B_RECT_TYPE, 0, &m_pFrames[n], sizeof(BRect));
	}
	file.WriteAttr("edgeSnap", B_BOOL_TYPE, 0, &bSnap, sizeof(bool));
	file.WriteAttr("float", B_BOOL_TYPE, 0, &bFloat, sizeof(bool));
}

// -------------------------------------------------- //
// main() stub
// -------------------------------------------------- //

int main() {
	App app;
	app.Run();
	return 0;
}



// END -- wsnote.cpp --
