/* 

    TiMidity -- Experimental MIDI to WAVE converter
    Copyright (C) 1995 Tuukka Toivonen <toivonen@clinet.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    linux_audio.c

    Functions to play sound on the VoxWare audio driver (Linux or FreeBSD)

*/

#include <Sound.h>
#include <SoundPlayer.h>
#include <scheduler.h>

#include <stdio.h>
#include <stdlib.h>

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include "config.h"
#include "output.h"
#include "controls.h"

static int open_output(void); /* 0=success, 1=warning, -1=fatal error */
static void close_output(void);
static void output_data(int32 *buf, int32 count);
static void flush_output(void);
static void purge_output(void);

/* BeOS Specific */
void BufferProc(void *theCookie, void *buffer, size_t size, const media_raw_audio_format &format);
void s32tof(float *sp, int32 *lp, int32 c);
BSoundPlayer *mSoundPlayer = 0;
media_raw_audio_format mFormat = {	44100.0f,
									2,
									media_raw_audio_format::B_AUDIO_FLOAT,
									B_MEDIA_LITTLE_ENDIAN,
									2 * sizeof(float) * AUDIO_BUFFER_SIZE
								};

float mOutputBuffer[2*AUDIO_BUFFER_SIZE];
sem_id mAudioSemID;

/* export the playback mode */
#define dpm beos_play_mode

PlayMode dpm =
{
	DEFAULT_RATE,
	PE_16BIT | PE_SIGNED,
	-1,
	{0,0,0,0,0}, /* default: get all the buffer fragments you can */
	"BeOS Media Kit", 'b',
	"BSoundPlayer",
	open_output,
	close_output,
	output_data,
	flush_output,
	purge_output  
};

/*************************************************************************/

static int open_output(void)
{
	mFormat.frame_rate = (float)dpm.rate;
	mFormat.channel_count = 1;
	if (!(dpm.encoding & PE_MONO))
		mFormat.channel_count = 2;
	mFormat.buffer_size = mFormat.channel_count * (sizeof(float) * AUDIO_BUFFER_SIZE);

	mSoundPlayer = new BSoundPlayer(&mFormat, "BeOS Player", BufferProc, NULL, (void *)&dpm);

	if (!mSoundPlayer)
		return -1;

	memset(mOutputBuffer, 0, sizeof(mOutputBuffer));
	mAudioSemID = create_sem(1, "timidity_sem");

	int32 priority = suggest_thread_priority(B_LIVE_AUDIO_MANIPULATION);
	set_thread_priority(find_thread(0), priority);

	mSoundPlayer->Start();
	return 0;
}


static void output_data(int32 *buf, int32 count)
{
	if (acquire_sem(mAudioSemID) == B_OK)
	{
		if (!(dpm.encoding & PE_MONO))
			count*=2; 		// Stereo samples
		s32tof(mOutputBuffer, buf, count);
		mSoundPlayer->SetHasData(true);
	}
}

static void close_output(void)
{
	set_thread_priority(find_thread(0), B_NORMAL_PRIORITY);

	if (mSoundPlayer)
	{
		mSoundPlayer->Stop();
		delete mSoundPlayer;
		mSoundPlayer = NULL;
		delete_sem(mAudioSemID);
	}
}

static void flush_output(void)
{
}

static void purge_output(void)
{
}



void BufferProc(void *theCookie, void *buffer, size_t size, const media_raw_audio_format &format)
{ 
	if (format.format != media_raw_audio_format::B_AUDIO_FLOAT)
		return; 

	memcpy(buffer, mOutputBuffer, size);
	release_sem(mAudioSemID);
}


void s32tof(float *sp, int32 *lp, int32 c)
{
	const float i16tof = (1.0f/32768.0f);
	while (c--)
	{
		int32 l = (*lp++) >> (32-16-GUARD_BITS);
		*sp++ = l * i16tof;
	}
}


