/* sparc_obuffer.cc

   Output buffer for Sun SPARC systems written by
   Tobias Bading (bading@cs.tu-berlin.de)

   Idea and first implementation for u-law output with fast downsampling by
   Jim Boucher (jboucher@flash.bu.edu)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#ifdef SPARC

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <iostream.h>

extern "C" {
#ifdef SunOS5+
#include <sys/audioio.h>
#else
#include <sun/audioio.h>
#endif // SunOS5+
}

#ifdef ULAW
#include "ulaw.h"
#endif

#ifdef SunOS
extern "C" int ioctl (int, int ...);		// Why...???
#endif

#include "args.h"
#include "header.h"
#include "obuffer.h"

int SparcObuffer::audio_fd = -1;

#ifdef ULAW
SparcObuffer::SparcObuffer (Header *header, MPEG_Args *maplay_args)
#else
SparcObuffer::SparcObuffer (uint32 number_of_channels, MPEG_Args *maplay_args)
#endif
{
#ifndef ULAW
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    cerr << "SparcObuffer: 0 < number of channels < " << MAXCHANNELS << "!\n";
    exit (1);
  }
#endif
#endif	// !ULAW

  if (audio_fd < 0)
  {
    cerr << "Internal error: SparcObuffer::audio_fd has to be initialized\n"
	    "by SparcObuffer::class_suitable()!\n";
    exit (1);
  }

  audio_info info;
  AUDIO_INITINFO (&info);
#ifdef SunOS4_1_3
  info.output_muted = False;
#endif
  info.play.port = 0;
  if (maplay_args->use_speaker)
    info.play.port |= AUDIO_SPEAKER;
  if (maplay_args->use_headphone)
    info.play.port |= AUDIO_HEADPHONE;
#ifdef SunOS4_1_3
  if (maplay_args->use_line_out)
    info.play.port |= AUDIO_LINE_OUT;
#endif

#ifdef ULAW
  bufferp = buffer;

  // configure the amd device:
  info.play.encoding = AUDIO_ENCODING_ULAW;
  info.play.precision = 8;
  info.play.channels = 1;
  info.play.sample_rate = 8000;
#else
  channels = number_of_channels;
  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;

  // configure the dbri device:
  info.play.encoding = AUDIO_ENCODING_LINEAR;
  info.play.precision = 16;
  info.play.channels = channels;
  info.play.sample_rate = maplay_args->MPEGheader->frequency ();
#endif	// !ULAW

  if (ioctl (audio_fd, AUDIO_SETINFO, &info))
  {
    perror ("configuration of /dev/audio failed");
    exit (1);
  }
}


SparcObuffer::~SparcObuffer (void)
{
  ioctl (audio_fd, AUDIO_DRAIN, NULL);
  close (audio_fd);
}


void SparcObuffer::append (uint32 channel, int16 value)
{
#ifdef ULAW
#ifdef DEBUG
  if (bufferp - buffer >= OBUFFERSIZE >> 1)
  {
    cerr << "SparcObuffer: buffer overflow!\n";
    exit (1);
  }
#endif

  // convert 16-bit PCM sample to 8-bit ulaw:
  *bufferp++ = linear2ulaw[value >> 3];
#else
#ifdef DEBUG
  if (channel >= channels)
  {
    cerr << "illegal channelnumber in SparcObuffer::append()!\n";
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    cerr << "SparcObuffer: buffer overflow!\n";
    exit (1);
  }
#endif

  *bufferp[channel] = value;
  bufferp[channel] += channels;
#endif	// !ULAW
}

void SparcObuffer::write_buffer (int)
{
#ifdef ULAW
  int length = (int)((char *)bufferp - (char *)buffer);
#else
  int length = (int)((char *)*bufferp - (char *)buffer);
#endif
  if (write (audio_fd, (char *)buffer, length) != length)
  {
    perror ("write to /dev/audio failed");
    exit (1);
  }
#ifdef ULAW
  bufferp = buffer;
#else
  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
#endif
}

#ifdef SEEK_STOP
void SparcObuffer::clear_buffer(void)
{
}

void SparcObuffer::set_stop_flag(void)
{
}
#endif // SEEK_STOP

int SparcObuffer::open_audio_device (void)
{
  int fd;

  if ((fd = open ("/dev/audio", O_WRONLY | O_NDELAY, 0)) < 0)
    if (errno == EBUSY)
    {
      cerr << "Sorry, the audio device is busy!\n";
      exit (1);
    }
    else
    {
      perror ("can't open /dev/audio for writing");
      exit (1);
    }

  // turn NDELAY mode off:
  int flags;
  if ((flags = fcntl (fd, F_GETFL, 0)) < 0)
  {
    perror ("fcntl F_GETFL on /dev/audio failed");
    exit (1);
  }
  flags &= ~O_NDELAY;
  if (fcntl (fd, F_SETFL, flags) < 0)
  {
    perror ("fcntl F_SETFL on /dev/audio failed");
    exit (1);
  }
  return fd;
}


#ifdef Solaris
void SparcObuffer::get_device_type (int fd, audio_device *devtype)
{
  if (ioctl (fd, AUDIO_GETDEV, devtype))
  {
    perror ("ioctl AUDIO_GETDEV on /dev/audio");
    exit (1);
  }
}
#else
int SparcObuffer::get_device_type (int fd)
{
#ifdef AUDIO_GETDEV
  int devtype;
  if (ioctl (fd, AUDIO_GETDEV, &devtype))
  {
    perror ("ioctl AUDIO_GETDEV on /dev/audio");
    exit (1);
  }
  return devtype;
#else
  cerr << "SparcObuffer::get_device_type(): AUDIO_GETDEV ioctl not available!\n";
  return -1;
#endif
}
#endif	// !Solaris


#ifdef ULAW
BOOL SparcObuffer::class_suitable (uint32 number_of_channels, BOOL force_amd)
#else
BOOL SparcObuffer::class_suitable (void)
#endif
{
#ifdef ULAW
  if (number_of_channels > 1)
  {
    cerr << "Your audio hardware cannot handle more than one audio channel.\n"
	    "Please use the option -l or -r for stereo streams.\n";
    return False;
  }
#endif

  // check for the dbri audio device:
  audio_fd = open_audio_device ();

#ifdef ULAW
  if (force_amd)
    return True;
#endif

#ifdef Solaris
  audio_device devtype;
  get_device_type (audio_fd, &devtype);
# ifdef ULAW
    if (!strcmp (devtype.name, "SUNW,am79c30"))
      return True;
    else if (!strcmp (devtype.name, "SUNW,dbri"))
    {
      cerr << "Your machine can produce CD-quality audio output,\n"
	      "but this binary was compiled for 8 kHz u-law ouput. (telephone quality)\n"
	      "Please recompile it without the ULAW define in COMPILERFLAGS.\n"
	      "(or use the -amd option to use this binary with low-quality output)\n";
      close (audio_fd);
      return False;
    }
# else
    if (!strcmp (devtype.name, "SUNW,dbri"))
      return True;
    else if (!strcmp (devtype.name, "SUNW,am79c30"))
    {
      cerr << "Your machine can produce 8 kHz u-law audio output only,\n"
	      "but this binary was compiled for CD-quality output.\n"
	      "Please recompile it with ULAW defined in COMPILERFLAGS\n"
	      "or use it in stdout mode as an decoder only.\n";
      close (audio_fd);
      return False;
    }
# endif
#else
  // !Solaris
# ifdef SunOS4_1_1
    // no AUDIO_GETDEV under SunOS 4.1.1, so we have to assume that there is
    // an amd device attached to /dev/audio
#   ifdef ULAW
      return True;
#   else
      cerr << "Your machine can produce 8 kHz u-law audio output only,\n"
	      "but this binary was compiled for CD-quality output.\n"
	      "Please recompile it with ULAW defined in COMPILERFLAGS\n"
	      "or use it in stdout mode as an decoder only.\n";
      close (audio_fd);
      return False;
#   endif	// !ULAW
# else
    // SunOS 4.1.3
    int device_type = get_device_type (audio_fd);
#   ifdef ULAW
      if (device_type == AUDIO_DEV_AMD)
	return True;
      else if (device_type == AUDIO_DEV_SPEAKERBOX)
      {
	cerr << "Your machine can produce CD-quality audio output,\n"
		"but this binary was compiled for 8 kHz u-law ouput. (telephone quality)\n"
		"Please recompile it without the ULAW define in COMPILERFLAGS.\n"
		"(or use the -amd option to use this binary with low-quality output)\n";
	close (audio_fd);
	return False;
      }
#   else
      if (device_type == AUDIO_DEV_SPEAKERBOX)
	return True;
      else if (device_type == AUDIO_DEV_AMD)
      {
	cerr << "Your machine can produce 8 kHz u-law audio output only,\n"
		"but this binary was compiled for CD-quality output.\n"
		"Please recompile it with ULAW defined in COMPILERFLAGS\n"
		"or use it in stdout mode as an decoder only.\n";
	close (audio_fd);
	return False;
      }
#   endif	// !ULAW
# endif	// !SunOS4_1_1
#endif	// !Solaris

#ifndef SunOS4_1_1
  close (audio_fd);
  cerr << "Sorry, I don't recognize your audio device.\n"
# ifdef ULAW
	  "Please try the -amd option or use the stdout mode.\n";
# else
	  "Please use the stdout mode.\n";
# endif
  return False;
#endif // SunOS4_1_1
}

Obuffer *create_obuffer(MPEG_Args *maplay_args)
{
  Obuffer *buffer;

  enum e_mode mode = maplay_args->MPEGheader->mode();
  enum e_channels which_channels = maplay_args->which_c;

#ifdef ULAW
  BOOL force_amd = maplay_args->force_amd;

  if (SparcObuffer::class_suitable ((mode == single_channel ||
                                     which_channels != both) ? 1 : 2,
                                     force_amd))     // amd device available?
     buffer = new SparcObuffer (maplay_args);
#else
  if (SparcObuffer::class_suitable())               // dbri device available?
     if (mode == single_channel || which_channels != both)
        buffer = new SparcObuffer (1, maplay_args);
     else
        buffer = new SparcObuffer (2, maplay_args);
#endif // ULAW
  else
     return(NULL);

  return(buffer);
}

#endif	// SPARC
