/* linux_obuffer.cc

   Output buffer for Linux written by
   Louis P. Kruger (lpkruger@phoenix.princeton.edu)

   This buffer should work for FreeBSD as well, thanks to the patches
   from the FreeBSD web site.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#if defined(LINUX) || defined (__FREEBSD__)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <iostream.h>

extern "C" {
#ifdef __FreeBSD__
#include <machine/soundcard.h>
#else
#include <sys/soundcard.h>
#endif // __FreeBSD__
}

#include "args.h"
#include "header.h"
#include "obuffer.h"

int LinuxObuffer::audio_fd = -1;

int LinuxObuffer::open_audio_device (void)
{
  int fd;

  if ((fd = open ("/dev/dsp", O_WRONLY | O_NDELAY, 0)) < 0)
    if (errno == EBUSY)
    {
      cerr << "Sorry, the audio device is busy!\n";
      exit (1);
    }
    else
    {
      perror ("can't open /dev/dsp for writing");
      exit (1);
    }


#ifdef LINUX

  // turn NDELAY mode off, for LINUX only

  int flags;
  if ((flags = fcntl (fd, F_GETFL, 0)) < 0)
  {
    perror ("fcntl F_GETFL on /dev/audio failed");
    exit (1);
  }
  flags &= ~O_NDELAY;
  if (fcntl (fd, F_SETFL, flags) < 0)
  {
    perror ("fcntl F_SETFL on /dev/audio failed");
    exit (1);
  }

#endif // LINUX

  return fd;
}


LinuxObuffer::LinuxObuffer (uint32 number_of_channels, MPEG_Args *maplay_args)
{
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    cerr << "LinuxObuffer: 0 < number of channels < " << MAXCHANNELS << "!\n";
    exit (1);
  }
#endif
  channels = number_of_channels;
  for (int i = 0; i < number_of_channels; ++i)
    bufferp[i] = buffer + i;

  if (audio_fd < 0)
  {
    cerr << "Internal error, LinuxObuffer::audio_fd has to be initialized\n"
	    "by LinuxObuffer::class_suitable()!\n";
    exit (1);
  }

  // configure the device:
  int play_precision = 16;
  int play_stereo = channels-1;
  int play_sample_rate = maplay_args->MPEGheader->frequency ();

  if(
      ioctl(audio_fd, SNDCTL_DSP_SAMPLESIZE, &play_precision) == -1 ||
      ioctl(audio_fd, SNDCTL_DSP_STEREO, &play_stereo) == -1 ||
      ioctl(audio_fd, SNDCTL_DSP_SPEED, &play_sample_rate) == -1
    )
  {
    perror ("configuration of /dev/dsp failed");
    exit (1);
  }
}

LinuxObuffer::~LinuxObuffer (void)
{
  sleep (1);
  close (audio_fd);
}

void LinuxObuffer::append (uint32 channel, int16 value)
{
#ifdef DEBUG
  if (channel >= channels)
  {
    cerr << "illegal channelnumber in LinuxObuffer::append()!\n";
    exit (1);
  }
  if (bufferp[channel] - buffer >= OBUFFERSIZE)
  {
    cerr << "buffer overflow!\n";
    exit (1);
  }
#endif
  *bufferp[channel] = value;
  bufferp[channel] += channels;
}

#ifdef SEEK_STOP
void LinuxObuffer::clear_buffer(void)
{
}

void LinuxObuffer::set_stop_flag(void)
{
}
#endif // SEEK_STOP

void LinuxObuffer::write_buffer (int)
{
  int length = (int)((char *)bufferp[0] - (char *)buffer);
  if (write (audio_fd, buffer, length) != length)
  {
    perror ("write to /dev/dsp failed");
    exit (1);
  }
  for (int i = 0; i < channels; ++i)
    bufferp[i] = buffer + i;
}


BOOL LinuxObuffer::class_suitable (uint32 number_of_channels)
{
  // open the dsp audio device:
  audio_fd = open_audio_device ();
  return(TRUE);
}

Obuffer *create_obuffer(MPEG_Args *maplay_args)
{
  Obuffer *buffer;

  enum e_mode mode = maplay_args->MPEGheader->mode();
  enum e_channels which_channels = maplay_args->which_c;

  if (LinuxObuffer::class_suitable((mode == single_channel) ||
    									       (which_channels != both)) ? 1 : 2)
    if (mode == single_channel || which_channels != both)
       buffer = new LinuxObuffer (1, maplay_args);
    else
       buffer = new LinuxObuffer (2, maplay_args);
  else
    return(NULL);

  return(buffer);
}

#endif	// LINUX || __FREEBSD__
