#include <GL/gl.h>
#include <GL/glu.h>
#include <TranslationUtils.h>


#include "StudioModel.h"
#include "ViewerSettings.h"
#include "GlView.h"


vec3_t g_vright = { 50, 50, 0 };
float g_lambert = 1.5;


GlView::GlView(BRect frame) :
   BGLView(frame, "GlView", B_FOLLOW_ALL, B_WILL_DRAW | B_PULSE_NEEDED,
	   BGL_RGB | BGL_DEPTH | BGL_DOUBLE)
{
   glDepthFunc (GL_LEQUAL);
}


GlView::~GlView(void)
{
   loadTexture (0, 0);
   loadTexture (0, 1);
}


void GlView::Pulse()
{
   g_studioModel.SetBlending (0, 0.0);
   g_studioModel.SetBlending (1, 0.0);
   
   static float prev;
   float curr = (float) clock() / CLOCKS_PER_SEC;
   
   g_studioModel.AdvanceFrame ((curr - prev) * g_viewerSettings.speedScale);
   
   prev = curr;
   
   Invalidate();
}


void GlView::MouseDown(BPoint point)
{
   int oldrx = (int)g_viewerSettings.rot[0];
   int oldry = (int)g_viewerSettings.rot[1];
   int oldtx = (int)g_viewerSettings.trans[0];
   int oldty = (int)g_viewerSettings.trans[1];
   int oldtz = (int)g_viewerSettings.trans[2];
   float oldx = point.x;
   float oldy = point.y;

   uint32 buttons = 0;

   Window()->CurrentMessage()->FindInt32("buttons", (int32 *)&buttons);
   while ( buttons ) {
      if (buttons == B_PRIMARY_MOUSE_BUTTON) {
	 g_viewerSettings.trans[0] = oldtx - (float) (point.x - oldx);
	 g_viewerSettings.trans[1] = oldty + (float) (point.y - oldy);
      } else if (buttons == B_SECONDARY_MOUSE_BUTTON) {
	 g_viewerSettings.rot[0] = oldrx + (float) (point.y - oldy);
	 g_viewerSettings.rot[1] = oldry + (float) (point.x - oldx);
      } else {
	 g_viewerSettings.trans[2] = oldtz + (float) (point.y - oldy);
      }

      Pulse();
      Draw(Bounds());

      GetMouse(&point, &buttons, true);
   }
}


void GlView::Draw(BRect updateRect)
{
   LockGL();
   draw();
   SwapBuffers();
   UnlockGL();
}


void GlView::drawFloor (void)
{
   if (g_viewerSettings.use3dfx)
   {
      glBegin (GL_TRIANGLE_STRIP);
      glTexCoord2f (1.0f, 0.0f);
      glVertex3f (100.0f, 100.0f, 0.0f);
      
      glTexCoord2f (1.0f, 1.0f);
      glVertex3f (100.0f, -100.0f, 0.0f);
      
      glTexCoord2f (0.0f, 0.0f);
      glVertex3f (-100.0f, 100.0f, 0.0f);
      
      glTexCoord2f (0.0f, 1.0f);
      glVertex3f (-100.0f, -100.0f, 0.0f);
      
      glEnd ();
   }
   else
   {
      glBegin (GL_TRIANGLE_STRIP);
      glTexCoord2f (0.0f, 0.0f);
      glVertex3f (-100.0f, 100.0f, 0.0f);
      
      glTexCoord2f (0.0f, 1.0f);
      glVertex3f (-100.0f, -100.0f, 0.0f);
      
      glTexCoord2f (1.0f, 0.0f);
      glVertex3f (100.0f, 100.0f, 0.0f);
      
      glTexCoord2f (1.0f, 1.0f);
      glVertex3f (100.0f, -100.0f, 0.0f);
      
      glEnd ();
   }
}


void GlView::setupRenderMode (void)
{
   if (g_viewerSettings.renderMode == RM_WIREFRAME)
   {
      glPolygonMode (GL_FRONT_AND_BACK, GL_LINE);
      glDisable (GL_TEXTURE_2D);
      glDisable (GL_CULL_FACE);
      glEnable (GL_DEPTH_TEST);
   }
   else if (g_viewerSettings.renderMode == RM_FLATSHADED ||
	    g_viewerSettings.renderMode == RM_SMOOTHSHADED)
   {
      glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
      glDisable (GL_TEXTURE_2D);
      glEnable (GL_CULL_FACE);
      glEnable (GL_DEPTH_TEST);
      
      if (g_viewerSettings.renderMode == RM_FLATSHADED)
	 glShadeModel (GL_FLAT);
      else
	 glShadeModel (GL_SMOOTH);
   }
   else if (g_viewerSettings.renderMode == RM_TEXTURED)
   {
      glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
      glEnable (GL_TEXTURE_2D);
      glEnable (GL_CULL_FACE);
      glEnable (GL_DEPTH_TEST);
      glShadeModel (GL_SMOOTH);
   }
}


void GlView::draw (void)
{
   glClearColor (g_viewerSettings.bgColor[0], g_viewerSettings.bgColor[1], 
		 g_viewerSettings.bgColor[2], 0.0f);
   
   if (g_viewerSettings.useStencil)
      glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
   else
      glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   
   glViewport (0, 0, w2 (), h2 ());
   
   //
   // show textures
   //
   
   if (g_viewerSettings.showTexture)
   {
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity ();
      
      glOrtho (0.0f, (float) w2 (), (float) h2 (), 0.0f, 1.0f, -1.0f);
      
      studiohdr_t *hdr = g_studioModel.getTextureHeader ();
      if (hdr)
      {
	 mstudiotexture_t *ptextures = (mstudiotexture_t *) ((byte *) hdr + hdr->textureindex);
	 float w = (float) ptextures[g_viewerSettings.texture].width 
	    * g_viewerSettings.textureScale;
	 float h = (float) ptextures[g_viewerSettings.texture].height 
	    * g_viewerSettings.textureScale;
	 
	 glMatrixMode (GL_MODELVIEW);
	 glPushMatrix ();
	 glLoadIdentity ();
	 
	 glDisable (GL_CULL_FACE);
	 
	 glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
	 float x = ((float) w2 () - w) / 2;
	 float y = ((float) h2 () - h) / 2;
	 
	 glDisable (GL_TEXTURE_2D);
	 glColor4f (1.0f, 0.0f, 0.0f, 1.0f);
	 glRectf (x - 2, y - 2, x  + w + 2, y + h + 2);
	 
	 glEnable (GL_TEXTURE_2D);
	 glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
	 glBindTexture (GL_TEXTURE_2D, g_viewerSettings.texture + 3);
	 
	 glBegin (GL_TRIANGLE_STRIP);
	 
	 glTexCoord2f (0, 0);
	 glVertex2f (x, y);
	 
	 glTexCoord2f (1, 0);
	 glVertex2f (x + w, y);
	 
	 glTexCoord2f (0, 1);
	 glVertex2f (x, y + h);
	 
	 glTexCoord2f (1, 1);
	 glVertex2f (x + w, y + h);
	 
	 glEnd ();
	 
	 glPopMatrix ();
	 
	 glClear (GL_DEPTH_BUFFER_BIT);
	 glBindTexture (GL_TEXTURE_2D, 0);
      }
      return;
   }
   
   //
   // draw background
   //
   
   if (g_viewerSettings.showBackground && d_textureNames[0] && !g_viewerSettings.showTexture)
   {
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity ();
      
      glOrtho (0.0f, 1.0f, 1.0f, 0.0f, 1.0f, -1.0f);
      
      glMatrixMode (GL_MODELVIEW);
      glPushMatrix ();
      glLoadIdentity ();
      
      glDisable (GL_CULL_FACE);
      glEnable (GL_TEXTURE_2D);
      
      glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
      glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);

      glBindTexture (GL_TEXTURE_2D, d_textureNames[0]);

      glBegin (GL_TRIANGLE_STRIP);

      glTexCoord2f (0, 0);
      glVertex2f (0, 0);

      glTexCoord2f (1, 0);
      glVertex2f (1, 0);

      glTexCoord2f (0, 1);
      glVertex2f (0, 1);

      glTexCoord2f (1, 1);
      glVertex2f (1, 1);

      glEnd ();

      glPopMatrix ();

      glClear (GL_DEPTH_BUFFER_BIT);
      glBindTexture (GL_TEXTURE_2D, 0);
   }

   glMatrixMode (GL_PROJECTION);
   glLoadIdentity ();
   gluPerspective (65.0f, (GLfloat) w () / (GLfloat) h (), 1.0f, 4096.0f);

   glMatrixMode (GL_MODELVIEW);
   glPushMatrix ();
   glLoadIdentity ();

   glTranslatef(-g_viewerSettings.trans[0], -g_viewerSettings.trans[1], 
		-g_viewerSettings.trans[2]);
    
   glRotatef (g_viewerSettings.rot[0], 1.0f, 0.0f, 0.0f);
   glRotatef (g_viewerSettings.rot[1], 0.0f, 0.0f, 1.0f);

   if (g_viewerSettings.useStencil)
   {
      /* Don't update color or depth. */
      glDisable(GL_DEPTH_TEST);
      glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);

      /* Draw 1 into the stencil buffer. */
      glEnable(GL_STENCIL_TEST);
      glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
      glStencilFunc(GL_ALWAYS, 1, 0xffffffff);

      /* Now render floor; floor pixels just get their stencil set to 1. */
      drawFloor();

      /* Re-enable update of color and depth. */ 
      glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
      glEnable(GL_DEPTH_TEST);

      /* Now, only render where stencil is set to 1. */
      glStencilFunc(GL_EQUAL, 1, 0xffffffff);
      glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
   }

   g_vright[0] = g_vright[1] = g_viewerSettings.trans[2];

   if (g_viewerSettings.mirror)
   {
      glPushMatrix ();
      glScalef (1, 1, -1);
      glCullFace (GL_BACK);
      setupRenderMode ();
      g_studioModel.DrawModel ();
      glPopMatrix ();
   }

   if (g_viewerSettings.useStencil)
      glDisable (GL_STENCIL_TEST);

   setupRenderMode ();

   glCullFace (GL_FRONT);
   g_studioModel.DrawModel ();

   //
   // draw ground
   //

   if (g_viewerSettings.showGround)
   {
      glPolygonMode (GL_FRONT_AND_BACK, GL_FILL);
      glEnable (GL_DEPTH_TEST);
      glEnable (GL_CULL_FACE);

      if (g_viewerSettings.useStencil)
	 glFrontFace (GL_CW);
      else
	 glDisable (GL_CULL_FACE);

      glEnable (GL_BLEND);
      if (!d_textureNames[1])
      {
	 glDisable (GL_TEXTURE_2D);
	 glColor4f (g_viewerSettings.gColor[0], g_viewerSettings.gColor[1], 
		    g_viewerSettings.gColor[2], 0.7f);
	 glBindTexture (GL_TEXTURE_2D, 0);
      }
      else
      {
	 glEnable (GL_TEXTURE_2D);
	 glColor4f (1.0f, 1.0f, 1.0f, 0.6f);
	 glBindTexture (GL_TEXTURE_2D, d_textureNames[1]);
      }

      glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		
      drawFloor ();

      glDisable (GL_BLEND);

      if (g_viewerSettings.useStencil)
      {
	 glCullFace (GL_BACK);
	 glColor4f (0.1f, 0.1f, 0.1f, 1.0f);
	 glBindTexture (GL_TEXTURE_2D, 0);
	 drawFloor ();

	 glFrontFace (GL_CCW);
      }
      else
	 glEnable (GL_CULL_FACE);
   }

   glPopMatrix ();
}


int GlView::loadTexture(const char *filename, int name)
{
   if (!filename || !strlen (filename)) {
      if (d_textureNames[name]) {
	 glDeleteTextures (1, (const GLuint *) &d_textureNames[name]);
	 d_textureNames[name] = 0;
	 
	 if (name == 0) {
	    strcpy (g_viewerSettings.backgroundTexFile, "");
	 } else {
	    strcpy (g_viewerSettings.groundTexFile, "");
	 }
      }
      
      return 0;
   }
   
   BBitmap *image = BTranslationUtils::GetBitmap(filename);
   
   if (image) {
      if (name == 0) {
	 strcpy (g_viewerSettings.backgroundTexFile, filename);
      } else {
	 strcpy (g_viewerSettings.groundTexFile, filename);
      }
      
      d_textureNames[name] = name + 1;
      
      glBindTexture (GL_TEXTURE_2D, d_textureNames[name]);
      glTexImage2D (GL_TEXTURE_2D, 0, 3, 
		    128, 128, //image->Bounds().IntegerWidth(), image->Bounds().IntegerHeight(),
		    0, GL_BGRA, GL_UNSIGNED_BYTE, 
		    image->Bits());
      glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      
      delete image;
      
      return name + 1;
   }
   
   return 0;
}
