// cpattr.cpp
//
// Copy an attribute (or all attributes) from one file to another.
//
// Ficus Kirkpatrick (ficus@ior.com)
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <storage/Entry.h>
#include <storage/Node.h>
#include <storage/StorageDefs.h>
#include <kernel/fs_attr.h>
#include <support/SupportDefs.h>

// Be's getopt.h doesn't seem to work for me.  (???)  Not a big
// deal -- this allows me to use int32 and friends in my code.
extern "C" {
  int32 getopt(int32, const char **, const char *);
  extern char *optarg;
  extern int32 optind;
}

// Option variables.
bool o_copy_all    = true;		// Copy all attributes or just one?
bool o_force       = false;		// Force all changes?
bool o_interactive = false;		// Interactive mode?

// Display the program usage!
//
void usage(void)
{
	printf("usage:  cpattr [ -afi ] source target [target2 ...]\n");
	printf("        -a <name>    only copy the specified attribute\n");
	printf("        -f           force;  don't ask any questions\n");
	printf("        -i           interactive;  ask before anything\n");
	exit(0);
}

// Copy an attribute from one file to another.
//
void copy_attr(const char *name, BNode &source, BNode &target)
{
	void *buf = NULL;	// Data buffer.
	char reply[32];		// Buffer for user replies.

	// Get the attribute's information.
	attr_info s_ai;
	if (source.GetAttrInfo(name, &s_ai) != B_OK) {
		printf("cpattr: attribute '%s' not found\n", name);
		goto explode;
	}

	// See if the target node already has that attribute.
	attr_info t_ai;
	if (target.GetAttrInfo(name, &t_ai) != B_ENTRY_NOT_FOUND)
	{
		if (!o_force) {
			printf("cpattr: target has an attribute '%s', overwrite? (y/n) ",
					name);
			fflush(stdout);
			fgets(reply, 32, stdin);
			if (reply[0] != 'y' && reply[0] != 'Y') {
				goto explode;
			}
		}
	}

	// Using the attr_info, allocate enough memory to copy
	// the source attribute into.
	buf = (void *)malloc(s_ai.size);
	if (!buf) {
		// We can just exit here -- no data will be lost.
		printf("cpattr: out of memory\n");
		exit(0);
	}

	if (o_interactive) {
		printf("cpattr: copy attribute '%s'? (y/n) ", name);
		fflush(stdout);
		fgets(reply, 32, stdin);
		if (reply[0] != 'y' && reply[0] != 'Y') {
			goto explode;
		}
	}

	ssize_t n;
	n = source.ReadAttr(name, 0, 0, buf, s_ai.size);
	if (n < 0) {
		printf("cpattr: unable to read attribute '%s'\n", name);
		goto explode;
	}

	// To make sure we write all of the data, keep a count of
	// how many bytes we have left to write.  Also make a temporary
	// pointer to the read buffer, so that we can increment it and
	// still be able to free all of our memory.
	int32 bytes_left = n;
	void *writebuf = buf;
	do {
		n = target.WriteAttr(name, s_ai.type, 0, writebuf, s_ai.size);
		if (n < 0) {
			printf("cpattr: unable to write attribute '%s'\n", name);
			goto explode;
		}
		bytes_left -= n;
		(char *)writebuf += n;
	} while (bytes_left > 0);

explode:		// Hey, goto sucks, but if I only have one exit point
				// I'm not going to miss anything.
	free(buf);
	return;
}

int main(int32 argc, char **argv)
{
	// Make sure we have enough arguments.
	if (argc < 3) usage();

	char attr_name[B_ATTR_NAME_LENGTH];

	// Parse command-line options.
	int32 opt;
	while ((opt = getopt(argc, argv, "a:fi")) != -1) {
		switch(opt) {
		case 'a':
			// Only copy one attribute.
			o_copy_all = false;
			strcpy(attr_name, optarg);
			break;

		case 'f':
			o_force = true;
			break;

		case 'i':
			o_interactive = true;
			break;

		default:
			usage();
			break;
		}
	}

	// Get BNodes for our source and target files, and make sure
	// we can actually open them.
	BNode source(argv[optind]);
	if (source.InitCheck() != B_OK) {
		printf("cpattr: can't open source file '%s'\n", argv[optind]);
		exit(0);
	}

	BNode target(argv[optind + 1]);
	if (target.InitCheck() != B_OK) {
		printf("cpattr: can't open target file '%s'\n", argv[optind + 1]);
		exit(0);
	}

	// If we're not copying all of the attributes, just do one shot
	// and then exit.  The option parser will have filled in attr_name
	// for us by now.
	if (!o_copy_all) {
		copy_attr(attr_name, source, target);
		exit(0);
	}

	// Okay, we're copying everything.  Walk the source node's
	// attribute list, and pass all of them to copy_attr().
	while (source.GetNextAttrName(attr_name) == B_NO_ERROR)
	{
		copy_attr(attr_name, source, target);
	}

	// All done!
	return(0);
}

