//:  BeOSSliderUI.java

package com.be.swing.plaf.beos;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.*;

/**
 * BeOS Look and Feel SliderUI delegate.
 *
 * @author  Michael Heuer
 * @version $Id: BeOSSliderUI.java,v 1.1 1999/11/10 03:59:17 heuermh Exp $
 */
public class BeOSSliderUI extends BasicSliderUI {
	
//	private static final BeOSSliderUI sliderUI = new BeOSSliderUI();

	private boolean defaultsInitialized = false;
	
	protected Color borderColor;
	protected Color trackColor;
	protected Color tickColor;
	protected Color cornerColor;
	protected Color highlightColor;
	protected Color shadowColor;
	protected Color disabledTextColor;
	
	protected static int trackWidth;
	protected int majorTickLength;
	protected int minorTickLength;
	
	protected Icon horizontalThumbIcon;
	protected Icon verticalThumbIcon;

    MouseListener thumbPressedListener;
    boolean thumbPressed = false;
	
	public BeOSSliderUI() {
		super(null);
	}
	
	public static ComponentUI createUI(JComponent c) {
		return new BeOSSliderUI();
	}
	
	public void installDefaults(JSlider slider) {
		super.installDefaults(slider);
		
		if (!defaultsInitialized) {
		
	        trackWidth = ((Integer)UIManager.get("Slider.trackWidth")).intValue();
	        majorTickLength = ((Integer)UIManager.get("Slider.majorTickLength")).intValue();
	        minorTickLength = ((Integer)UIManager.get("Slider.minorTickLength")).intValue();

			borderColor = UIManager.getColor("Slider.borderColor");
			trackColor = UIManager.getColor("Slider.trackColor");
			tickColor = UIManager.getColor("Slider.tickColor");
			cornerColor = UIManager.getColor("Slider.cornerColor");
			highlightColor = UIManager.getColor("Slider.highlightColor");
			shadowColor = UIManager.getColor("Slider.shadowColor");
			disabledTextColor = UIManager.getColor("Slider.disabledTextColor");
			horizontalThumbIcon = UIManager.getIcon("Slider.horizontalThumbIcon");
			verticalThumbIcon = UIManager.getIcon("Slider.verticalThumbIcon");
			
			defaultsInitialized = true;
		}

        scrollListener.setScrollByBlock( false );
	}

	// no uninstallDefaults method in BasicSliderUI
	
	public void uninstallUI(JComponent c) {
		super.uninstallUI(c);
		defaultsInitialized = false;
	}
	
    protected void installListeners( JSlider slider ) {
        super.installListeners( slider );
        if ( (thumbPressedListener = createThumbPressedListener() ) != null ) {
            slider.addMouseListener( thumbPressedListener );
        }
    }

    protected void uninstallListeners( JSlider slider ) {
        if ( thumbPressedListener != null ) {
            slider.removeMouseListener( thumbPressedListener );
            thumbPressedListener = null;
        }
        super.uninstallListeners( slider );
    }

    protected MouseListener createThumbPressedListener() {
       return new ThumbListener();
    }
	
	protected Color getBorderColor() {
		return borderColor;
	}
	protected Color getTrackColor() {
		return trackColor;
	}
	protected Color getTickColor() {
		return tickColor;
	}
	protected Color getCornerColor() {
		return cornerColor;
	}
	protected Color getHighlightColor() {
		return highlightColor;
	}
	protected Color getShadowColor() {
		return shadowColor;
	}
	protected Color getDisabledTextColor() {
		return disabledTextColor;
	}
	protected Icon getHorizontalThumbIcon() {
		return horizontalThumbIcon;
	}
	protected Icon getVerticalThumbIcon() {
		return verticalThumbIcon;
	}
	
//	protected int getTickLength() {
//		return 6;
//	}

    public int getTickLength() {
        final int bufferBetweenTicksAndLabels = 4;
        int length = 0;
        int majorSpacing = slider.getMajorTickSpacing();
        int minorSpacing = slider.getMinorTickSpacing();
		
        if ( slider.getPaintTicks() && (majorSpacing != 0 || minorSpacing != 0) ) {
            length += majorSpacing != 0 ? majorTickLength : minorTickLength;
        }

        if ( slider.getPaintLabels() ) {
            length += bufferBetweenTicksAndLabels;
        }

        return length;
    }

	protected int getTrackWidth() {
		return trackWidth;
	}
	protected int getTrackLength() {
        if ( slider.getOrientation() == JSlider.HORIZONTAL ) {
            return trackRect.width; 
        }
        return trackRect.height;
	}
	
	// don't paint focus on a slider
	public void paintFocus(Graphics g) {}
	
    public void paintTrack(Graphics g)  {        
        int cx, cy, cw, ch;
        int pad;

        Rectangle trackBounds = trackRect;
		boolean enabled = slider.isEnabled();

        if ( slider.getOrientation() == JSlider.HORIZONTAL ) {
            pad = trackBuffer;
            cx = pad;
            cy = (trackBounds.height / 2) - 2;
            cw = trackBounds.width;

            g.translate(trackBounds.x, trackBounds.y + cy);

			g.setColor(getCornerColor());		
			g.drawLine(0,0,cw,0);
			g.drawLine(0,0,0,5);
			
			g.setColor(getBorderColor());
			g.drawLine(0,1,0,4);
			g.drawLine(1,0,cw-1,0);
			
			// disabled text color not "disabled" enough I think
			
			g.setColor( enabled ? getShadowColor() : getDisabledTextColor() );
			g.drawLine(1,1,1,4);
			g.drawLine(1,1,cw-1,1);

			g.setColor( enabled ? getTrackColor() : getDisabledTextColor() );
			g.fillRect(2,2,cw-2,3);
			
			g.setColor(getHighlightColor());
			g.drawLine(1,5,cw,5);
			g.drawLine(cw,1,cw,5);
						
            g.translate(-trackBounds.x, -(trackBounds.y + cy));
        }
        else {
            pad = trackBuffer;
            cx = (trackBounds.width / 2) - 2;
            cy = pad;
            ch = trackBounds.height;

            g.translate(trackBounds.x + cx, trackBounds.y);

			g.setColor(getCornerColor());
			g.drawLine(0,0,5,0);
			g.drawLine(0,0,0,ch);
			
			g.setColor(getBorderColor());
			g.drawLine(0,1,0,ch-1);
			g.drawLine(1,0,4,0);
			
			g.setColor( enabled ? getShadowColor() : getDisabledTextColor() );
			g.drawLine(1,1,4,1);
			g.drawLine(1,2,1,ch-1);

			g.setColor( enabled ? getTrackColor() : getDisabledTextColor() );
			g.fillRect(2,2,3,ch-2);
			
			g.setColor(getHighlightColor());
			g.drawLine(1,ch,5,ch);
			g.drawLine(5,1,5,ch);		
			
            g.translate(-(trackBounds.x + cx), -trackBounds.y);
        }
    }
	
	private static Dimension thumbIconSize = new Dimension(0,0);
	
	protected Dimension getThumbSize() {
		if (slider.getOrientation() == JSlider.HORIZONTAL) {
			thumbIconSize.height = getHorizontalThumbIcon().getIconHeight();
			thumbIconSize.width = getHorizontalThumbIcon().getIconWidth();
		} else {
			thumbIconSize.height = getVerticalThumbIcon().getIconHeight();
			thumbIconSize.width = getHorizontalThumbIcon().getIconWidth();
		}
		return thumbIconSize;
	}
	
	public void paintThumb(Graphics g) {
		Rectangle knobBounds = thumbRect;
		
		g.translate(knobBounds.x, knobBounds.y);
		if (slider.getOrientation() == JSlider.HORIZONTAL) {
			getHorizontalThumbIcon().paintIcon(slider, g, 0,0);
		} else {
			getVerticalThumbIcon().paintIcon(slider, g, 0,0);
		}
		g.translate(-knobBounds.x, knobBounds.y);
	}		

    protected int getThumbOverhang() {
        return 3;
    }

    protected void paintMinorTickForHorizSlider( Graphics g, Rectangle tickBounds, int x ) {
        paintTickForHorizSlider( g, x, minorTickLength );
    }

    protected void paintMajorTickForHorizSlider( Graphics g, Rectangle tickBounds, int x ) {
        paintTickForHorizSlider( g, x, majorTickLength );
    }

    protected void paintMinorTickForVertSlider( Graphics g, Rectangle tickBounds, int y ) {
        paintTickForVertSlider( g, y, minorTickLength );
    }

    protected void paintMajorTickForVertSlider( Graphics g, Rectangle tickBounds, int y ) {
        paintTickForVertSlider( g, y, majorTickLength );
    }

    protected void paintTickForHorizSlider( Graphics g, int x, int length ) {
        g.translate(x,0);

		// maybe if I wasn't drawing out of my bounds
		//    I wouldn't have to worry about the redraw problem?
		
		int over = getThumbOverhang();

		// same colors for disabled?		

		g.setColor(getBorderColor());
//		g.drawLine(0,0,0,length);

		g.drawLine(0,0-over,0,length-over);
		
        g.setColor(getHighlightColor());
//		g.drawLine(1,0,1,length);

		g.drawLine(1,0-over,1,length-over);

        g.translate( -x, 0 );

    }

    protected void paintTickForVertSlider( Graphics g, int y, int length ) {
        g.translate( 0, y );

		// same colors for disabled?
		
		int over = getThumbOverhang();
		
		g.setColor(getBorderColor());
//		g.drawLine(0,0,length,0);

		g.drawLine(0-over,0,length-over,0);
		
		g.setColor(getHighlightColor());
//		g.drawLine(0,1,length,1);

		g.drawLine(0-over,1,length-over,1);

        g.translate( 0, -y );

    }

	// to keep the ticks repainting while the mouse is down
	
    class ThumbListener extends MouseAdapter {

		// quite a hack!
		//
		// doesn't seem to make the slider any less responsive
		// than it normally is .. could overwhelm a platform
		// that doesn't keep up with the event thread well... 
		
		private Timer t = new Timer(60, new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				beosGetSlider().repaint();
			}
		});

        public void mousePressed( MouseEvent e ) {
            if ( beosGetThumbBounds().contains( e.getX(), e.getY() ) ) {
                thumbPressed = true;
                
//                beosGetSlider().repaint();
				t.setRepeats(true);
				t.setCoalesce(true);
				t.start();
            }
        }


        public void mouseReleased( MouseEvent e ) {
            thumbPressed = false;
            
            t.stop();
//            beosGetSlider().repaint();
        }
    }

    Rectangle beosGetThumbBounds() {
        return thumbRect;
    }

    JSlider beosGetSlider() {
        return slider;
    }
}