#include "ArpEcho.h"

#ifndef AMPUBLIC_AMFILTERCONFIG_H
#include "AmPublic/AmFilterConfigLayout.h"
#endif
#include "AmPublic/AmControls.h"

#include <experimental/ResourceSet.h>

#ifndef ARPKERNEL_ARPDEBUG_H
#include <ArpKernel/ArpDebug.h>
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

ArpMOD();

static BResourceSet gResources;
static int32 gInitResources = 0;
BResourceSet& Resources()
{
	if (atomic_or(&gInitResources, 1) == 0) {
		gResources.AddResources((void*)Resources);
		atomic_or(&gInitResources, 2);
	} else {
		while ((gInitResources&2) == 0) snooze(20000);
	}
	return gResources;
}

ArpEchoFilter::ArpEchoFilter(ArpEchoFilterAddOn* addon,
							 AmFilterHolderI* holder,
							 const BMessage* config)
	: AmFilterI(addon),
	  mAddOn(addon),
	  mHolder(holder),
	  mDepth(3),
	  mMultiplier(1), mQuantize(PPQN), mEighths(2)
{
	if (config) PutConfiguration(config);
}

ArpEchoFilter::~ArpEchoFilter()
{
}

AmEvent* ArpEchoFilter::HandleEvent(AmEvent* event, const am_filter_params* /*params*/)
{
	if( !event ) return event;
	
	ArpVALIDATE(mAddOn != 0 && mHolder != 0, return event);
	
	AmEvent* 	head = event;
	AmTime		time = mMultiplier * ((mQuantize*2)/mEighths);

	if( event->Type() == event->NOTEON_EVENT || event->Type() == event->NOTEOFF_EVENT ) {
		AmEvent*	prevEvent = event;
		for (int32 k=0; k<mDepth; k++) {
			AmEvent* nextEvent = event->Copy();
			nextEvent->SetTime(nextEvent->Time() + (time * (k+1)));
			AmNoteOn* nextOn = dynamic_cast<AmNoteOn*>(nextEvent);
			AmNoteOff* nextOff = nextOn ? NULL : dynamic_cast<AmNoteOff*>(nextEvent);
			if( nextOn ) {
				// Adjust start velocity by echo index.
				float	velocityDelta = float(nextOn->Velocity()) / (mDepth + 1);
				nextOn->SetVelocity(nextOn->Velocity() - uint8((k+1)*velocityDelta));
				// Adjust release velocity by echo index.
				velocityDelta = float(nextOn->ReleaseVelocity()) / (mDepth + 1);
				nextOn->SetReleaseVelocity(nextOn->ReleaseVelocity() - uint8((k+1)*velocityDelta));
				// Add note to resulting chain.
				prevEvent->AppendEvent(nextEvent);
				// Adjust duration if intermediate note, so that it doesn't
				// go past next echo.
				if( k < (mDepth-1) && event->Duration() > time )
					event->SetDuration(time);
				prevEvent = nextEvent;
			} else if( nextOff ) {
				// Adjust velocity by echo index.
				float	velocityDelta = float(nextOff->Velocity()) / (mDepth + 1);
				nextOff->SetVelocity(nextOff->Velocity() - uint8((k+1)*velocityDelta));
				// Add note to resulting chain.
				prevEvent->AppendEvent(nextEvent);
				prevEvent = nextEvent;
			} else {
				nextEvent->Delete();
			}
		}
	}
	
	return head;
}

class ArpEchoFilterSettings : public AmFilterConfigLayout
{
public:
	ArpEchoFilterSettings(AmFilterHolderI* target,
						  const BMessage& initSettings)
		: AmFilterConfigLayout(target, initSettings)
	{
		try {
			AddLayoutChild((new ArpRunningBar("TopVBar"))
				->SetParams(ArpMessage()
					.SetInt32(ArpRunningBar::OrientationP, B_VERTICAL)
					.SetFloat(ArpRunningBar::IntraSpaceP, .5)
				)
				->AddLayoutChild((new ArpTextControl(
										SZ_FILTER_LABEL, "Label:","",
										mImpl.AttachTextControl(SZ_FILTER_LABEL)))
					->SetParams(ArpMessage()
						.SetString(ArpTextControl::MinTextStringP, "8")
						.SetString(ArpTextControl::PrefTextStringP, "8888888888")
					)
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,3)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
						.SetBool(ArpRunningBar::AlignLabelsC,true)
					)
				)
				->AddLayoutChild((new ArpTextControl(
										"depth", "Depth:","",
										mImpl.AttachTextControl("depth")))
					->SetParams(ArpMessage()
						.SetString(ArpTextControl::MinTextStringP, "888")
						.SetString(ArpTextControl::PrefTextStringP, "88888888")
					)
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,3)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
						.SetBool(ArpRunningBar::AlignLabelsC,true)
					)
				)
				->AddLayoutChild((new AmDurationControl("duration", "Duration:", this, initSettings))
					->SetConstraints(ArpMessage()
						.SetFloat(ArpRunningBar::WeightC,3)
						.SetInt32(ArpRunningBar::FillC,ArpEastWest)
						.SetBool(ArpRunningBar::AlignLabelsC,true)
					)
				)
			);
		} catch(...) {
			throw;
		}
		
		Implementation().RefreshControls(mSettings);
	}
	
protected:
	typedef AmFilterConfigLayout inherited;
};

status_t ArpEchoFilter::GetConfiguration(BMessage* values) const
{
	status_t err = AmFilterI::GetConfiguration(values);
	if (err != B_OK) return err;
	
	if ((err=values->AddInt32("depth", mDepth)) != B_OK) return err;
	if ( (err = values->AddInt32(AM_MULTIPLIER_CONTROL_KEY_STR, mMultiplier)) != B_OK ) return err;
	if ( (err = add_time(*values, AM_QUANTIZE_CONTROL_KEY_STR, mQuantize)) != B_OK ) return err;
	if ( (err = values->AddInt32(AM_EIGHTHS_CONTROL_KEY_STR, mEighths)) != B_OK ) return err;

	return B_OK;
}

status_t ArpEchoFilter::PutConfiguration(const BMessage* values)
{
	status_t err = AmFilterI::PutConfiguration(values);
	if (err != B_OK) return err;
	
	int32 i;
	if (values->FindInt32("depth", &i) == B_OK) mDepth = i;
	AmTime t;
	/* Backwards compatibility
	 */
	if (find_time(*values, "time", &t) == B_OK) {
		AmDurationControl::SplitTicks(t, &mMultiplier, &mQuantize, &mEighths);
	}
	
	if (values->FindInt32(AM_MULTIPLIER_CONTROL_KEY_STR, &i) == B_OK) mMultiplier = i;
	if (find_time(*values, AM_QUANTIZE_CONTROL_KEY_STR, &t) == B_OK) mQuantize = t;
	if (values->FindInt32(AM_EIGHTHS_CONTROL_KEY_STR, &i) == B_OK) {
		if (i <= 2) i = 2;
		else if (i <= 4) i = 3;
		else if (i <= 6) i = 5;
		else i = 7;
		mEighths = i;
	}

	return B_OK;
}

status_t ArpEchoFilter::Configure(ArpVectorI<BView*>& panels)
{
	BMessage config;
	status_t err = GetConfiguration(&config);
	if (err != B_OK) return err;
	panels.push_back(new ArpEchoFilterSettings(mHolder, config));
	return B_OK;
}

/* ----------------------------------------------------------------
   ArpEchoFilterAddOn Class
   ---------------------------------------------------------------- */
void ArpEchoFilterAddOn::GetVersion(int32* major, int32* minor) const
{
	*major = 1;
	*minor = 0;
}

BBitmap* ArpEchoFilterAddOn::Image(BPoint requestedSize) const
{
	const BBitmap* bm = Resources().FindBitmap("Class Icon");
	if (bm) return new BBitmap(bm);
	return NULL;
}

extern "C" _EXPORT AmFilterAddOn* make_nth_filter(int32 n, image_id /*you*/,
												  const void* cookie, uint32 /*flags*/, ...)
{
	if (n == 0) return new ArpEchoFilterAddOn(cookie);
	return NULL;
}
