/*
 * Copyright (c)1997 - 2000 by Angry Red Planet and Eric Hackborn.
 * All rights reserved.
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Eric Hackborn,
 * at <hackborn@angryredplanet.com>.
 *
 * ----------------------------------------------------------------------
 *
 * AmTool.h
 *
 * The AmTool class defines the abstract behaviour that a
 * tool operating on Sequitur data needs to implement.  Tools appear
 * as buttons to the user, so in the constructor subclasses should
 * assign the appropriate BPictures to the pic and shadow pic (the
 * standard and 'pressed' look for the button, respectively).
 *
 * This file also contains some functions for accessing the installed
 * tools.  These functions are written from the perspective of a three-
 * button mouse (PrimaryTool(), etc are the functions).  I'm not sure
 * if this is the best place for this data.  It used to be stored in
 * the app, but now that this is in a library, we technically should
 * have no knowledge of the app.
 *
 * ----------------------------------------------------------------------
 *
 * Known Bugs
 * ~~~~~~~~~~
 *
 *	- None.  Ha, ha!
 *
 * ----------------------------------------------------------------------
 *
 * History
 * ~~~~~~~
 * Jan 10, 1999		hackborn
 * Mutated this file from SeqTool, added the convenience functions for
 * the current tool (PrimaryTool(), SetPrimaryTool() etc).
 */
 
 
#ifndef AMPUBLIC_AMTOOL_H
#define AMPUBLIC_AMTOOL_H

#include <be/interface/Menu.h>
#include <be/interface/Bitmap.h>
#include "AmPublic/AmSongObserver.h"
#include "AmPublic/AmTrackRef.h"
class AmSelectionsI;
class AmToolTarget;

extern const char* SZ_TOOL_CLASS_NAME;		// AmTool::ClassName()

/* These are the choices for the flags variable.
 */
enum {
	AM_TOOL_SCROLL_X_TO_MOUSE	= 0x00000001,
	AM_TOOL_SCROLL_Y_TO_MOUSE	= 0x00000002,
	AM_TOOL_SCROLL_TO_MOUSE		= 0x00000003
};

/*************************************************************************
 * AM-TOOL
 * This is the generic interface for all tools.  There are five basic
 * responsibilities subclasses have: answer a Name(), install an image in
 * the constructor, then implement the three mouse messages to do something
 * interesting.
 *
 * The complete editing interface available to tools is the API's of the
 * AmToolTarget (used mainly to abstract any view-specific behaviour), the
 * AmTrack, and to some degree the AmSong.
 *************************************************************************/
class AmTool : public AmSongObserver
{
public:
	AmTool(uint32 flags = AM_TOOL_SCROLL_TO_MOUSE);
	virtual ~AmTool();

	uint32				Flags() const;
	void				SetFlags(uint32 flags);
	virtual BPoint		ImageSize() const;
	const BBitmap*		ImageNormal() const;
	const BBitmap*		ImageOver() const;
	const BBitmap*		ImagePressed() const;
	virtual const char*	Name() const = 0;
	/* The format for the sequitur tool identifier, which is your
	 * company or individual ID, followed by a tool name.  Ergo:
	 * "arp:Pencil"
	 */
	virtual const char*	ClassName() const = 0;
	virtual const char*	ToolTip() const = 0;
	/* Get and set my current state.  Subclasses should be aware that
	 * the views will use these methods to configure tools for use in
	 * a specific view.  Therefore, if the tool has properties made
	 * by the user that should override what the tool would configure,
	 * those user properties should be stored elsewhere.
	 */
	virtual status_t	GetSettings(BMessage* settings) const;
	virtual status_t	SetSettings(const BMessage* settings);
	/* Subclasses should reset to the defaults or the last setting
	 * state made by the user, if this tool can be configured by users.
	 */
	virtual void		ResetSettings();
	/* Subclasses can answer with a new view that contains any controls
	 * necessary for setting the tool properties.  By default, zero is returned,
	 * which means that the subclass has no properties.
	 */
	virtual BView*		NewPropertiesView() const;
	
	/* Subclasses override these messages to deal with mouse notifications
	 * coming from a view.
	 */
	virtual void		MouseDown(	AmTrackRef trackRef,
									AmToolTarget* target,
									BPoint where) = 0;

	virtual void		MouseUp(	AmTrackRef trackRef,
									AmToolTarget* target,
									BPoint where) = 0;

	virtual void		MouseMoved(	AmTrackRef trackRef,
									AmToolTarget* target,
									BPoint where,
									uint32 code) = 0;

	/* Subclasses are given the opportunity to handle messages that
	 * are dropped on their on-screen representation.  They should respond
	 * with true if they handle the given message, otherwise false allows
	 * it to be delegated in the standard manner.
	 * NOTE:  This will probably go away.
	 */
	virtual bool AcceptMessage(BMessage *msg)	{ return false; }
	
	/* Properties accessing
	 * NOTE: This will probably go away.
	 */
	virtual BMenu* PropertyMenu() 				{ return 0; }

	/* When finding or moving events, tools will generally want to look
	 * first directly where the mouse is positioned, and then look at a
	 * quantized value based on that -- for example, if the mouse is half-
	 * way between beat one and two, and a control change exists at beat
	 * one, and it's the only control change in the track, then the program
	 * should know the duration of notes being placed, and, if it's an eighth
	 * note or above, it should know to grab that control change.
	 *
	 * This is a problem, because the duration of events that the user is
	 * entering is based on a filter that may or may not exist.  Therefore,
	 * the first AmDurationFilter in the track's edit pipeline is considered
	 * a special filter, as it is used for the basis of quantizing.  This is
	 * an easy utility function for getting that value.
	 */
	 AmTime		QuantizeTime(const AmTrack* track);

protected:
	/* These images are expected to come from a resource file -- they
	 * will not be deleted when this instance destructs.
	 */
	const BBitmap*	mImageNormal;
	const BBitmap*	mImageOver;
	const BBitmap*	mImagePressed;
	/* Use this method to generate a time to be used for a new event, given the
	 * x pixel value.  This method will make sure the new time falls into the
	 * current constraints -- if it should be snapped to a grid, etc.
	 */
	AmTime		NewEventTime(float xPixel, const AmToolTarget* target, AmTime quantize) const;

private:
	/* The flags supplied in the constructor.
	 */
	uint32		mFlags;
	/* Little bit of a hack so we can reset the settings.  This whole
	 * mechanism needs to be reworked.
	 */
	uint32		mOriginalFlags;
};

#endif 

