
#ifndef AMPUBLIC_AMFILTERI_H
#define AMPUBLIC_AMFILTERI_H

#include <be/interface/Bitmap.h>
#include <be/support/List.h>
#include <be/support/String.h>

#include <ArpBuild.h>

#include <AmPublic/AmEvents.h>
#include <AmPublic/AmInstrumentI.h>
#include <ArpKernel/ArpConfigurableI.h>
#include <ArpKernel/ArpConfigureWatch.h>
#include <ArpKernel/ArpRef.h>
#include "AmPublic/AmSongRef.h"
#include "AmPublic/AmTrackRef.h"
class BMessage;
class BView;
class AmFilterHolderI;
class AmFilterI;
class AmFilterAddOnHandle;

/* Names for various string constants used in filter
 * messages.  These are the message values filled in
 * by AmFilterAddOn::GetArchiveTemplate() and
 * AmFilterI::Archive().
 */
extern const char* SZ_FILTER_NAME;				// AmFilterAddOn::Name()
extern const char* SZ_FILTER_CLASS_NAME;		// AmFilterAddOn::ClassName()
extern const char* SZ_FILTER_SHORT_DESCRIPTION;	// AmFilterAddOn::ShortDescription()
extern const char* SZ_FILTER_AUTHOR;			// AmFilterAddOn::Author()
extern const char* SZ_FILTER_MAJOR_VERSION;		// AmFilterAddOn::GetVersion()
extern const char* SZ_FILTER_MINOR_VERSION;		// AmFilterAddOn::GetVersion()
extern const char* SZ_FILTER_TYPE;				// AmFilterAddOn::Type()
extern const char* SZ_FILTER_ASSOC_TYPE;		// AmFilterAddOn::ClassNameForType()
extern const char* SZ_FILTER_IMAGE;				// AmFilterAddOn::Image()
extern const char* SZ_FILTER_LABEL;				// AmFilterI::Label()
extern const char* SZ_FILTER_ARCHIVE;			// All of the above

enum {
	AM_ARCHIVE_IMAGE			= 1<<0,
	AM_ARCHIVE_DESCRIPTION		= 1<<1,
	
	AM_ARCHIVE_ALL				= 0xFFFF
};
	
/*****************************************************************************
 *
 *	AM-FILTER-ADD-ON CLASS
 *
 *	The AmFilterAddOn class is the public definition of a type of
 *	filter add-on.
 *
 *****************************************************************************/

class AmFilterAddOn : public ArpRefable
{
public:
	AmFilterAddOn(const void* cookie);
	
	virtual void AddReference(ArpBaseRef* owner=NULL) const;
	virtual void RemReference(ArpBaseRef* owner=NULL) const;
	
	// --------------------------------------------------------
	// INTERFACE VERSIONING.
	// --------------------------------------------------------
	
	/* Return the version of the addon interface that this
	 * object is implementing.  Add ons should define this
	 * to always return VERSION_CURRENT; the MIDI system will
	 * use this to make sure it doesn't try to make your addon
	 * do anything it can't.
	 */

	enum {
		VERSION_1 = 19980926,	// First version
		
		VERSION_CURRENT = VERSION_1
	};
	typedef uint32 VersionType;
	
	virtual VersionType Version(void) const = 0;

	// --------------------------------------------------------
	// BASIC INTERFACE.
	// --------------------------------------------------------
	
	/* Return a name for this filter.  This is shown to the user.
	 */
	virtual const char* Name() const = 0;
	/* Return a uniquely identifying name for this filter.
	 * It should be formed as an implementor prefix followed
	 * by the name, i.e. "arp:echo" or "be:quantize".
	 */
	virtual const char* ClassName() const = 0;
	/* Return a short description for the filter.
	 */
	virtual const char* ShortDescription() const = 0;
	/* Return the author's name, typically first and last, i.e.
	 * "Jane Smith".  Returning NULL is acceptable.
	 */
	virtual const char* Author() const = 0;
	/* Fill the major and minor version numbers in the supplied args.
	 * An addon which supplies a major number of 1 and a minor of 0
	 * would appear to the user as 'v1.0'.
	 */
	virtual void		GetVersion(int32* major, int32* minor) const = 0;
	
	/* Return the type of this filter, one of the values
	 * in the "type" enum.
	 */
	enum type {
		// A normal filter is one that transforms an input event
		// into an output event stream.  A filter of this type
		// will usually run in a batch mode.
		NORMAL_FILTER		= 0,
		// A source filter is one that interfaces with an input
		// device.  It can only be the first filter in a track's
		// input pipeline.
		SOURCE_FILTER		= 1,
		// A destination filter is on that interfaces with an
		// output device.  It can only be the last filter in a
		// track's output pipeline, and its HandleEvent() method
		// executes in the playback thread context.  This means
		// that it can (and actually must) block until it is time
		// for the event given to it to be performed.
		DESTINATION_FILTER	= 2
	};
	virtual type	Type() const = 0;

	/* Return the name for a corresponding filter class name of
	 * the given type.  The default implementation returns ClassName()
	 * for your Type(), and NULL for all others.  You can implement
	 * this to associate SOURCE and DESTINATION filters.
	 */
	virtual const char* ClassNameForType(type inType) const;
	
	/* Answer with a bitmap that is equal to or larger than the
	 * requested dimensions.  Currently, all subclasses should
	 * respond to this with a bitmap that is 20 pixels by 20 pixels;
	 * this might change in the future.
	 * NOTE: BE PREPARED TO RECEIVE A NULL BITMAP.
	 */
	virtual BBitmap* Image(BPoint requestedSize) const;
	
	/* Fill in the BMessage with an archive for the prototypical
	 * instance of this filter add-on.  You only need to implement
	 * this if you need to store non-standard information in the
	 * archive.
	 */
	virtual status_t GetArchiveTemplate(BMessage* into, uint32 flags) const;
	
	/* Return how appropriate you are at instantiating the filter
	 * in 'config'.  The default implementation returns a match
	 * quality of .5 for add-ons of the same class, and -1 ("no
	 * match") for non-matching classes.  You only need to implement
	 * this if you need to modify that behaviour.
	 */
	virtual float CheckInstantiation(const BMessage* archive) const;
	
	/* Create a new instance of this filter.  The instance, an
	 * AmFilterI, is what does the actual work.  During creation,
	 * you are given an AmFilterHolderI object, which is this
	 * filter instance's context inside of the MIDI system,  You will
	 * almost definately want to keep ahold of this object inside of
	 * your filter instance [it is guaranteed to remain valid for the
	 * lifetime of your AmFilterI], as it provides your main
	 * interface back into the MIDI system.
	 *
	 * The optional 'config' parameter provides an initial settings
	 * configuration for this filter instance; see
	 * AmFilterI::GetConfiguration() for how to get a BMessage
	 * to use here.
	 */
	virtual AmFilterI* NewInstance(AmFilterHolderI* holder,
									const BMessage* config = NULL) = 0;

	static float MatchClassArchive(const char* className,
								   const BMessage* archive);
	
	// --------------------------------------------------------
	// SUPPORT FUNCTIONS.
	// --------------------------------------------------------
	
	/* Return the image for this add-on that should be displayed
	 * to the user.  This is Image() with any requested tinting
	 * transformation added.
	 */
	BBitmap* FinalImage(BPoint requestedSize) const;
	
	/* Change the tinting for this add-on.  The default tint is
	 * B_TRANSPARENT_COLOR, i.e. nothing.
	 */
	void SetTint(rgb_color tint);
	rgb_color Tint() const;
	
	/* Helper function for others who want to perform tinting.
	 */
	static BBitmap* TintImage(BBitmap* image, rgb_color tint);
	
protected:
	virtual ~AmFilterAddOn();

private:
	friend class AmFilterAddOnHandle;
	
	const void* mCookie;
	rgb_color mTint;
};

/*****************************************************************************
 *
 *	AM-FILTER-HOLDER-I CLASS
 *
 *	The AmFilterHolderI class is the AmKernel system's context for
 *	a filter in a pipeline.  It "holds" the actual instance of a particular
 *	filter, an AmFilterI object.  The AmFilterI defines the
 *	interface from the MIDI system into a particular filter implementation;
 *	this AmFilterHolderI class defines the interface from the filter
 *	addon back into its surrounding context in the MIDI system, most
 *	importantly the filters connected to it in the pipeline.
 *
 *	NOTE: The ArpConfigurableI interface implemented by this class calls
 *	through to its AmFilterI's configurable interface.  However, this
 *	class adds observer notification and thread safety, so you -must-
 *	use this one when implementing configuration views.
 *
 *****************************************************************************/

class AmFilterHolderI : public AmSafeDelete, public ArpConfigureWatch
{
public:
	/* This is the actual filter instance that the object holds.
	 */
	virtual AmFilterI* Filter() const = 0;
	
	/* Shortcut for Filter()->AddOn()->Type().
	 */
	virtual AmFilterAddOn::type Type() const = 0;

	/* Answer the instrument for the track I reside in, or NULL if none.
	 */
	virtual ArpCRef<AmInstrumentI> TrackInstrument() const = 0;
	
	/* Answer true if this filter's processing should be bypassed.
	 */
	virtual bool IsBypassed() const = 0;
	virtual void SetBypassed(bool bypass) = 0;
	
	/* Transform between real-time and pulses.  Only valid while
	 * a song is playing; when not playing, these always return
	 * -1.
	 */
	virtual AmTime RealtimeToPulse(bigtime_t time) const = 0;
	virtual bigtime_t PulseToRealtime(AmTime pulse) const = 0;
	
	/* Record the given even chain into the currently recording
	 * song.  Ownership of the given events is transfered to the
	 * callee.
	 */
	virtual status_t GenerateEvents(AmEvent* events) = 0;
	
	/* This is the main/primary/default successor to this holder
	 * in the pipeline.  This is the next filter that will process
	 * an event in a pipeline.  The MIDI system will automatically
	 * set the next filter for each event it hands to you, so simple
	 * filters will not need to deal with this.  However, if you
	 * generate any new events, you will need to fill their
	 * NextFilter() field with this [or another] filter holder, so
	 * that they are correctly processed by the rest of the pipeline.
	 */
	virtual AmFilterHolderI* DefSuccessor() const = 0;
	
	/* This is a list of pointers to AmFilterHolderI objects,
	 * which are immediate successors to this one in the pipeline.
	 * The first object in this list is always the same as
	 * DefSuccessor().  Usually there is only one successor; however,
	 * some filters may allow multiple successors, and they are
	 * available here.
	 */
	virtual const BList& Successors() const = 0;

	/* This is a convenience method that will refresh the display of
	 * all views on my filter.
	 */
	virtual void	FilterChangeNotice() = 0;

protected:
	/* You are not allowed to delete these yourself. */
	virtual ~AmFilterHolderI() { }
};

/*****************************************************************************
 *
 *	AM-FILTER-I CLASS
 *
 *	The AmFilterI class is the public interface to an instance
 *	of a filter add-on.
 *
 *****************************************************************************/

enum {
	AMFF_RECORDING = (1<<0),
	AMFF_STARTING = (1<<1)
};

struct am_filter_params {
	size_t					size;				// amount of data in structure
	uint32					flags;				// state flags
	bigtime_t				performance_time;	// for destination filter, exact
												// time to perform for all other
												// filters, zero.
	const AmTempoChange*	cur_tempo;			// last tempo event in song.
	const AmSignature*		cur_signature;		// last signature event in song.
	
	am_filter_params();
	am_filter_params(const am_filter_params& o);
	~am_filter_params();
	am_filter_params& operator=(const am_filter_params& o);

private:
	bool operator==(const am_filter_params& o) const;
	bool operator!=(const am_filter_params& o) const;
};

#define AM_SIZE_TO(STRUCT, FIELD)		\
		( ((size_t)&((STRUCT*)0)->FIELD) + sizeof(STRUCT::FIELD) )

class AmFilterI : public ArpConfigurableI
{
public:
	AmFilterI(AmFilterAddOn* addon);
	
	/* The MIDI system will delete this filter instance when it is
	 * no longer needed.
	 */
	virtual ~AmFilterI() { }

	/* Default ArpConfigurableI implementation.  Be sure to call the
	 * inherited form when overriding.
	 * NOTE: You should never call PutConfiguration() here.  Instead,
	 * call the filter's AmFilerHolderI::PutConfiguration(), which
	 * will send out a change notification to all objects watching
	 * the filter.
	 */
	virtual status_t GetConfiguration(BMessage* values) const;
	virtual status_t PutConfiguration(const BMessage* values);
	virtual status_t Configure(ArpVectorI<BView*>& panels);

	/* Answer a unique ID for this filter.
	 */
	filter_id		Id() const;
	/* Answer a name for this filter.  By default, answer the filter
	 * addon's name.  However, users can override this to supply their
	 * own name for the filter.
	 */
	const char*		Name() const;
	
	/* Every filter instance can optionally be labelled.  If no label
	 * is set, Label() returns the Name(); otherwise, the selected
	 * label is returned.  The filter's label is handled in the
	 * default implemention of GetConfiguration() and PutConfiguration().
	 */
	bool			HasLabel() const;
	const char*		Label() const;
	
	/* Return the addon class definition for the filter -- i.e., the
	 * object whose NewInstance() method created this object.
	 */
	AmFilterAddOn*	AddOn() const;
	
	/* Fill in BMessage with data describing this filter instance,
	 * which can be passed into NewInstance() to make a duplicate.
	 */
	status_t Archive(BMessage* into, uint32 flags) const;
	
	/* Fill in a configuration message that will set up one of
	 * your instances for the "next" in order from yourself.  Return
	 * an error if you are the last.
	 */
	virtual status_t GetNextConfiguration(BMessage* config) const;
	
	/* It might be convenient for a window to display a miniature
	 * control to quickly access some of a filter's functionality.
	 * Filters that want to take part in this can answer a BView with
	 * the relevant controls.
	 */
	virtual BView* NewEditView(BPoint requestedSize) const;

	/* Fill in the string with text you would like to display for
	 * your tool tip.  The default implementation simply fills in
	 * the filter add-on's Name().
	 */
	virtual status_t GetToolTipText(BString* out) const;
	
	/* By default, answer the addon's image.
	 */
	virtual BBitmap* Image(BPoint requestedSize) const;
	
	/* This is called when the user invokes an action on the
	 * filter.  Normally, this is B_SECONDARY_BUTTON to show the
	 * popup menu; 'released' is true if being called as a result
	 * of the button being released.
	 */
	virtual void	MouseAction(BRect frame, BPoint where,
								uint32 buttons, bool released);
	
	/* Return an instrument
	 */
	virtual ArpCRef<AmInstrumentI> Instrument() const;

	/* Return a hint about what midi channel this filter
	 * corresponds to.  Only really useful for output filters,
	 * and then currently only used for writing standard midi
	 * files.  The default implementation returns -1, indicating
	 * it has nothing to hint.
	 */
	virtual int32 HintChannel() const;
	
	/* These are called as the performance moves through the song,
	 * playing it in batches.  When the transport begins processing
	 * a range of time, StartSection() is called; when over,
	 * FinishSection() is called.  For both of these, you can return
	 * any events you want to generate during that range.
	 *
	 * Note that you should -not- expect to handle events whose time
	 * is only in the given range.  For many reasons, such as filters
	 * before you modifying event times, no such assumption can be
	 * made.
	 *
	 * The default implementation of these simply returns NULL.
	 */
	virtual AmEvent* StartSection(AmTime firstTime, AmTime lastTime,
								  const am_filter_params* params = NULL);
	virtual AmEvent* FinishSection(AmTime firstTime, AmTime lastTime,
								   const am_filter_params* params = NULL);
	
	/* This is called when the user presses the stop button on the
	 * transport.
	 */
	virtual void Stop();
	
	/* The main reason to exist: this function implements your
	 * actual filter algorithm.  The MIDI system will call it with
	 * an AmEvent object that is to be filtered; you may do
	 * whatever you want with it -- change it, delete it, copy it,
	 * add events before or after it -- and then return the resulting
	 * changed and/or new events.
	 * Note that if you return more than one event, they must be linked
	 * in time sorted order, with the earliest event positioned first
	 * and returned as the result.  These events will then be merged
	 * back into the sequence that is being filtered.
	 * The 'params' argument suppies additional performance information.
	 * It may be NULL.
	 */
	virtual AmEvent* HandleEvent(AmEvent* event,
								 const am_filter_params* params = NULL) = 0;
	
	/* This function is called when an event is being filtered that is
	 * not a part of the song.  These will occur outside of corresponding
	 * StartSection() and FinishSection() calls.
	 *
	 * The default implementation is to call HandleEvent().  You can
	 * override this version if you need to do something special with
	 * this situation.
	 */
	virtual AmEvent* InteractiveEvent(AmEvent* event,
									  const am_filter_params* params = NULL);
	 
	/* This function is called when a real-time event is moving through
	 * your filter.  (That is, one that came directly from an input
	 * device rather than an existing sequence.)  The semantics are
	 * slightly different in this case -- such events will appear out of
	 * sequence with other regular events, and you will see individual
	 * AmNoteOn (with duration of 0) and AmNoteOff events, rather than
	 * one AmNoteOn with a duration.
	 *
	 * The default implementation is to call HandleEvent().  You can
	 * override this version if you need to do something special with
	 * this situation.
	 */
	virtual AmEvent* RealtimeEvent(AmEvent* event,
								   const am_filter_params* params = NULL);
	
private:
	ArpRef<AmFilterAddOn>	mAddOn;
	BString					mLabel;
};

// interface
typedef AmFilterAddOn* (*make_nth_filter_type)(int32 n, image_id you,
												const void* cookie, uint32 flags, ...);
extern "C" _EXPORT AmFilterAddOn* make_nth_filter(int32 n, image_id you,
												  const void* cookie, uint32 flags, ...);

#endif
