// Copyright  1997  Ian Russell Ollmann.
// All rights reserved. 

#ifndef PARSER_HEADER
#define PARSER_HEADER

typedef short Boolean;

//
//  See class FunctionInterpreter below for proper use of this code.
//	Read the accompanying README file for a description of its operation.
//

enum ParseErrorType
{
//	These return no useful information in errorInfo;
	pDivisionByZero,
	pSqrtNegNumber,
	pLogNegNumber,
	pLnNegNumber,
	pOutOfMemory,
	pArcSinOutOfBounds,
	pArcCosOutOfBounds,
	
//	These return text offsets as the loWord and hiWord of their errorInfoField. 
//	These text offsets correspond to start and end of the error containing portion of ParseInfo.string
	pEmptyExpression,
	pMissingOperator,
	pSyntaxError,
	pUnbalancedParen,
	pUnknownFunction,
	pUnknownOperator
};

// This struct is passed when errors are thrown. ErrorType codes for the type of messages, and 
// for some errorTypes, the startOffset and endOffset fields denote text offsets relative to 
// ParseInfo->string which bound a region containing an error
typedef struct
{
	ParseErrorType	errorType;
	short			startOffset;
	short			endOffset;
}ParseErrorMsg;


// Each Operator below must correspond to a OperatorParser child class and 
// must be described in the Parser::FindOperator() and Parser::CreateNewOperator
enum OperatorType 
{
	noOperator = 0,			// (none)
	plusOperator = 1,		//	+
	minusOperator = 2,		//	-
	multiplyOperator = 3,	//	*
	divideOperator = 4,		//	/
	modulusOperator = 5,	//	%
	powerOperator = 6		//	^
};

//The following array sets the priority of each operator, whose array index is given above in the OperatorType definition
const short		kPriorityArray[7] = { 4, 1, 1, 2, 2, 2, 3 }; //zero is reserved for non-operators
const short		kMaxPriority = 5; //Should be one higher than the highest priority in kPriorityArray

enum OperatorUsage
{
	useAsAnOperator = 0,	//Most operators will take this role
	useAsASign = 1,			//Depending on their context, some + and - symbols should be interpreted as signs and not operators
	useAsAChar = 2			//If a "-" appears in the middle of a number in scientific notation, it is treated as a char and lumped in with 
							//the rest of the string segment
};

// All functions must be declared here as a UPPERCASE char[4] array, with \0 padding for unused chars.
// To add a new function, a new class inheriting from FunctionParser or Parser must be declared below
// which performs the desired mathematical operation in its Evaluate() method and parses any ParseInfo 
// input in its constructor. Lastly, an entry for the function must be made in Parser::FindFunction() 
// and Parser::CreateNewFunction()
enum FunctionType
{
	syntaxErr = '\0\0\0\0',	//It is possible that this value is no longer used
	noFunction = '\0\0\0\1',
	sinFunction = 'SIN\0',
	cosFunction = 'COS\0',
	tanFunction = 'TAN\0',
	arcSinFunction = 'ASIN',
	arcCosFunction = 'ACOS',
	arcTanFunction = 'ATAN',
	hyperbolicSinFunction = 'SINH',
	hyperbolicCosFunction = 'COSH',
	hyperbolicTanFunction = 'TANH',
	expFunction = 'EXP\0',
	lnFunction = 'LN\0\0',
	logFunction = 'LOG\0',
	absFunction = 'ABS\0',
	sqrtFunction = 'SQRT',
	negFunction = 'NEG\0',		
	randomFunction = 'RAND'
};

class FunctionInterpreter;
typedef struct 
{
	char 					*string;		//This is the string to be parsed, e.g. "3+4-(sin(x))". It does not have to be null terminated
	char					*segmentStart;	//This is a pointer to the char in the string that you want to start parsing at
	char					*segmentEnd;	//This is a pointer to the last char on the string that you want to be parsed
	FunctionInterpreter		*root; 			//This pointer is reserved for use by the Variable class. 		
}ParseInfo;


//This is the base class that describes the Parser behavior, used by all other classes
//in this hierarchy. It codes all the operations necessary to parse an expression, 
//determine what operation should be done next, and init a Parser derived object to
//do it. It has no member data of its own.
class Parser
{
	public:
		virtual double			Evaluate( void );
		virtual	Boolean			IsConstant( void );
	
	
		void					*operator new( size_t size );	//new is overloaded to throw a pOutOfMemoryError if allocation fails
		void					operator delete( void *parserPtr, size_t size);
								Parser( void );
								~Parser();
							
	protected:
		virtual Boolean		 	ConvertStringToValue( ParseInfo *parseInfo,  double *value);
		virtual void			CreateNewFunction( ParseInfo *parseInfo, FunctionType functionType, Parser **newFunction );
		virtual void	 		CreateNewOperator( ParseInfo *parseInfo, char *theOperator, OperatorUsage usage, Parser **newParser );
		virtual void			EvaluateConstant( ParseInfo *parseInfo, double *value, Parser **newParser );
		virtual FunctionType	FindFunction( ParseInfo *parseInfo );
		virtual char			*FindOperator( ParseInfo *parseInfo, OperatorUsage *usage );
		virtual OperatorUsage	OperatorUsageByContext( ParseInfo *parseInfo, char *operatorPtr );
		virtual void	 		Parse( ParseInfo *parseInfo, Parser **nextParser, double *value );
		virtual short			Priority( char *c, OperatorUsage operatorUsage );
		virtual void			ThrowError( ParseErrorType errorType, short startOffset = 0, short endOffset = 0 );
		virtual void			StripWhiteSpaceFromEnds( ParseInfo *parseInfo );
		virtual void 			TrimString( ParseInfo *parseInfo );
};

typedef struct
{
	char			*name;	//this is a pointer to a '\0' terminated char array which contains the name of the variable taken from the string provided to FunctionInterpreter's constructor
	double			value;	//The value of the variable
}ParserVariable;


// The FunctionInterpreter class should be the one you construct in your program to parse your string.
// The use of this class is as follows:
//
// Construct a new FunctionInterpreter object, passing a pointer to a ParseInfo 
// struct which lists a pointer to the string you want to parse, and start and end 
// pointers to the portion of the string that is to be parsed. You are responsible 
// for deallocating string storage after the FunctionInterpreter constructor is done.
// The FunctionInterpreter and related objects will not change the string. The string 
// does not have to be '\0' terminated.
//
// Use GetNumberOfVariables() to determine if there are any variables
//
// If there are variables, use GetVariableName() to determine what the variable is called and
// and SetVariableValue() to set its value. Variables are indexed from 0 to (numberOfVariables-1)
//
// Last, use Evaluate() to solve the equation.  This operation should be fast. 
// 
// If you want to solve a new equation, delete this FunctionInterpreter and construct another.
// If you want to change variable values, just use SetVariableValue() and the call Evaluate() 
// again.
//
//
// !!! IMPORTANT: ERROR HANDLING !!!
// The FunctionInterpreter constructor will throw exceptions in response to syntax errors
// so it must be used in the bounds of a try{} field. In addition, both the constructor and 
// Evaluate() will throw calculation related errors such as division by zero, so Evaluate() 
// must also be used inside a try{} field. The errors will all be of type ParseErrorMsg. 
// If you do not catch{} these errors, your program with terminate.
//
class FunctionInterpreter : Parser
{
	public:
	//These functions are reserved for use by internal Variable objects. 
		virtual int				AddVariable( ParseInfo *parseInfo );			// Returns the index of the variable					
		virtual double			GetVariableValue( int index );					// Returns the value of the variable at index index

	//Use these in your program to handle variables. Variables are indexed from 0 to #Variables - 1 
		virtual int				GetNumberOfVariables( void );					// Returns the number of variables. 
		virtual char*			GetVariableName( int index ); 						// Returns a handle to a '\0' terminated char array corresponding to the varaible of index index, or NULL if the index is out of bounds
		virtual Boolean			SetVariableValue( double newValue, int index );	// Returns false if index is out of bounds

	//Call Evaluate() to solve the equation and return its value. It should remain commented out.
		virtual double			Evaluate( void );

								FunctionInterpreter(  ParseInfo *parseInfo );
								~FunctionInterpreter();
								
	
	protected:
		Parser					*calculationStart;
		double					value;
		ParserVariable 			*variableList;
		int						variableCount;

		virtual void 			AddVariableToList( ParseInfo *parseInfo );

};

// Function parser is a child of parser which takes a single expression as an argument
// {e.g. sin(expr) or abs(expr) } It is not meant to be called in this form, but rather 
// as a child class, for which Evaluate() is defined to do some operation. Value stores 
// the numerical value of the function argument, if constant. Otherwise call Evaluate()
// using the next pointer before letting Evaluate return an answer vased on value.
class FunctionParser : public Parser
{
	public:
		virtual double		Evaluate( void );
		virtual	Boolean		IsConstant( void );

							FunctionParser( ParseInfo *parseInfo );
							~FunctionParser();

	protected:
		double				value;
		Parser				*next;

};

// OperationParse is a base class for operators "+-*/%^" which take two arguments, the 
// string to the left of the operator and the string to the right of the operator. 
class OperationParser : public Parser
{
	public:
		virtual double		Evaluate( void );
		virtual	Boolean		IsConstant( void );

							OperationParser( ParseInfo *leftInfo, ParseInfo *rightInfo );
							~OperationParser();

	protected:
		double				leftValue, rightValue;
		Parser				*left, *right;
							
};

//Variable is a place marker in the object tree for a variable. It does not store the
//actual variable value or name. FunctionInterpreter does. This is so to make sure that 
//multiple calls to the same variable return the same value. 
class Variable : public Parser
{
	public:
		virtual double		Evaluate( void );
		virtual	Boolean		IsConstant( void );
		
							Variable( ParseInfo *parseInfo );
							
	protected:
		FunctionInterpreter *root;
		int					index;
};




//Children of Operation Parser
class PlusOperator : public OperationParser // does addition in its Evaluate() function
{
	public:
		virtual double		Evaluate( void );

							PlusOperator( ParseInfo *leftInfo, ParseInfo *rightInfo );
};

class MinusOperator : public OperationParser // Handles subtraction
{
	public:
		virtual double		Evaluate( void );

							MinusOperator( ParseInfo *leftInfo, ParseInfo *rightInfo );
};

class MultiplyOperator : public OperationParser // Handles multplication
{
	public:
		virtual double		Evaluate( void );

							MultiplyOperator( ParseInfo *leftInfo, ParseInfo *rightInfo );
};

class DivideOperator : public OperationParser //Handles division
{
	public:
		virtual double		Evaluate( void );

							DivideOperator( ParseInfo *leftInfo, ParseInfo *rightInfo );
};

class ModulusOperator : public OperationParser //Handles Modulus operations
{
	public:
		virtual double		Evaluate( void );

							ModulusOperator( ParseInfo *leftInfo, ParseInfo *rightInfo );
};

class PowerOperator : public OperationParser //Handles x^y operations
{
	public:
		virtual double		Evaluate( void );

							PowerOperator( ParseInfo *leftInfo, ParseInfo *rightInfo );
};


//Functions
class SinFunction : public FunctionParser // sin(expr)
{
	public:
		virtual double		Evaluate( void );

							SinFunction(  ParseInfo *parseInfo );
};

class CosFunction : public FunctionParser // cos(expr)
{
	public:
		virtual double		Evaluate( void );

							CosFunction(  ParseInfo *parseInfo );
};


class TanFunction : public FunctionParser //tan(expr)
{
	public:
		virtual double		Evaluate( void );

							TanFunction( ParseInfo *parseInfo );
};

class ExpFunction : public FunctionParser //e^expr
{
	public:
		virtual double		Evaluate( void );

							ExpFunction(  ParseInfo *parseInfo);
};

class LnFunction : public FunctionParser //natural logarythm(expr). throws an error if expr < 0
{
	public:
		virtual double		Evaluate( void );

							LnFunction(  ParseInfo *parseInfo );
};

class LogFunction : public FunctionParser // log base 10 (expr). throws an error if expr <= 0
{
	public:
		virtual double		Evaluate( void );

							LogFunction(  ParseInfo *parseInfo );
};

class SqrtFunction : public FunctionParser // square root(expr). throws an error if expr < 0
{
	public:
		virtual double		Evaluate( void );

							SqrtFunction(  ParseInfo *parseInfo);
};

class AbsFunction : public FunctionParser // absolute value
{
	public:
		virtual double		Evaluate( void );

							AbsFunction(  ParseInfo *parseInfo);
};

class NegFunction : public FunctionParser// inverts the sign of its argument
{
	public:
		virtual double		Evaluate( void );

							NegFunction(  ParseInfo *parseInfo);
};

class ArcSinFunction : public FunctionParser // ArcSin(expr) returns an error if abs(expr) > 1
{
	public:
		virtual double		Evaluate( void );

							ArcSinFunction(  ParseInfo *parseInfo);
};

class ArcCosFunction : public FunctionParser// ArcCos(expr) returns an error if abs(expr) > 1
{
	public:
		virtual double		Evaluate( void );

							ArcCosFunction(  ParseInfo *parseInfo);
};

class ArcTanFunction : public FunctionParser//ArcTan(expr)
{
	public:
		virtual double		Evaluate( void );

							ArcTanFunction(  ParseInfo *parseInfo);
};

class HSinFunction : public FunctionParser // Hyperbolic Sine (expr)
{
	public:
		virtual double		Evaluate( void );

							HSinFunction(  ParseInfo *parseInfo);
};

class HCosFunction : public FunctionParser // Hyperbolic Cosine (expr)
{
	public:
		virtual double		Evaluate( void );

							HCosFunction(  ParseInfo *parseInfo);
};

class HTanFunction : public FunctionParser// Hyperbolic Tangemt (expr)
{
	public:
		virtual double		Evaluate( void );

							HTanFunction(  ParseInfo *parseInfo);
};

//Rand(expr) returns a random number between 0 and expr. If there is no expr then it returns a random number between 0 and 1 
class RandomFunction : public Parser {
	public:
		virtual double		Evaluate( void );
		virtual	Boolean		IsConstant( void );

							RandomFunction(  ParseInfo *parseInfo);
	protected:
		Parser				*next;
		double 				value;
};

#endif