/*
 *  main.cpp - Main program
 *
 *  MIDIPlayer (C) 1997 Christian Bauer
 */

#include <AppKit.h>
#include <InterfaceKit.h>
#include <MidiKit.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


// Constants
const char APP_SIGNATURE[] = "application/x-vnd.cebix-MIDIPlayer";

const rgb_color light_color = {255, 255, 255, 0};
const rgb_color fill_color = {216, 216, 216, 0};
const rgb_color dark_color = {184, 184, 184, 0};

const uint32 MSG_PLAY_PAUSE = 'plpa';
const uint32 MSG_STOP = 'stop';
const uint32 MSG_RESET = 'rset';
const uint32 MSG_LOAD_MIDI = 'load';
const uint32 MSG_PULSE = 'puls';


// Application object
class MIDIPlayer : public BApplication {
public:
	MIDIPlayer();
	virtual void ArgvReceived(int32 argc, char **argv);
	virtual void RefsReceived(BMessage *msg);
	virtual void ReadyToRun(void);
	virtual void AboutRequested(void);
};


// Main window object
class MainWindow : public BWindow {
public:
	MainWindow();
	virtual bool QuitRequested(void);
	virtual void MessageReceived(BMessage *msg);

private:
	BStringView *make_name_display(BRect frame, char *label_text, BView *parent);
	bool load_midi(entry_ref *file);
	void reset_midi(void);

	BMidiPort *the_port;	// Pointer to MIDI port
	BMidiStore *the_store;	// Pointer to MIDI Store

	char file_name[B_FILE_NAME_LENGTH];		// MIDI file name

	BStatusBar *position_view;
	uint32 num_events;
};


// Top view object (handles drag&drop)
class TopView : public BView {
public:
	TopView(BRect frame, const char *name, uint32 resizingMode, uint32 flags);
	virtual void MessageReceived(BMessage *msg);
	virtual void KeyDown(const char *bytes, int32 numBytes);
	virtual void Pulse(void);
};


// Buttons
class PlayPauseButton : public BButton {
public:
	PlayPauseButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class StopButton : public BButton {
public:
	StopButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};


// Global: entry_ref of currently loaded MIDI file
entry_ref FileRef;


/*
 *  Create application object and start it
 */

int main(int argc, char **argv)
{	
	MIDIPlayer *the_app = new MIDIPlayer();
	the_app->Run();
	delete the_app;
	return 0;
}


/*
 *  Constructor
 */

MIDIPlayer::MIDIPlayer() : BApplication(APP_SIGNATURE) {}


/*
 *  Shell arguments received
 */

void MIDIPlayer::ArgvReceived(int32 argc, char **argv)
{
	if (argc == 2) {
		BEntry entry(argv[1], true);
		entry.GetRef(&FileRef);
	} else
		printf("Usage: %s [name of MIDI file]\n", argv[0]);
}


/*
 *  Browser arguments received
 */

void MIDIPlayer::RefsReceived(BMessage *msg)
{
	msg->FindRef("refs", &FileRef);
}


/*
 *  Arguments processed, open player window
 */

void MIDIPlayer::ReadyToRun(void)
{
	new MainWindow();
}


/*
 *  About requested
 */

void MIDIPlayer::AboutRequested(void)
{
	BAlert *the_alert = new BAlert("",
		"MIDIPlayer V1.0 by Christian Bauer\n<cbauer@iphcip1.physik.uni-mainz.de>\n"
		"\nFreely distributable.", "OK");
	the_alert->Go();
}


/*
 *  Window constructor
 */

MainWindow::MainWindow() : BWindow(BRect(0, 0, 249, 64), "MIDIPlayer", B_TITLED_WINDOW, B_NOT_ZOOMABLE | B_NOT_RESIZABLE)
{
	Lock();
	BRect b = Bounds();

	// Create and connect MIDI objects
	the_port = new BMidiPort("midi1");
	the_port->AllNotesOff();
	the_store = new BMidiStore();
	the_store->Connect(the_port);

	// Move window to right position
	MoveTo(80, 80);

	// Light gray background
	TopView *top = new TopView(BRect(0, 0, b.right, b.bottom), "main", B_FOLLOW_NONE, B_WILL_DRAW | B_PULSE_NEEDED);
	AddChild(top);
	top->SetViewColor(fill_color);

	// Position display
	position_view = new BStatusBar(BRect(8, 2, 241, 21), "position", "Welcome to MIDIPlayer");
	top->AddChild(position_view);
	SetPulseRate(1000000);

	// Buttons
	top->AddChild(new PlayPauseButton(BRect(6, 36, 46, 60), new BMessage(MSG_PLAY_PAUSE)));
	top->AddChild(new StopButton(BRect(47, 36, 87, 60), new BMessage(MSG_STOP)));
	top->AddChild(new BButton(BRect(88, 36, 128, 60), NULL, "R", new BMessage(MSG_RESET)));

	// Show window
	top->MakeFocus();
	Show();
	Unlock();

	// Load MIDI file
	PostMessage(MSG_LOAD_MIDI);
}


/*
 *  Quit requested, close all and quit program
 */

bool MainWindow::QuitRequested(void)
{
	// Delete MIDI objects
	the_store->Disconnect(the_port);
	delete the_port;
	delete the_store;

	be_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}


/*
 *  Message received
 */

void MainWindow::MessageReceived(BMessage *msg)
{
	switch (msg->what) {

		case MSG_PLAY_PAUSE:
			if (the_store->IsRunning())
				the_store->Stop();
			else
				the_store->Start();
			break;

		case MSG_STOP:
			the_store->Stop();
			while (the_store->IsRunning()) ;
			the_port->AllNotesOff();
			break;

		case MSG_RESET: {
			the_store->Stop();
			while (the_store->IsRunning()) ;
			reset_midi();
			break;
		}

		case MSG_LOAD_MIDI:
			the_store->Stop();
			the_store->Disconnect(the_port);
			delete the_store;
			the_store = new BMidiStore();
			the_store->Connect(the_port);
			reset_midi();

			if (load_midi(&FileRef)) {
				num_events = 0;
				Lock();
				position_view->Reset(file_name);
				position_view->SetMaxValue(the_store->CountEvents());
				Unlock();
				the_store->Start();
			}
			break;

		case MSG_PULSE: {
			uint32 events = the_store->CurrentEvent();
			Lock();
			position_view->Update(events - num_events);
			Unlock();
			num_events = events;
			break;
		}

		default:
			BWindow::MessageReceived(msg);
			break;
	}
}


/*
 *  Load MIDI file
 */

bool MainWindow::load_midi(entry_ref *file)
{
	if (the_store->Import(file) == B_NO_ERROR) {
		BEntry entry(file);
		entry.GetName(file_name);
		return true;
	} else
		return false;
}


/*
 *  Reset MIDI system
 */

void MainWindow::reset_midi(void)
{
	the_port->AllNotesOff();
	uint8 data[] = {0x7e, 0x7f, 0x09, 0x01};	// GM Mode On
	the_port->SystemExclusive(data, 4);
	snooze(100000);
	for (int i=0; i<16; i++) {
		the_port->ControlChange(i, 120, 0);		// All Sound Off
		the_port->ControlChange(i, 121, 0);		// Reset All Controllers
	}
	snooze(50000);
}


/*
 *  TopView handles dropped messages (load new PSID module) and keypresses
 */

TopView::TopView(BRect frame, const char *name, uint32 resizingMode, uint32 flags)
 : BView(frame, name, resizingMode, flags) {};

void TopView::MessageReceived(BMessage *msg)
{
	if (msg->what == B_SIMPLE_DATA) {
		if (msg->FindRef("refs", &FileRef) == B_NO_ERROR)
			Window()->PostMessage(MSG_LOAD_MIDI);
	} else
		BView::MessageReceived(msg);
}

void TopView::KeyDown(const char *bytes, int32 numBytes)
{
	switch (bytes[0]) {
		case 'p':
		case 'P':
			Window()->PostMessage(MSG_PLAY_PAUSE);
			break;

		case B_ESCAPE:
		case B_SPACE:
		case 's':
		case 'S':
			Window()->PostMessage(MSG_STOP);
			break;

		case 'r':
		case 'R':
			Window()->PostMessage(MSG_RESET);
			break;
	}
}


/*
 *  Periodic action: Update position view
 */

void TopView::Pulse(void)
{
	Window()->PostMessage(MSG_PULSE);
}


/*
 *  Play/Pause button
 */

PlayPauseButton::PlayPauseButton(BRect frame, BMessage *msg) : BButton(frame, "play", "", msg) {};

void PlayPauseButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw play/pause image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(16, 8, 18, 16));
	FillTriangle(BPoint(21, 8), BPoint(21, 16), BPoint(25, 12));
}


/*
 *  Stop button
 */

StopButton::StopButton(BRect frame, BMessage *msg) : BButton(frame, "stop", "", msg) {};

void StopButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw stop image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(16, 8, 25, 16));
}
