/*!
 * [BUG] I get no sound from my second CD player.
 *
 * This also happens to Be's CDPlayer, so I guess it's not my fault.
 */

/*!
 * [NOTE] Large Deskbar replicant is put in the wrong position.
 *
 * In an earlier version of Hustler, my replicant used to be about 60 pixels
 * wide. When it was first added to the Deskbar, it would be drawn about 50
 * pixels to the left of the time replicant. The space between these two
 * replicants was way bigger than the space that Deskbar normally puts 
 * between the time replicant and other replicants. The strange thing is,
 * after resizing the Deskbar and back again, my replicant and the time
 * replicant were drawn neatly next to each other. 
 * 
 * Nowadays Hustler is much smaller (about 30 pixels) and always appears 
 * right next to the time replicant. So I guess that replicants wider than 16
 * pixels aren't really supported, or there is a resizing bug in the 
 * Deskbar's tray.
 */

// ***************************************************************************
// Implementation System Includes
// ***************************************************************************

#include <Alert.h>
#include <AppFileInfo.h>
#include <File.h>
#include <MenuItem.h>
#include <PopUpMenu.h>
#include <stdio.h>        // sprintf()

// ***************************************************************************
// Implementation Project Includes
// ***************************************************************************

#include "Hustler.h"
#include "HustlerApp.h"
#include "HustlerMain.h"
#include "Version.h"

// ***************************************************************************
// Implementation Constant Definitions
// ***************************************************************************

const rgb_color COLOR_BACKGROUND = {40, 80, 40};
const rgb_color COLOR_FOREGROUND = {140, 220, 140};

// ***************************************************************************
// Implementation Type Definitions
// ***************************************************************************

// ***************************************************************************
// Implementation Variable Definitions
// ***************************************************************************

// ***************************************************************************
// Implementation Function Declarations
// ***************************************************************************

// ***************************************************************************
// Global Variable Definitions
// ***************************************************************************

// ***************************************************************************
// PUBLIC Member Function Definitions: Lifecycle
// ***************************************************************************

// ===========================================================================
// Constructor
// ===========================================================================

tHustler::tHustler()
  : BView(BRect(0,0,1,1), NULL, 0, 0)
{
  BBG_WRITE(("[tHustler::tHustler] Constructed."));
  
  // Initialize attributes.
  mpCdEngine = NULL;
  mpIcon = NULL;
  
  // Get the application's mini icon (16x16 pixels, 8-bit color) and put it
  // in our bitmap object, so we can archive it later on (in Archive()) and
  // send it to the Deskbar as part of our replicant.
  app_info lAppInfo; 
  if (be_app->GetAppInfo(&lAppInfo) == B_OK) {

    BFile lFile;
    if (lFile.SetTo(&lAppInfo.ref, B_READ_WRITE) == B_OK) {

      BAppFileInfo lAppFileInfo;
      if (lAppFileInfo.SetTo(&lFile) == B_OK) {

        mpIcon = new BBitmap(BRect(0,0,15,15), B_COLOR_8_BIT);
        lAppFileInfo.GetIcon(mpIcon, B_MINI_ICON);
      } 
    }
  }  
}

// ===========================================================================
// Constructor
// ===========================================================================

tHustler::tHustler(BMessage* apMessage)
  : BView(BRect(0,0,15,15),
          "Hustler",
          B_FOLLOW_NONE,
          B_WILL_DRAW | B_PULSE_NEEDED) 
{
  BBG_WRITE(("[tHustler::tHustler] Constructed."));

  // Initialize attributes.
  mpCdEngine = NULL;
  mDeviceIndex = 0;
  mpIcon = NULL;
  mBlink = FALSE;
  mOldTrack = 0;
  mOldState = tCdEngine::STATE_NO_DISC;

  // Get the icon from the message.
  BMessage lBitmapArchive;
  apMessage->FindMessage("Icon", &lBitmapArchive);
  mpIcon = new BBitmap(&lBitmapArchive);

  // Calculate the maximum width for our replicant. Note that the Deskbar
  // accepts replicants that are wider than 16 pixels, but not replicants
  // that are higher than 16 pixels.
  tFloat lWidth = StringWidth("99") + 4 + 16;
  ResizeTo(lWidth,15);
  
  // Initialize the CD player.
  mpCdEngine = new tCdEngine();
  if (mpCdEngine->GetNumberOfDevices() > 0) {
    mpCdEngine->OpenDevice(mDeviceIndex);
  }

  // Set our view's background color.
  SetViewColor(COLOR_BACKGROUND);
}

// ===========================================================================
// Destructor
// ===========================================================================

tHustler::~tHustler()
{
  BBG_WRITE(("[tHustler::~tHustler] Destructed."));

  // Although we haven't created a CD engine when the application deletes us
  // (the CD engine is only created when we are running as a replicant), we
  // can safely call "delete mpCdEngine" here, because delete simply ignores
  // null pointers.
  delete mpCdEngine;
  delete mpIcon;
}

// ***************************************************************************
// PUBLIC Member Function Definitions: Operators
// ***************************************************************************

// ***************************************************************************
// PUBLIC Member Function Definitions: Operations
// ***************************************************************************

// ===========================================================================
// Instantiate
// ===========================================================================

__declspec(dllexport) tHustler* 
tHustler::Instantiate(BMessage* apArchive)
{
  if (!validate_instantiation(apArchive, "tHustler")) {
    return NULL;
  }
  return new tHustler(apArchive);
}

// ===========================================================================
// Archive
// ===========================================================================

status_t 
tHustler::Archive(BMessage* apArchive, bool aDeep) const
{
  // Tell the overridden BView to archive itself.
  BView::Archive(apArchive, aDeep);
  
  // Store our signature and the name of our class in the archive.
  apArchive->AddString("add_on", APP_SIGNATURE);
  apArchive->AddString("class", "tHustler");

  // Also add our icon to the message.  
  if (aDeep) {
    BMessage lBitmapArchive;
    if (mpIcon->Archive(&lBitmapArchive, aDeep) == B_OK) {
      apArchive->AddMessage("Icon", &lBitmapArchive);
    }
  }
  return B_OK;
}  

// ===========================================================================
// Draw
// ===========================================================================

void 
tHustler::Draw(BRect aUpdateRect)
{
  // Draw the icon. We use drawing mode B_OP_OVER, so transparent pixels are
  // really being treated as transparent.
  SetDrawingMode(B_OP_OVER);
  if (mpIcon) {
    DrawBitmap(mpIcon, BPoint(0,0));
  }

  // Only draw the track number if we are not blinking.
  if (mBlink == FALSE) {

    // Create the string.
    tChar lString[5];
    switch (mpCdEngine->GetState()) {
  
      case tCdEngine::STATE_NO_DISC: {
        sprintf(lString,"?");
        break;
      }
    
      case tCdEngine::STATE_PLAYING:
        // Fall through.
      
      case tCdEngine::STATE_PAUSED: {
        sprintf(lString,"%d",mpCdEngine->GetCurrentTrack());
        break;
      }
    
      case tCdEngine::STATE_STOPPED: {
        sprintf(lString,"-");
        break;
      }    
    }

    // Center the string horizontally and vertically.
    font_height lHeight;
    GetFontHeight(&lHeight);
    BRect lRect = Bounds();
    tFloat lWidth = StringWidth(lString);
    tFloat lX = 16 + (lRect.Width() - lWidth - 16)/2;
    tFloat lY = lHeight.ascent
              + (lRect.Height() - (lHeight.ascent + lHeight.descent))/2;

    // Draw the string. 
    SetLowColor(COLOR_BACKGROUND);
    SetHighColor(COLOR_FOREGROUND);
    SetDrawingMode(B_OP_OVER);
    DrawString(lString,BPoint(lX,lY));
  }
}

// ===========================================================================
// MouseDown
// ===========================================================================

void
tHustler::MouseDown(BPoint aPoint) 
{
  /*!
   * [NOTE] The pop-up menu is created on-the-fly.
   *
   * I create the pop-up menu on-the-fly because that makes it easier to deal
   * with the "Go to" submenu and the CD player's state. I did some testing, 
   * but building the pop-up menu once and re-using it is not more responsive
   * than doing it on-the-fly, so I guess it's cool to do it like this.
   */
   
  // Get the CD player's state.
  tInt32 lCdState = mpCdEngine->GetState();

  // Get the current CD's table of contents.
  mpCdEngine->GetContents();

  // Create a pop-up menu object that is not in radio mode.
  BPopUpMenu* lpMenu = new BPopUpMenu("Hustler",false,false);
  BMenuItem* lpItem; 

  // "Play" item. ------------------------------------------------------------

  lpItem = new BMenuItem("Play", new BMessage(MSG_MENU_PLAY));
  if ((lCdState == tCdEngine::STATE_NO_DISC) 
  ||  (lCdState == tCdEngine::STATE_PLAYING)
  ||  ((mpCdEngine->GetNumberOfTracks() ==1) 
      && (mpCdEngine->IsDataTrack(0)))) {
    lpItem->SetEnabled(false);
  } else {
    lpItem->SetEnabled(true);
  }
  lpMenu->AddItem(lpItem);

  // "Pause" item. -----------------------------------------------------------

  lpItem = new BMenuItem("Pause", new BMessage(MSG_MENU_PAUSE));
  if ((lCdState == tCdEngine::STATE_NO_DISC) 
  ||  (lCdState == tCdEngine::STATE_PAUSED)
  ||  (lCdState == tCdEngine::STATE_STOPPED)) {
    lpItem->SetEnabled(false);
  } else {
    lpItem->SetEnabled(true);
  }
  lpMenu->AddItem(lpItem);
  
  // "Stop" item. ------------------------------------------------------------

  /*!
   * [BUG] Stop is disabled because the CD driver doesn't stop correctly.
   *
   * When the CD engine sends a B_SCSI_STOP_AUDIO command to a CD player
   * device, it goes into a sort-of not-working pause mode instead of
   * stopping the CD. I believe that this is a problem with the CD driver,
   * because Be's CDPlayer application also does this.
   *
   * This bug also affects switching between devices: I cannot stop the old
   * device, so both keep playing.
   * 
   * @date Tuesday, 16 March 1999
   */
  
  //lpItem = new BMenuItem("Stop", new BMessage(MSG_MENU_STOP));
  //if ((lCdState == tCdEngine::STATE_NO_DISC)
  //||  (lCdState == tCdEngine::STATE_STOPPED)) {
  //  lpItem->SetEnabled(false);
  //} else {
  //  lpItem->SetEnabled(true);
  //}
  //lpMenu->AddItem(lpItem);
  
  // Separator bar. ----------------------------------------------------------

  lpMenu->AddSeparatorItem();

  // "Previous" item. --------------------------------------------------------
  
  lpItem = new BMenuItem("Previous", new BMessage(MSG_MENU_PREVIOUS));
  if ((lCdState == tCdEngine::STATE_NO_DISC)
  ||  (mpCdEngine->GetCurrentTrack() < 2)
  ||  ((mpCdEngine->GetCurrentTrack() == 2) 
      && (mpCdEngine->IsDataTrack(0)))) {
    lpItem->SetEnabled(false);
  } else {
    lpItem->SetEnabled(true);
  }
  lpMenu->AddItem(lpItem);

  // "Next" item. ------------------------------------------------------------
  
  lpItem = new BMenuItem("Next", new BMessage(MSG_MENU_NEXT));
  if ((lCdState == tCdEngine::STATE_NO_DISC)
  ||  (mpCdEngine->GetCurrentTrack() >= mpCdEngine->GetNumberOfTracks())
  ||  ((mpCdEngine->GetCurrentTrack() == 0) 
      && (mpCdEngine->IsDataTrack(0)))) {
    lpItem->SetEnabled(false);
  } else {
    lpItem->SetEnabled(true);
  }
  lpMenu->AddItem(lpItem);
  
  // "Go to" submenu. --------------------------------------------------------
  
  // This menu is in "radio mode", so only one item at a time can be selected.
  BMenu* lpGoToMenu = new BMenu("Go to");
  lpGoToMenu->SetRadioMode(true);

  if (lCdState == tCdEngine::STATE_NO_DISC) {

    lpItem = new BMenuItem("no disc", NULL);
    lpGoToMenu->AddItem(lpItem);
    lpGoToMenu->SetEnabled(false);

  } else {
  
    // Loop through the tracks in the CD's table of contents.
    for (tInt8 t=0; t<mpCdEngine->GetNumberOfTracks(); ++t) {

      // Get the track number.
      tInt8 lTrackNum = mpCdEngine->GetTrackNumber(t);
      
      // Create the message.
      BMessage* lpMessage = new BMessage(MSG_MENU_GO_TO);
      lpMessage->AddInt8("Track", lTrackNum);

      // Create the menu item.    
      tChar lString[8];
      sprintf(lString, "%d", lTrackNum);
      lpItem = new BMenuItem(lString, lpMessage);
      lpGoToMenu->AddItem(lpItem);

      // If this track is currently playing, then mark it.
      if (mpCdEngine->GetCurrentTrack() == lTrackNum) {
        lpItem->SetMarked(true);
      }
      
      // If this is a data track, then disable it.
      if (mpCdEngine->IsDataTrack(t)) {
        lpItem->SetEnabled(false);
      }
    }
    lpGoToMenu->SetEnabled(true);
    lpGoToMenu->SetTargetForItems(this);
  }
  lpMenu->AddItem(lpGoToMenu);

  // Separator bar. ----------------------------------------------------------

  lpMenu->AddSeparatorItem();

  // "Eject" item. -----------------------------------------------------------

  /*!
   * [TODO] Make the "Eject" menu item smarter.
   *
   * When a CD is loaded, the menu should have an "Eject" item. When the tray
   * is opened, "Eject" should be replaced by "Load". When the tray is closed
   * but empty, the item should also read "Eject".
   *
   * Using the current code I can fulfill the first two requirements, but not
   * the last. One of the newsletter articles talks about doing this, but I 
   * believe that this functionality is only available for IDE drives under 
   * R4 and not SCSI. I might want to check this out again...
   *
   * So until this is also supported on SCSI, I will only provide an "Eject"
   * item when a CD is loaded, and disable it otherwise. This means that I 
   * won't change "Eject" into "Load" when the tray is empty, because that
   * would also make "Load" appear when the tray is closed without a CD, and
   * the B_LOAD_MEDIA ioctl doesn't do anything in that case.
   */

  lpItem = new BMenuItem("Eject", new BMessage(MSG_MENU_EJECT));
  if (lCdState == tCdEngine::STATE_NO_DISC) {
    lpItem->SetEnabled(FALSE);
  } else {
    lpItem->SetEnabled(TRUE);
  }
  lpMenu->AddItem(lpItem);

  // Separator bar. ----------------------------------------------------------

  lpMenu->AddSeparatorItem();

  // "Options" submenu. ------------------------------------------------------

  /*!
   * [TODO] Add "Options" submenu.
   *
   *  Possible options are continous play and random play (shuffle).
   */
  
  //BMenu* lpOptionsMenu = new BMenu("Options");
  //
  //lpItem = new BMenuItem("Continous play", NULL);
  //lpOptionsMenu->AddItem(lpItem);
  //
  //lpItem = new BMenuItem("Random play", NULL);
  //lpOptionsMenu->AddItem(lpItem);
  //
  //lpMenu->AddItem(lpOptionsMenu);
  
  // "Time" submenu. ---------------------------------------------------------

  /*!
   * [TODO] Display time information.
   *
   * Right now, time information isn't available because I don't know how to
   * retrieve this information from the CD driver (it won't fit into a small
   * Deskbar replicant anyway). But if I decide to add a time display in the
   * future, the pop-up menu will also have a submenu where users can select
   * the various ways to display the time.
   */
  
  //BMenu* lpTimeMenu = new BMenu("Time");
  //lpTimeMenu->SetRadioMode(true);
  //
  //lpItem = new BMenuItem("Track elapsed", NULL);
  //lpTimeMenu->AddItem(lpItem);
  //
  //lpItem = new BMenuItem("Track remaining", NULL);
  //lpItem->SetMarked(true);
  //lpTimeMenu->AddItem(lpItem);
  //
  //lpItem = new BMenuItem("Disc elapsed", NULL);
  //lpTimeMenu->AddItem(lpItem);
  //
  //lpItem = new BMenuItem("Disc remaining", NULL);
  //lpTimeMenu->AddItem(lpItem);
  //
  //lpMenu->AddItem(lpTimeMenu);

  // "Device" submenu. -------------------------------------------------------
  
  BMenu* lpDeviceMenu = new BMenu("Device");
  lpDeviceMenu->SetRadioMode(true);

  if (mpCdEngine->GetNumberOfDevices() > 0) {

    for (tInt8 t=0; t<mpCdEngine->GetNumberOfDevices(); ++t) {

      // Create the message.
      BMessage* lpMessage = new BMessage(MSG_MENU_DEVICE);
      lpMessage->AddInt8("Device", t);

      // Create the menu item.
      lpItem = new BMenuItem(mpCdEngine->GetDeviceName(t), lpMessage);
      lpDeviceMenu->AddItem(lpItem);
    
      // If this is the current device, then mark it.
      if (mDeviceIndex == t) {
        lpItem->SetMarked(true);
      }
    }
    
    lpDeviceMenu->SetTargetForItems(this);
    
  // No CD players found.
  } else {
    lpItem = new BMenuItem("no CD players", NULL);
    lpDeviceMenu->AddItem(lpItem);
    lpDeviceMenu->SetEnabled(false);
  }

  lpMenu->AddItem(lpDeviceMenu);

  // Separator bar. ----------------------------------------------------------

  lpMenu->AddSeparatorItem();

  // "About" item. -----------------------------------------------------------

  lpItem = new BMenuItem("About" B_UTF8_ELLIPSIS, 
                          new BMessage(B_ABOUT_REQUESTED));
  lpMenu->AddItem(lpItem);

  // "Quit" item. ------------------------------------------------------------

  /*!
   * [NOTE] Quitting Hustler is not implemented.
   *
   * The reason for this is that quitting a Deskbar replicant doesn't really
   * make sense, and installing a new version of the replicant requires a
   * restart of the Deskbar anyway.
   *
   * @see The other remark about removing replicants from the Deskbar.
   */
   
  //lpItem = new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED));
  //lpMenu->AddItem(lpItem);

  // -------------------------------------------------------------------------

  // The messages of the items should be sent to ourselves.
  lpMenu->SetTargetForItems(this);

  // Pop up the menu and wait for the user to make a selection.
  ConvertToScreen(&aPoint); 
  lpMenu->Go(aPoint, true, false, true); 

  // Delete the things we don't need anymore.
  delete lpMenu;
}

// ===========================================================================
// MessageReceived
// ===========================================================================

void
tHustler::MessageReceived(BMessage* apMessage)
{
  // Figure out what kind of message was received.
  switch(apMessage->what) {

    // "Play" item selected. -------------------------------------------------
    case MSG_MENU_PLAY : {
      if (mpCdEngine->GetState() == tCdEngine::STATE_PAUSED) {
        mpCdEngine->Resume();
      } else {
        mpCdEngine->Play();
      }
      mBlink = FALSE;
      break;
    }

    // "Pause" item selected. ------------------------------------------------
    case MSG_MENU_PAUSE : {
      mpCdEngine->Pause();
      break;
    }

    // "Stop" item selected. -------------------------------------------------
    case MSG_MENU_STOP : {
      mpCdEngine->Stop();
      break;
    }

    // "Go to" item selected. ------------------------------------------------
    case MSG_MENU_GO_TO: {
      tInt8 lTrackNum;
      if (apMessage->FindInt8("Track", &lTrackNum) == B_OK) {
        mpCdEngine->Play(lTrackNum);
      }
      break;
    }

    // "Next" item selected. -------------------------------------------------
    case MSG_MENU_NEXT: {
      mpCdEngine->Next();
      break;
    }

    // "Previous" item selected. ---------------------------------------------
    case MSG_MENU_PREVIOUS: {
      mpCdEngine->Previous();
      break;
    }

    // "Eject" item selected. ------------------------------------------------
    case MSG_MENU_EJECT: {
      mpCdEngine->Eject();
      break;
    }

    // "Device" item selected. -----------------------------------------------
    case MSG_MENU_DEVICE: {
      tInt8 lDeviceIndex;
      if (apMessage->FindInt8("Device", &lDeviceIndex) == B_OK) {
      
        // Change devices only if the user did not select the current one.
        if (lDeviceIndex != mDeviceIndex) {
          mDeviceIndex = lDeviceIndex;
          mpCdEngine->OpenDevice(mDeviceIndex);
        }
      }
      break;
    }
    
    // Show the about window. ------------------------------------------------
	case B_ABOUT_REQUESTED: {
       
      (new BAlert("About Hustler",
	              "Hustler -- Simple Deskbar CD Player\n\n"
                  "Version " VERSION_STRING "\n"
                  "Released on " VERSION_DATE_STRING "\n\n"
                  "Created by Matthijs Hollemans\n"
                  "mahlzeit@bigfoot.com\n"
                  "http://home.concepts.nl/~hollies/\n\n"
                  "This program contains software components developed"
                  " by Peter Urbanec and Be, Inc.\n\n"
                  "Hustler is public domain. No warranties expressed"
                  " or implied.",
                  "OK"))->Go(NULL);

	  break;
	}
	
	// Do nothing. -----------------------------------------------------------
    case B_QUIT_REQUESTED: {

      /*!
       * [NOTE] Removing a replicant from the Deskbar.
       *
       * The correct way to remove a replicant from the Deskbar is to use 
       * scripting:
       * <PRE>
       *   hey Deskbar delete Replicant "Hustler" of Shelf of View "Status"
       *   of View "BarView" of Window "Deskbar"
       * </PRE>
       * Unfortunately, the next time you put the replicant into the Deskbar
       * it still uses the replicant image it loaded earlier, not the current
       * image from disk. You have to kill and restart Deskbar for it to
       * recognize the changes. (The CDButton Readme file also mentions 
       * something about this: "Also, if you make any changes to the binary, 
       * they will not get propagated to all the active replicant instances 
       * until you quit/restart the replicant shelf application.")
       *
       * I also tried to do it like this:
       * <PRE>
       *   if (RemoveSelf()) {
       *     delete this;
       *   }
       * </PRE>
       * but that doesn't work; the view is removed from the Deskbar, but its
       * housekeeping gets screwed up. The Deskbar still thinks that the
       * replicant exists, and won't release its rectangle.
       */

      break;
    }

    // Let the original BView handle the message. ----------------------------
    default: {
      BView::MessageReceived(apMessage);
      break;
    }
  }
  
  // Most of the messages above require an immediate redraw of the view.
  Invalidate();
}

// ===========================================================================
// Pulse
// ===========================================================================

void 
tHustler::Pulse(void)
{
  // Get the state of the CD engine.
  tCdEngine::tCdState lCurrentState = mpCdEngine->GetState();

  /*!
   * [NOTE] Problems with continous play.
   *
   * With normal audio CDs, we simply call Play() as soon as the CD engine 
   * goes into the stopped state. But when the CD driver encounters a data
   * track, it also goes into STATE_STOPPED. Play() is smart enough to
   * recognize whether the CD's first track is a data track, and skips it.
   * Play() immediately returns if the CD doesn't contain any audio tracks
   * at all.
   *
   * To be able to do this, Play() needs to read the CD's table of contents.
   * Unfortunately, if the CD is stopped because it only contains a data 
   * track, Play() is called on every Pulse() and accesses the CD every 500
   * milliseconds or so to read its table of contents. I think this is a bad
   * thing, so I've disabled continous play for now. Maybe it doesn't really
   * matter, and I can turn it on again (probably as an option the Options
   * submenu).
   *
   * Note that if automounting is turned on in the BeOS Mount Settings, it
   * may take a while before a CD with a data track is ready to start playing
   * because the BeOS needs to mount the data track first.
   */

  //if (lCurrentState == tCdEngine::STATE_STOPPED) {
  //  mpCdEngine->Play();
  //}
  
  // Determine whether we need to blink the track number.
  if (lCurrentState == tCdEngine::STATE_PAUSED) {
    if (mBlink == FALSE) {
      mBlink = TRUE;
    } else {
      mBlink = FALSE;
    }
    Invalidate();
  } else {
    mBlink = FALSE;
  }

  // Determine whether we need to redraw ourselves.
  tInt8 lCurrentTrack = mpCdEngine->GetCurrentTrack();
  if ((mOldState != lCurrentState)
  ||  (mOldTrack != lCurrentTrack)) {
    Invalidate();
  }
  mOldState = lCurrentState;
  mOldTrack = lCurrentTrack;
}

// ***************************************************************************
// PUBLIC Member Function Definitions: Access
// ***************************************************************************

// ***************************************************************************
// PUBLIC Member Function Definitions: Inquiry
// ***************************************************************************

// ***************************************************************************
// PROTECTED Member Function Definitions
// ***************************************************************************

// ***************************************************************************
// PRIVATE Member Function Definitions
// ***************************************************************************

// ***************************************************************************
// Helper Function Definitions
// ***************************************************************************

// ***************************************************************************
// Global Function Definitions
// ***************************************************************************

// ***************************************************************************
// Implementation Function Definitions
// ***************************************************************************
