#ifndef __PLUGIN__
#define __PLUGIN__

// (c)1997-1999 P.Cisler
//
// Plugin API for Eddie
//
// This is an early version of the plugin API, make sure you
// check the distribution site for latest versions/more description
//
// you may also send email to pavel@be.com for more help with writing an
// Eddie plugin
//
// Current plugin API allows plugins to scan/modify/select text, scroll
// to different locations in the text, process mouse clicks in the text.
// The different API calls were added more or less to support the current
// suite of plugins that comes with Eddie. More calls will follow. The API
// is maintained in such a way as to prevent/decrease the likelyhood of breaking
// the compatibility with older plugins.
// The goal of the plugin API is to provide maximum extensibility and openness to
// the application, the plugin API will be evolved to meet that goal.
// If you are developing a plugin and are missing a certain call, let me know
// and I'll provide the call for you.
//
// By looking at the current call set you might find several unimplemented
// callbacks/selector calls. These are placeholders for functionality
// that will be added soon. Having these placeholders prevent potential breaking
// of existing plugins as the calls get implemented.
//
// Check out the PluginDeveloperKit for plugin building instructions

#include <View.h>
#include <Window.h>
#include <Debug.h>
#include <stdlib.h>
#include <string.h>


#if __POWERPC__
 #define _EXPORT_WITH_PRAGMA 1
 #define _PLUGIN_EXPORT
 #define _PLUGIN_IMPORT
#else
 #if BUILDING_APP
  #define _PLUGIN_EXPORT _IMPORT
  #define _PLUGIN_IMPORT _EXPORT
 #else
  #define _PLUGIN_EXPORT _EXPORT
  #define _PLUGIN_IMPORT _IMPORT
 #endif
#endif


class PrefsPanel;
class _PLUGIN_IMPORT PluginInterface;
class SearchEngineParams;
class EddieWindowProxy;

const uint32 kEddieToPluginInterfaceVersion = 3;
	// this version indicates which set of the selectors the plugin fully supports; if the
	// existing selectors get renumbered or changed in some other incompatible way, this
	// number gets incremented and older plugins cease to work; Once the
	// API is stabilized, this is not likely to happen; Note that just adding a new selector
	// will typically not break existing plugins, the old plugin will just not be able to
	// take advantage of the call the new selector provides
const uint32 kPluginToEddieInterfaceVersion = 15;
	// this version number would change if new calls/data members were added to
	// PluginInterface; in a case like this a new plugin, compiled with this new
	// PluginInterface layout would fail when ran on an older version of Eddie; further
	// the older version of Eddie might not provide all the selector calls the plugin
	// relies on

enum DocumentLanguageType {
	kUnknown,
	kShell,				// every file with no suffix
	kCC,
	kCH,
	kCCPlus,
	kCHPlus,
	kJava,
	kHTML,
	kPerl,
	kYacc,
	kLex,
	kAsm,
	kPascal,
	kModulaDef,
	kModulaOberonImpl,
	kFortran,
	kRez,
	kCLikeAsm,			// asm with C-style comments, ifdefs
	kMakefile,			// makefile, Makefile, make*, etc.
	kPython,
	kVHDL,
	kTCL,
	kPlainText,
	kPHP,
	kLanguageLast = kPHP
};


// button part selector used for split buttons
// part press result codes
enum {
	kSolidButton = 0,
	kVerticallySplitButton = 1,
	kHorizontalySplitButton = 2,
	kFourWaySplitButton = 3,
	
	kEntireButton = 0,
	kTopButtonPart,
	kBottomButtonPart,
	kLeftButtonPart,
	kRightButtonPart,
	kLeftTopButtonPart,
	kRightTopButtonPart,
	kRightBottomButtonPart,
	kLeftBottomButtonPart
};

class PluginInterface {
public:

	enum ExtraSearchEngineFeatures {
		// If a plugin is a search engine, these are used to specify special
		// features the search engine can support
		kSupportsRegularExpressionFinds 	= 0x00000001,
			// can search using regular expressions (grep-like engine)
		kSupportsRegularExpressionReplace	= 0x00000002
			// can search and replace using regular expressions (sed-like engine)
	};

	enum PluginResult {
		kIgnored,		// did nothing during call
		kAccepted,		// handled call
		kExhausted,		// used with messages - fully handled message
		kFailed			// something screwed up
	};

	enum PluginCallSelector {
		kGetPluginVersion = 1, 	// never change this, otherwise version checking
								// will fail

		kInit,
			// called when Eddie starts up
		kUnload,
			// called when plugins are unloaded
		kQuit,
			// called when Eddie quit
		
		kButtonClick = 100,
			// plugin button clicked
		kButtonPress,
			// plugin button pressed
		kButtonDraw,
			// plugin button just draw, plugin may do some additional drawing to
			// the button
			// plugin must explicitly request this selector call

		kPrimitiveInvoked,
			// a keyboard shortcut associated with a primitive function handled
			// by this plugin was invoked
		kArgvPassed,
			// shell command handled by this plugin invoked; Eddie saves settings
			// as argv-style commands in the settings file; plugins may use the
			// same technique;
			// in future releases plugins will also be passed argv commands
			// they registered, executed from the shell

		kPulse = 200,
			// called during a pulse event of the text view
			// plugin must explicitly request this selector call
		kDraw,
			// called right after a draw call to the text view
			// plugin must explicitly request this selector call
		kMessage,
			// called from the beginning of the MessageReceived call of
			// the current text view
			// plugin must explicitly request this selector call
			// return kExhausted from plugin if the message is completely
			// processed in the call
		kMouseDown,
			// text view got a MouseDown event
			// plugin must explicitly request this selector call
		kKeyDown,
			// called from the beginning of the KeyDown call of
			// the current text view
			// plugin must explicitly request this selector call
		kRefsReceived,
			// called from the beginning of the RefsReceived call of
			// the application
			// plugin must explicitly request this selector call
			// return kExhausted from plugin if the message is completely
			// processed in the call
			

		kWindowActivated = 300,
			// was kDocumentActivated
			// window brought to front
		kWindowDeactivated,
			// was kDocumentDectivated
			// document window sent to the background
		kDocumentTextChanged,
			// was kDocumentChanged
			// document text in window changed
		kDocumentOpened,
			// document opened - called for every document in multiple document
			// windows
		kDocumentClosed,
			// document closed
		kDocumentStateChanged,
			// document file renamed, saved, made readonly, writeable, etc.
		kDocumentActivated,
			// a document was activated; in single document windows this
			// is identical to kWindowActivated, if multiple documents in window
			// this gets called when one of the documents is made the current focus
		kWindowOpened,
			// window opened - time to install buttons
		kWindowClosed,
			// window closed
		
		kGlobalSettingsSaved = 400,
			// settings for the whole application are being saved, plugin
			// may save it's own settings using the WriteSettings call
		kDocumentSettingsSaved,
			// settings are being saved for the whole document, plugin may
			// save it's own settings as attributes in the documents node
		kSettingValueRequested,
			// Eddie wants to know the value of a setting this plugin contains;
			// plugin should return the value using the ReturnSettingNameValue
			// call; the name of the requested setting is obtained by SettingName

		kGlobalPrefsShowing = 500,
			// the application settings window is being shown, plugin may
			// add it's own settings panel
		kDocumentPrefsShowing,
		
		// specialized SearchEngine callbacks
		kSearchEngineFind = 1000,
		kSearchEngineFindReverse,
		kSearchEngineReplace,
	
		// specialized FileGroup callbacks
		kNewFileGroupRequested = 1100,

		// internal plugin calls
		kSyntaxColoringOff = 10000,
		kFindQuotedString,
		
		kPluginWindowInterfaceCall = 100000
	};

	// some selector calls get sent only if a plugin requests them
	// because sending them to all plugins by default would be too
	// costly;
	// These selectors need to be explicitly requested by the plugin
	// by sending a mask of all the respective selector in the
	// RequireSelector during plugin initialization
	// 
	enum {
		kRequestButtonDrawMask = 0x1,
		kRequestPulseMask = 0x2,
		kRequestDrawMask = 0x4,
		kRequestMessageMask = 0x8,
		kRequestMouseDownMask = 0x10,
		kRequestKeyDownMask = 0x20,
		kRequestDocumentChangedMask = 0x40,
		kRequestSearchEngineCalls = 0x80,
		kRequestRefsReceivedCalls = 0x100
	};
	
	enum {
		kNewFileGroupMessage = 'nfgr'
	};
	
	// version calls, used to determine if Eddie can use plugin and vice
	// versa	
	void ReturnSupportedPluginVersion(uint32 version
		= kEddieToPluginInterfaceVersion);
		// what the plugin guarantees to respond to
	void ReturnRequiredPluginVersion(uint32 version
		= kPluginToEddieInterfaceVersion);
		// what Eddie guarantees to respond to

	// the followng calls provide details for the different selector 
	// calls

	BWindow *Window() const;
		// the window plugin is currently being called from	
	uint32 DocumentUniqueID() const;
		// uinque ID of the document the plugin is being called on;
		// this is needed for windows with multiple documents -
		// the PluginMain is called for each document in multiple document
		// windows
	
	int32 ButtonID() const;
		// button id of pressed/clicked button
	uint32 ClickSelector() const;
		// button part selector that got clicked/pressed

	// details about kDocumentTextChanged
	uint32 ChangeStart() const;
		// returns the starting position of a recent change
	uint32 RemovedLength() const;
		// returns the length of text removed by a recent change
	uint32 InsertedLength() const;
		// returns the length of text inserted by a recent change
	
	// details about kButtonDraw
	BRect ButtonRedrawRect() const;
	bool Inverted() const;
	
	// details about kKeyDown
	const char *KeyBytes() const;
	uint32 NumKeyBytes() const;
	
	// details about kMouseDown
	BPoint Point() const;
		// point of current mouse down

	// details about kMessage
	BMessage *CurrentMessage() const;
	
	// details about kDraw
	BRect CurrentDrawRect() const;
		
		
	virtual void SetupPluginButton(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits);
		// show a button for this plugin with the supplied bits
		// call this during kWindowOpened
	virtual void ChangeButtonBitmaps(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, const BWindow *thisWindowOnly = 0);
		// call this to change the button bitmaps globally of in a specified window

	
	// text calls 
	virtual void GetSelection(int32 *start, int32 *end) const;
		// get the current selection
	virtual void Select(int32 start, int32 end);
		// change the current selection
	virtual int32 TextLength() const;
		// return the length of the current text

	// insert/remove calls; work on current selection, undo is handled for the
	// plugin, undo name may be specified
	virtual void Insert(const char *text, int32 length, const char *undoName = 0);
		// paste text into current selection; an undo name may be specified, otherwise
		// the undo menu item will use Plugin Paste
	virtual void Clear(const char *undoName = 0);
		// clear current selection; an undo name may be specified, otherwise
		// the undo menu item will use Plugin Remove	
	
	virtual void GetText(char *buffer, int32 offset, int32 length) const;
		// make a copy of <length> characters at <offset> and return the result
		// in buffer 
	virtual char CharAt(int32 offset) const;
		// get the character at offset

	virtual int32 CountLines() const;
		// return line count
	virtual int32 CurrentLine() const;
		// return the index of the line containing the selection start
	virtual void GoToLine(int32 lineNum);
		// select line

	virtual void ScrollToOffset(int32 inOffset);
		// scroll text to reveal offset
	virtual void ScrollToSelection();
		// scroll to reveal start of selection
	
	virtual void SetColor(rgb_color color);
		// sets a specified color on the current text selection
	
	// fast color setting calls for setting multiple ranges at one and forcing only
	// a single update
	virtual void StartSettingColorOnRange();
		// start setting
	virtual void SetOneColorOnRange(rgb_color color, int32 start, int32 end);
		// set one
	virtual void EndSettingColorOnRange();
		// done setting, force proper update

	virtual bool SyntaxColoring() const;
		// is syntax coloring on for this document?
	
	virtual PluginInterface &operator<<(const char *);
		// currently unused
	
	virtual void GetMouse(BPoint *, uint32 *, bool checkMessageQueue = true) const;
		// return mouse location and button states
	virtual int32 PointToOffset(BPoint) const;
		// conver a point in text to an offset
	
	virtual BRect TextViewBounds() const;
	virtual BRect WindowBounds() const;
	
	virtual const char *PluginName() const;
		// return the name of this plugin
	virtual BRect GetButtonRect() const;
		// window relative location of the plugins button
		
	
	virtual void ShowClipboardPanel(const char *text);
		// display a gray panel with supplied text under the plugin button;
		// keep showing while mouse is down 

	virtual void RegisterHandledShellCommand(const char *);
		// call during plugin initialization
		// used to publish argv style commands, handled by plugin
		// currently these are used to read settings during plugin initialization
	virtual void RegisterHandledPrimitiveCommand(const char *command,
		const char *description);
		// call during plugin initialization
		// used to publish primitive functions handled by plugin
		// primitives can be bound to a keyboard shortcut
	
	virtual void RequireSelectors(uint32 selectorMask);
		// call during plugin initialization
		// specify which of the explicit selectors are needed by the plugin
		// selectors that have to be explicitly required by a plugin are listed
		// with their respective masks in an enum earlier in this file
		
	virtual void AnimateButtonClick(int32 selector = 0);
		// call from when handling a keyboard primitive to simulate button
		// being cliked; selector indicates which part of the button when
		// button is split; call this when your
		// plugin is responding to a primitive keyboard shortcut function
		// call that maps to a button click to give user a positive feedback


	// Information about current document
	virtual const BFile *DocumentNode() const;
		// the document is open as this BFile; returns null if new document
		// not yet saved; node changes after SaveAs
	virtual const entry_ref *DocumentRef() const;
		// the entry_ref of the document file; changes when document saved as,
		// renamed; the plugin gets notified about these changes be receiving the
		// selector
	virtual bool DocumentDirty() const;
		// returns true if document needs saving;
	virtual bool ReadOnly() const;
		// returns true if document cannot be modified on disk
	virtual bool IsShellWindow() const;
		// returns true if document is in shell mode
	virtual bool IsWorksheet() const;
		// returns true if document is the worksheet
	
	virtual DocumentLanguageType LanguageType() const;
		// returns the document language type

	virtual void WriteSettings(const char *format, ...);
		// called during global save settings call to save plugin settings
		// into the current settings file; plugin just writes it's settings string
		// using this call

	virtual void AddAppPrefsPanel(const char *, PrefsPanel *);
		// call to add a panel to the app settings window
	virtual void AddDocumentPrefsPanel(PrefsPanel *);
		// call to add a panel to the app settings window

	const char *CurrentPrimitive() const;
		// the keyboard primitive used to invoke plugin
	const char ***CurrentArgv() const;
		// reading settings or other argv command, registered by the plugin
		// was passed

	virtual rgb_color TextColor() const;
		// returns the TextColor of the current textView; syntax coloring
		// plugins use this to correctly revert coloring
	virtual rgb_color BackgroundColor() const;
		// returns the background color of the current text view

	// new plugin calls supporting split buttons
	virtual void SetupPluginButton(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, uint32 splitMode);
		// show a button for this plugin with the supplied bits
		// call this during kWindowOpened
	virtual void ChangeButtonBitmaps(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, uint32 splitMode,
		const BWindow *thisWindowOnly = 0);
		// call this to change the button bitmaps globally of in a specified window

	virtual int32 StartOfLine(int32 position) const;
		// returns the beginning of the line containing position
	virtual int32 EndOfLine(int32 position) const;
		// returns the end of the line containing position

	// insert/remove calls for more complex undo handling; work on current selection,
	// undo is initiated by StartUndo call; all calls to Insert and Clear are accumulated
	// within that undo
	virtual void StartUndo(int32 restoreSelStartTo, int32 restoreSelEndTo,
		const char *undoName = 0);
		// starts a new undo record with a specified or default (undo plugin) name
	virtual void SealUndo();
		// ends an undo record
	virtual void CurrentUndoInsert(const char *text, int32 length);
		// paste text into current selection; will be accumulated into the current undo record
	virtual void CurrentUndoClear();
		// clear current selection; will be accumulated into the current undo record

	// ---------- virtuals after here were added by plugin API 1.1 -----------------

private:
	PluginInterface();
	virtual ~PluginInterface();
		// have the private destructor here so that we can keep adding virtuals

public:

	virtual void RegisterSupportedSearchEngineFeatures(ExtraSearchEngineFeatures);
		// publishes which searchEngineFeatures are supported
		// during the kInit call; only needed if plugin supports any of the
		// extra features
	virtual void GetSearchParams(SearchEngineParams *) const;
		// get search params, figure out how long the search pattern is
	virtual const char *SearchString() const;
		// get the search pattern, valid only during the course of the search callback
	virtual void GetSearchText(const char **textStream, int32 *streamLength,
		char *previousChar) const;
		// get memory-mapped text to search
	virtual void SetSearchResult(int32 start, int32 end, int32 searchResultToken);
		// return offsets of search result or -1, -1; tag on a resultToken, that
		// will be passed back to the search engine during a replace; -1 is
		// an empty searchResultToken value

	virtual void GetReplaceParams(const char **replaceString, int32 *replaceStringLength,
		int32 *searchResultToken) const;
		// get the text from the replace box in the find panel and the searchResultToken
		// of the search context we are in. If -1, we are just asked to return the
		// replacement, it will be pasted into the current selection when ReplaceAndFind
		// is invoked without a prior find
	virtual void ApplyReplace(const char *replaceString, int32 replaceStringLength);
		// return the replacement string, it will get pasted for us. Simple
		// search engines just pass back the replaceString the got in GetReplaceParams,
		// regular expression search engines may do pattern expansion, etc.

	virtual bool IsWordBreakChar(char) const;
		// word breaking characters settable by the user

	// more insert/remove calls - the following insert/remove is faster than
	// the CurrentUndoInsert/CurrentUndoRemove that operate on the selection
	// the necessary text redraw and change notification is only sent at the
	// end of the edit, during SealUndo
	virtual void CurrentUndoInsert(const char *text, int32 length, int32 pos);
		// paste text at pos; will be accumulated into the current undo record
		// redraw will be performed during the closing SealUndo call
	virtual void CurrentUndoRemove(int32 pos, int32 length);
		// remove text at pos; will be accumulated into the current undo record
		// redraw will be performed during the closing SealUndo call
	virtual void SealUndo(int32 newSelStart, int32 newSelEnd);
		// sets new selection and ends an undo record; handy for the two fast
		// insert/remove calls - the entire redraw needed for text change and
		// selection change can be done at once in the single call
		// SealUndo does not perform a ScrollToSelection explicitly, you
		// need to call it yourself, if applicable
	
	virtual const char *SettingAt(int32 index) const;
	virtual char *GetSettingValue(const char *settingName);
		// returns the value of a specified setting or NULL if not found
		// caller has to delete [] the returned string
	
	// document iteration
	virtual void EachDocument(
		bool (*eachDocumentFunction)(PluginInterface *, uint32 documentID,
			const entry_ref *, void *context),
		void *context);
		// simple document iteration, returns ID and entry_ref for each document
		// iteration stops early if eachDocumentFunction returns true

	virtual void EachDocument(
		bool (*eachDocumentFunction)(PluginInterface *, uint32 documentID,
			const entry_ref *, const BFile *, BWindow *documentWindow,
			bool dirty, bool readOnly,  bool isShell, bool isWorksheet,
			DocumentLanguageType type, void *context),
		void *context);
		// detailed document iteration
		// iteration stops early if eachDocumentFunction returns true

	// Plugin window API
	virtual BWindow *NewEddieWindow(EddieWindowProxy *proxy, BRect frame,
		const char *title, uint32 look,
		uint32 feel, uint32 flags, const char * = 0);
	virtual EddieWindowProxy *FindMatchingWindow(
		bool (*matchFunc)(PluginInterface *, const EddieWindowProxy *, void *),
		void *matchContext); 
	
	// Worksets API
	virtual int32 FileGroupNew(const char * = 0);
		// create a new file group from scratch, user will get prompted
		// for location during save; returns id of the new file group
	virtual int32 FileGroupLoad(const entry_ref *, const char *optionalName = 0);
		// create a new file group and load it's contents from a specified location,
		// returns id of the new file group
	virtual status_t FileGroupSave(int32,
		status_t (*saveHook)(BFile *, void *context) = 0,
		void *context = 0);
		// save file group, pass a function in save hook to save attributes,
		// etc.
	virtual status_t FileGroupSaveAs(int32, const entry_ref *,
		status_t (*saveHook)(BFile *, void *context) = 0,
		void *context = 0);
		// save file group as
	virtual void FileGroupRemove(int32);
		// remove file group

	virtual void FileGroupGetName(int32 groupID, char *buffer);
		// pass in name buffer 256 chars long
	virtual void FileGroupGetRef(int32 groupID, entry_ref *);
		// get the entry_ref of the file group
	
	virtual void FileGroupStartObserving(int32, BHandler *);
		// request notification messages to be sent to a specified
		// handlers
	virtual void FileGroupStopObserving(int32, BHandler *);
		// no longer send notification messages
	
	virtual void FileGroupEachFile(int32 groupID,
		bool (*eachSubGroupFunction)(PluginInterface *, const char *subgroupName,
			const entry_ref *, int32 subgroupID, int32 parentID, int32 level,
			void *context),
		bool (*eachFileFunction)(PluginInterface *, const char *fileName,
			const entry_ref *, int32 fileID, int32 parentID, int32 level,
			void *context),
		void *context, bool recursive);
		// iterate file group; eachSubGroupFunction gets called for every
		// subgroup in file group, eachFileFunction gets called for every file;
		// either of the two can be null; context is an optional pointer to
		// a parameter block allocated by the caller of this each function -
		// gets passed to the two callback functions 

	virtual void FileGroupEachFileInSubgroup(int32 groupID, int32 subgroupID,
		bool (*eachSubGroupFunction)(PluginInterface *, const char *subgroupName,
			const entry_ref *, int32 subgroupID, int32 parentID, int32 level,
			void *context),
		bool (*eachFileFunction)(PluginInterface *, const char *fileName,
			const entry_ref *, int32 fileID, int32 parentID, int32 level,
			void *context),
		void *context, bool recursive);
		// same as above, iterate a specified subgroup
	
	virtual int32 FileGroupAddFile(int32 groupID, const entry_ref *);
		// add a file to the group
	virtual int32 FileGroupAddFileToDefaultGroup(int32 groupID, const entry_ref *);
		// add a new file to the last group
	virtual int32 FileGroupAddFile(int32 groupID, const entry_ref *, int32 subgroupID,
		int32 atIndex = -1);
		// add new file to a specified subgroup at a specified index
	virtual void FileGroupRemoveFile(int32 groupID, int32 fileID);
		// remove a specified file
	
	virtual void FileGroupGetFileName(int32 groupID, int32 fileID, char *buffer);
		// get file name
	virtual void FileGroupGetFileRef(int32 groupID, int32 fileID, entry_ref *);
		// get file ref
	virtual int32 FileGroupGetFileDocumentID(int32 groupID, int32 fileID);
		// get file document ID
	
	virtual int32 FileGroupGetFileSubGroup(int32 groupID, int32 fileID);
		// currently noop

	virtual int32 FileGroupAddSubGroup(int32 groupID, const entry_ref *,
		const char *optionalName = 0);
		// add a new subgroup
		
	virtual int32 FileGroupAddSubGroup(int32 groupID, const entry_ref *,
		const char *optionalName, int32 subGroupID, int32 afterItem);
		// currently noop
	virtual void FileGroupRemoveSubGroup(int32 groupID, int32 subGroupID);
	
	virtual void FileGroupGetSubGroupName(int32 groupID, int32 subGroupID, char *buffer);
		// get group name
	virtual status_t FileGroupSetSubGroupName(int32 groupID, int32 subGroupID,
		const char *buffer);
		// set subgroup name, returns B_ERROR if name not unique
	virtual bool FileGroupGetSubGroupRef(int32 groupID, int32 subGroupID, entry_ref *);
		// get group ref, returns false if no directory associated or if subgroup
		// not found
	virtual void FileGroupSetSubGroupRef(int32 groupID, int32 subGroupID, const entry_ref *);
		// set subgroup ref; 0 is valid
	virtual bool FileGroupSubGroupInMake(int32 groupID, int32 subGroupID);
		// return true if subgroup participates in make generation
	virtual void FileGroupSetSubGroupInMake(int32 groupID, int32 subGroupID, bool on);
		// set if subgroup participates in make generation


	virtual int32 FileGroupGetSubGroupSubGroup(int32 groupID, int32 subGroupID);
		// currently noop

	virtual void FileGroupAllChanged(int32 groupID);
		// currently noop
	
	virtual void FileGroupMoveItemTo(int32 groupID, int32 groupOrFileID, int32 subgroupID,
		int32 atIndex);
		// move a file/group into a different group or into a different position
		// in the same group

	// script evaluation
	virtual char *ScriptingEvaluateToken(const char *token, int32 documentID = -1);
		// evaluates variables like $WORKSHEET, $DOCUMENT, $TARGET, etc.
		// caller has to delete [] the returned string
		// if documentID specified the corresponding document is used to evaluate
		// $DOCUMENT and $TARGET, otherwise current document is used
	virtual status_t ScriptingExecute(const char *string, bool useWorksheet,
		bool appendToEnd, bool pasteOnly);
		// executes a specified scripting string either in the background or by
		// pasting it into a worksheet	

	// ---------- virtuals after here were added by plugin API 1.2 -----------------

	enum {
		kKeepCurrent,
		kOn,
		kOff
	};
	
	virtual void SetSearchParams(const char *findText, int32 caseSensitive = kKeepCurrent,
		int32 wholeWord = kKeepCurrent, int32 regexp = kKeepCurrent, int32 wrap = kKeepCurrent); 
		// pass NULL for findText to use current, use previous enum for the other values
	virtual bool Search(bool forward);
	virtual const char *RecentSearchStringAt(int32 index) const;
		// you get to keep (and delete []) the results

	virtual void SetupPluginButton(int32 id, const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, uint32 splitMode);
		// show a button for this plugin with the supplied bits
		// call this during kWindowOpened
		// this flavor is used when a plugin installs more than 1 button - an
		// id is passed that is then returned during callbacks to allow the
		// plugin to distinguish which button was pressed

	virtual void ChangeButtonBitmaps(int32 id, const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, uint32 splitMode,
		const BWindow *thisWindowOnly = 0);
		// call this to change the button bitmaps globally of in a specified window
		// can specify buttin ID for plugins with more than one buttons

	virtual BRect GetButtonRect(int32 id) const;
		// window relative location of the plugins button

	virtual void OpenFindPanel();
	virtual void ClearMultiFileSearch();
	virtual void SetMultiFileSearchDirectory(const entry_ref *ref, bool recursive);
	virtual void SetMultiFileSearchWorkset(int32 id, bool searchWorksetDirectories,
		bool recursive, uint32 filterSet = 0xffffffff);
	// ---------------------------------------------------------
	BRect PrefsViewRect() const;
		// contains the rect for the preferece panel
	
	// special purpose calls
	uint32 Pos() const;
	uint32 *StartResult() const;
	uint32 *EndResult() const;

	// settings 
	const char *SettingName() const;
	void ReturnSettingNameValue(const char *);
	
	// new file group calls
	const entry_ref *NewFileGroupRef() const;
	const char *NewFileGroupName() const;
private:

	BWindow *targetWindow;
	void *textView;
	PluginCallSelector selector;	// copy of the selector passed in call main
	void *port;
	
	uint32 changeStart;
	uint32 removedLength;
	uint32 insertedLength;

	uint32 pos;
	uint32 *start;
	uint32 *end;

	BPoint point;
	BMessage *message;

	uint32 clickSelector;
	BRect redrawRect;

	uint32 requiredPluginVersion;
	uint32 supportedPluginVersion;

	bool inverted;

	const char *currentPrimitive;
	const char ***currentArgvCommand;	// currently unused
	
	const char *bytes;
	uint32 num;
	
	BRect rect;
	uint32 documentID;
	
	int32 editStart;
	int32 editEnd;
	int32 editRemovedLength;
	int32 editInsertedLength;
	
	char *nameResult;

	entry_ref newFileGroupRef;
	const char *newFileGroupName;

	int32 buttonID;

	int32 reserved[127];

friend class PluginList;
friend class PluginPort;
friend class PluginWindow;
friend PluginPort *CallOnePlugin(PluginPort *, void *);
friend PluginPort *CheckOnePluginVersion(PluginPort *, void *);
};

class PrefsPanel {
public:
	PrefsPanel(BView *);
	virtual ~PrefsPanel();

	BView *View() const;

	virtual void Revert() = 0;
	virtual void Defaults() = 0;
	virtual void Apply() = 0;
	
	virtual bool CanRevert() const;
	virtual bool CanDefault() const;
	virtual bool CanApply() const;

	virtual bool ShowRevert() const;
	virtual bool ShowDefault() const;
	virtual bool ShowApply() const;

private:
	BView *panel;

friend class PrefsNameItem;
};

class SearchEngineParams {
public:
	bool caseSensitive;
	bool wholeWord;
	bool regExp;
	int32 reserved1;
	int32 reserved2;
	int32 reserved3;
	int32 reserved4;
	int32 reserved5;
	int32 reserved6;
	int32 reserved7;
	int32 reserved8;
	int32 reserved9;
	int32 reserved10;
};

class EddieWindowProxy {
public:
	virtual ~EddieWindowProxy() {}

	BWindow *Target();

	virtual thread_id Run();
	virtual void Quit();
	virtual void MessageReceived(BMessage *);
	virtual	void FrameMoved(BPoint);
	virtual void WorkspacesChanged(uint32, uint32);
	virtual void WorkspaceActivated(int32, bool);
	virtual	void FrameResized(float, float);
	virtual void ScreenChanged(BRect, color_space);
	virtual void Minimize(bool);	
	virtual void Zoom(BPoint, float, float);
	virtual	void MenusBeginning();
	virtual	void MenusEnded();
	virtual	void WindowActivated(bool);
	virtual	void Show();
	virtual	void Hide();
	virtual bool QuitRequested();
	
	PluginInterface *Interface();
		// only valid while target window is constructed
		
private:
	BWindow *target;
	PluginInterface *interface;

	friend class PluginInterface;
};

const uint32 kFileGroupChange = 'fgch';
 
enum FileGroupChangeType {
	kListNameChanged,
	kListRefChanged,
	kFileAdded,
	kFileRemoved,		// as a result of Eddie or a plugin removing the file from the list
	kFileDeleted,		// as a result of deleting the file from the Tracker, etc.
	kFileChanged,
	kFileMoved,			// file moved on disk
	kGroupAdded,
	kGroupRemoved,		// as a result of Eddie or a plugin removing the group from the list
	kGroupDeleted,		// as a result of deleting the group from the Tracker, etc.
	kGroupRenamed,
	kGroupChanged,
	kGroupMoved,		// group moved on disk
	kPathAdded,
	kPathRemoved,
	kBulkChange,
	kFileOrGroupMovedInList
};

#ifdef _EXPORT_WITH_PRAGMA
#pragma export on
#endif

extern "C" {

typedef PluginInterface::PluginResult (*PluginMainCall)(PluginInterface::PluginCallSelector, 
	PluginInterface *);
typedef const char *(*PluginHelpCall)(void);

_PLUGIN_EXPORT
PluginInterface::PluginResult PluginMain(PluginInterface::PluginCallSelector, 
	PluginInterface *);

_PLUGIN_EXPORT
const char *PluginHelp(void);

}

#ifdef _EXPORT_WITH_PRAGMA
#pragma export reset
#endif

// ---------------------------------------------

inline BWindow *
PluginInterface::Window() const
{
	return targetWindow;
}

inline uint32 
PluginInterface::ChangeStart() const
{
	ASSERT(selector == kDocumentTextChanged);
	return changeStart;
}

inline uint32 
PluginInterface::RemovedLength() const
{
	ASSERT(selector == kDocumentTextChanged);
	return removedLength;
}

inline uint32 
PluginInterface::InsertedLength() const
{
	ASSERT(selector == kDocumentTextChanged);
	return insertedLength;
}

inline BPoint 
PluginInterface::Point() const
{
	ASSERT(selector == kMouseDown);
	return point;
}

inline uint32 
PluginInterface::Pos() const
{
	ASSERT(selector == kFindQuotedString);
	return pos;
}

inline uint32 *
PluginInterface::StartResult() const
{
	ASSERT(selector == kFindQuotedString);
	return start;
}

inline uint32 *
PluginInterface::EndResult() const
{
	ASSERT(selector == kFindQuotedString);
	return end;
}

inline uint32 
PluginInterface::ClickSelector() const
{
	return clickSelector;
}

inline int32 
PluginInterface::ButtonID() const
{
	return buttonID;
}

inline BRect 
PluginInterface::ButtonRedrawRect() const
{
	return redrawRect;
}

inline bool 
PluginInterface::Inverted() const
{
	return inverted;
}

inline void
PluginInterface::ReturnSupportedPluginVersion(uint32 version)
{
	supportedPluginVersion = version;
}

inline void
PluginInterface::ReturnRequiredPluginVersion(uint32 version)
{
	requiredPluginVersion = version;
}

inline const char *
PluginInterface::CurrentPrimitive() const
{
	ASSERT(selector == kPrimitiveInvoked);
	return currentPrimitive;
}

inline const char *** 
PluginInterface::CurrentArgv() const
{
	ASSERT(selector == kArgvPassed);
	return currentArgvCommand;
}

inline const char *
PluginInterface::KeyBytes() const
{
	ASSERT(selector == kKeyDown);
	return bytes;
}

inline uint32 
PluginInterface::NumKeyBytes() const
{
	ASSERT(selector == kKeyDown);
	return num;
}

inline BMessage *
PluginInterface::CurrentMessage() const
{
	ASSERT(selector == kMessage || selector == kRefsReceived);
	return message;
}

inline BRect 
PluginInterface::CurrentDrawRect() const
{
	ASSERT(selector == kDraw);
	return rect;
}

inline BRect 
PluginInterface::PrefsViewRect() const
{
	ASSERT(selector == kGlobalPrefsShowing);
	return rect;
}

inline uint32 
PluginInterface::DocumentUniqueID() const
{
	return documentID;
}

inline const char *
PluginInterface::SettingName() const
{
	ASSERT(selector == kSettingValueRequested);
	return currentPrimitive;
}

inline void 
PluginInterface::ReturnSettingNameValue(const char *result)
{
	ASSERT(selector == kSettingValueRequested);
	if (!result)
		nameResult = 0;
	else {
		int32 length = strlen(result);
		nameResult = new char [length + 1];
		strcpy(nameResult, result);
	}
}

inline const entry_ref *
PluginInterface::NewFileGroupRef() const
{
	return &newFileGroupRef;
}

inline const char *
PluginInterface::NewFileGroupName() const
{
	return newFileGroupName;
}


inline
PrefsPanel::PrefsPanel(BView *panelView)
	:	panel(panelView)
{
}

inline
PrefsPanel::~PrefsPanel()
{
}

inline BView *
PrefsPanel::View() const
{
	 return panel;
}

inline bool 
PrefsPanel::CanRevert() const
{
	 return true;
}

inline bool 
PrefsPanel::CanDefault() const
{
	 return true;
}

inline bool 
PrefsPanel::CanApply() const
{
	 return true;
}

inline bool 
PrefsPanel::ShowRevert() const
{
	 return true;
}

inline bool 
PrefsPanel::ShowDefault() const
{
	 return true;
}

inline bool 
PrefsPanel::ShowApply() const
{
	 return true;
}

inline BWindow *
EddieWindowProxy::Target()
{
	return target;
}

inline thread_id 
EddieWindowProxy::Run()
{
	return target->BWindow::Run();
}

inline void 
EddieWindowProxy::Quit()
{
	target->BWindow::Quit();
}

inline void 
EddieWindowProxy::MessageReceived(BMessage *message)
{
	target->BWindow::MessageReceived(message);
}

inline void 
EddieWindowProxy::FrameMoved(BPoint point)
{
	target->BWindow::FrameMoved(point);
}

inline void 
EddieWindowProxy::WorkspacesChanged(uint32 old, uint32 nu)
{
	target->BWindow::WorkspacesChanged(old, nu);
}

inline void 
EddieWindowProxy::WorkspaceActivated(int32 ws, bool state)
{
	target->BWindow::WorkspaceActivated(ws, state);
}

inline void 
EddieWindowProxy::FrameResized(float x, float y)
{
	target->BWindow::FrameResized(x, y);
}

inline void 
EddieWindowProxy::ScreenChanged(BRect rect, color_space space)
{
	target->BWindow::ScreenChanged(rect, space);
}

inline void 
EddieWindowProxy::Minimize(bool state)
{
	target->BWindow::Minimize(state);
}

inline void 
EddieWindowProxy::Zoom(BPoint point, float width, float height)
{
	target->BWindow::Zoom(point, width, height);
}

inline void 
EddieWindowProxy::MenusBeginning()
{
	target->BWindow::MenusBeginning();
}

inline void 
EddieWindowProxy::MenusEnded()
{
	target->BWindow::MenusEnded();
}

inline void 
EddieWindowProxy::WindowActivated(bool on)
{
	target->BWindow::WindowActivated(on);
}

inline void 
EddieWindowProxy::Show()
{
	target->BWindow::Show();
}

inline void 
EddieWindowProxy::Hide()
{
	target->BWindow::Hide();
}

inline bool 
EddieWindowProxy::QuitRequested()
{
	return target->BWindow::QuitRequested();
}

inline PluginInterface *
EddieWindowProxy::Interface()
{
	ASSERT(target);
	return interface;
}

#if 0

const char kExampleNormalBits = {
	0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1C,
	0x1F,0x1E,0x1E,0x1E,0x1E,0x1E,0x16,0x00,0x00,0x16,0x1E,0x1E,0x1E,0x1E,0x1C,0x17,
	0x1F,0x1E,0x1E,0x1C,0x1C,0x1C,0x00,0x1E,0x1E,0x00,0x1C,0x1C,0x1C,0x1C,0x1A,0x17,
	0x1F,0x1E,0x15,0x00,0x00,0x00,0x00,0x0E,0x0E,0x00,0x00,0x00,0x00,0x15,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x00,0x1E,0x1D,0x1E,0x1E,0x1E,0x1E,0x00,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x00,0x1E,0x1E,0x1E,0x1E,0x1D,0x1E,0x00,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1C,0x15,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x17,0x17,
	0x1C,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x15
};

const char kExamplePressedBits = {
	0x15,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x1C,
	0x17,0x17,0x1A,0x1A,0x1A,0x1A,0x16,0x00,0x00,0x15,0x1A,0x1A,0x1A,0x1A,0x1C,0x1F,
	0x17,0x1A,0x1B,0x1C,0x1C,0x1C,0x00,0x1E,0x1E,0x00,0x1C,0x1C,0x1C,0x1C,0x1E,0x1F,
	0x17,0x1A,0x16,0x00,0x00,0x00,0x00,0x0E,0x0E,0x00,0x00,0x00,0x00,0x16,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x00,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x00,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x00,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x00,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x16,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x1E,0x1F,
	0x1C,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F
};

#endif

#endif