//--------------------------------------------------------------------
//	
//	DrumCircleData.cpp
//
//	Written by: Steffen Yount
//	
//	Copyright 1999 Steffen Yount All Rights Reserved.
//	
//--------------------------------------------------------------------

#include "DrumCircleData.h"
#if __POWERPC__
#include "my_strstream.h"
#endif
#if __INTEL__
#include <strstream.h>
#endif
#include <Message.h>
#include <Errors.h>

//====================================================================
//	Helper functions
/*
uint32 longuget(ifstream & in)
{
	unsigned char temp;
	long unsigned int out = 0;
	for (int i = 0; i < 4; i++)
	{
		in.get(temp);
		out = (out << 8) + temp;
	}
	return out;
}

void longuput(ofstream & out, uint32 val)
{
	for (int i = 3; i >= 0; i--)
	{
		out.put((unsigned char)((val >> (8*i)) & 0xff));
	}
}

*/

//====================================================================
//	DrumCircleData Implementation


//--------------------------------------------------------------------
//	DrumCircleData constructors, destructors, operators

DrumCircleData::DrumCircleData(const BMessage * dcdmessage):
		drumcount(p_drumcount), measurelen(p_measurelen), drums(p_drums), 
		velocity(p_velocity), duration(p_duration), rythm(p_rythm), 
		changelust(p_changelust), hitlust(p_hitlust), channel(p_channel), 
		on(p_on), name(p_name)
	{
	status_t err;
	p_drumcount = dcdmessage->FindInt16("drumcount");
	p_measurelen = dcdmessage->FindInt16("measurelen");
	ssize_t sz;
	uint16 * tu16;
	
	err = dcdmessage->FindData("drums", B_UINT16_TYPE, (const void **)&tu16, &sz); 
	if (sz != drumcount*2)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_drums = new uint16[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			drums[i] = tu16[i];
	}

	err |= dcdmessage->FindData("velocity", B_UINT16_TYPE, (const void **)&tu16, &sz); 
	if (sz != drumcount*2)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_velocity = new uint16[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			velocity[i] = tu16[i];
	}
	
	float * tf;
	err |= dcdmessage->FindData("duration", B_FLOAT_TYPE, (const void **)&tf, &sz); 
	if (sz != drumcount*4)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_duration = new float[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			duration[i] = tf[i];
	}
	
	bool * tb;
	err |= dcdmessage->FindData("rythm", B_BOOL_TYPE, (const void **)&tb, &sz); 
	if (sz != drumcount*measurelen)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_rythm = new bool * [drumcount];
		for (uint16 i = 0; i < drumcount; i++)
		{
			p_rythm[i] = new bool[measurelen];
			for (uint16 k = 0; k < measurelen; k++)
				rythm[i][k] = tb[i*measurelen+k];
		}
	}
	
	err = dcdmessage->FindData("changelust", B_UINT16_TYPE, (const void **)&tu16, &sz); 
	if (sz != drumcount*2)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_changelust = new uint16[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			changelust[i] = tu16[i];
	}
	
	err = dcdmessage->FindData("hitlust", B_UINT16_TYPE, (const void **)&tu16, &sz); 
	if (sz != drumcount*2)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_hitlust = new uint16[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			hitlust[i] = tu16[i];
	}
	
	err = dcdmessage->FindData("channel", B_UINT16_TYPE, (const void **)&tu16, &sz); 
	if (sz != drumcount*2)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_channel = new uint16[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			channel[i] = tu16[i];
	}

	err |= dcdmessage->FindData("on", B_BOOL_TYPE, (const void **)&tb, &sz); 
	if (sz != drumcount)
		err |= B_BAD_VALUE;
	if (!err) 
	{
		p_on = new bool[drumcount];
		for (uint16 i = 0; i < drumcount; i++)
			on[i] = tb[i];
	}
	
	ostrstream buf;
	buf << dcdmessage->FindString("dcd_name") << '\0';
	p_name = buf.str();

	if (err)
	{
		this->~DrumCircleData();
		p_drumcount = 0;
		p_measurelen = 0;
//		cout << "problems constructing DrumCircleData!!!";
	}	
	
	
	/*
	//old copy constructor
	p_drumcount = d.drumcount;
	p_measurelen = d.measurelen;
	p_drums = new uint16[d.drumcount];
	for (uint16 i = 0; i < d.drumcount; i++)
	{
		drums[i] = d.drums[i];
	}
	p_velocity = new uint16[d.drumcount];
	p_duration = new float[d.drumcount];
	p_rythm = new bool * [d.drumcount];
	p_changelust = new uint16[d.drumcount];
	p_hitlust = new uint16[d.drumcount];
	p_channel = new uint16[d.drumcount];
	p_on = new bool[d.drumcount];

	ostrstream buf;
	buf << d.name << '\0';
	p_name = buf.str();

	for (uint16 i = 0; i < d.drumcount; i++)
	{
		p_rythm[i] = new bool[d.measurelen];
		for (uint16 k = 0; k < d.measurelen; k++)
			rythm[i][k] = d.rythm[i][k];
		velocity[i] = d.velocity[i];
		duration[i] = d.duration[i];
		changelust[i] = d.changelust[i];
		hitlust[i] = d.hitlust[i];
		channel[i] = d.channel[i];
		on[i] = d.on[i];
	}*/
}

DrumCircleData::DrumCircleData( uint16 measure_length, uint16 drum_count):
		drumcount(p_drumcount), measurelen(p_measurelen), drums(p_drums), 
		velocity(p_velocity), duration(p_duration), rythm(p_rythm), 
		changelust(p_changelust), hitlust(p_hitlust), channel(p_channel), 
		on(p_on), name(p_name)
{
	p_drumcount = drum_count;
	p_measurelen = measure_length;
	const uint16 def_drums[16] = { 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
							0x23, 0x56, 0x38, 0x1f,
							0x2c, 0x36, 0x45, 0x4c, 0x4d, 0x52,
							0x46 };
	p_drums = new uint16[drumcount];
	for (uint16 i = 0; i < 16 && i < drumcount; i++)
	{
		drums[i] = def_drums[i];
	}
	for (uint16 i = 16; i < drumcount; i++)
	{
		drums[i] = 0x01;
	}
/*
	p_drums = new uint16[drumcount];
	for (uint16 i = 0; i < drumcount; i++)
	{
		drums[i] = i+27;
	}
*/
	p_velocity = new uint16[drumcount];
	p_duration = new float[drumcount];
	p_rythm = new bool * [drumcount];
	p_changelust = new uint16[drumcount];
	p_hitlust = new uint16[drumcount];
	p_channel = new uint16[drumcount];
	p_on = new bool[drumcount];

	ostrstream buf;
	buf << "DrumCircle" << '\0';
	p_name = buf.str();

	for (uint16 i = 0; i < drumcount; i++)
	{
		p_rythm[i] = new bool[measurelen];
		for (uint16 k = 0; k < measurelen; k++)
			rythm[i][k] = false;
		velocity[i] = 0x63;
		duration[i] = 0.25;
		changelust[i] = 10;
		hitlust[i] = 25;
		channel[i] = 10;
		on[i] = true;
	}
}


status_t DrumCircleData::Archive(BMessage *archive, bool deep) const
{
	status_t err;
	err = BArchivable::Archive(archive, deep);
	err |= archive->AddString("class", "DrumCircleData"); 
	err |= archive->AddInt16("drumcount", drumcount);
	err |= archive->AddInt16("measurelen", measurelen);
	err |= archive->AddData("drums", B_UINT16_TYPE, 
         drums, drumcount*2, true, drumcount);
	err |= archive->AddData("velocity", B_UINT16_TYPE, 
         velocity, drumcount*2, true, drumcount);
    err |= archive->AddData("duration", B_FLOAT_TYPE, 
         duration, drumcount*4, true, drumcount);
    bool * temp = new bool[drumcount*measurelen];
    if (temp == NULL)
    	err |= B_NO_MEMORY;
    else
    {
	    for (uint16 i = 0; i < drumcount; i++)
    		for (uint16 j = 0; j < measurelen; j++)
    		{
    			temp[i*measurelen+j] = rythm[i][j];
	    	}
    	err |= archive->AddData("rythm", B_BOOL_TYPE, 
        	 temp, drumcount*measurelen, true, drumcount*measurelen);
	    delete temp;
	}
	err |= archive->AddData("changelust", B_UINT16_TYPE, 
         changelust, drumcount*2, true, drumcount);
	err |= archive->AddData("hitlust", B_UINT16_TYPE, 
         hitlust, drumcount*2, true, drumcount);
	err |= archive->AddData("channel", B_UINT16_TYPE, 
         channel, drumcount*2, true, drumcount);
    err |= archive->AddData("on", B_BOOL_TYPE, 
         on, drumcount, true, drumcount);
    err |= archive->AddString("dcd_name", name);
	return err;
} 

BArchivable * DrumCircleData::Instantiate(BMessage *archive)
{
	if ( validate_instantiation(archive, "DrumCircleData")) 
		return new DrumCircleData(archive); 
	return NULL; 

}

void DrumCircleData::Rename(const char * newname)
{
	ostrstream buf;
	buf << newname << '\0';
	delete [] p_name;
	p_name = buf.str();
}

void DrumCircleData::Setto(const DrumCircleData & d)
{
	for (uint16 i = 0; i < drumcount && i < d.drumcount; i++)
	{
		drums[i] = d.drums[i];
	}
	for (uint16 i = 0; i < drumcount && i < d.drumcount; i++)
	{
		for (uint16 k = 0; k < measurelen && k < d.measurelen; k++)
			rythm[i][k] = d.rythm[i][k];
		velocity[i] = d.velocity[i];
		duration[i] = d.duration[i];
		changelust[i] = d.changelust[i];
		hitlust[i] = d.hitlust[i];
		channel[i] = d.channel[i];
		on[i] = d.on[i];
	}
	ostrstream buf;
	buf << d.name << '\0';
	delete [] p_name;
	p_name = buf.str();

}

void DrumCircleData::Move(uint16 from, uint16 to)
{
	if (from < drumcount && to < drumcount)
	{
		uint16 drum = drums[from];
		uint16 vel = velocity[from];
		float dur = duration[from];
		bool * ryth = p_rythm[from];
		uint16 chng = changelust[from];
		uint16 hit = hitlust[from];
		uint16 chan = channel[from];
		bool t_on = on[from];
		
		if (from < to)
		{
			for (uint16 i = from; i < to; i++)
			{
				drums[i] = drums[i+1];
				velocity[i] = velocity[i+1];
				duration[i] = duration[i+1];
				p_rythm[i] = p_rythm[i+1];
				changelust[i] = changelust[i+1];
				hitlust[i] = hitlust[i+1];
				channel[i] = channel[i+1];
				on[i] = on[i+1];
			}	
		}
		else if (from > to)
		{
			for(uint16 i = from; i > to; i--)
			{
				drums[i] = drums[i-1];
				velocity[i] = velocity[i-1];
				duration[i] = duration[i-1];
				p_rythm[i] = p_rythm[i-1];
				changelust[i] = changelust[i-1];
				hitlust[i] = hitlust[i-1];
				channel[i] = channel[i-1];
				on[i] = on[i-1];
			}
		}
		if (from != to)
		{
			drums[to] = drum;
			velocity[to] = vel;
			duration[to] = dur;
			p_rythm[to] = ryth;
			changelust[to] = chng;
			hitlust[to] = hit;
			channel[to] = chan;
			on[to] = t_on;
		}
	}
}
	
DrumCircleData::~DrumCircleData()
{
	delete [] p_drums;
	delete [] p_velocity;
	delete [] p_duration;
	for (uint16 i = 0; i < drumcount; i++)
	{
		delete [] p_rythm[i];
	}
	delete [] p_rythm;					
	delete [] p_changelust;
	delete [] p_hitlust;
	delete [] p_channel;
	delete [] p_on;
	delete [] p_name;
}


