/******************************************************************************/
/* ModTracker Player Interface.                                               */
/*                                                                            */
/* Original players by:                                                       */
/*     SoundTracker15: Karsten Obarski.                                       */
/*     SoundTracker31: Unknown of D.O.C.                                      */
/*     NoiseTracker  : Mahoney & Kaktus.                                      */
/*     StarTrekker   : Bjorn Wesen (Exolon).                                  */
/*     ProTracker    : Lars Hamre.                                            */
/*     FastTracker   : Fredrik Muss.                                          */
/*     TakeTracker   : Anders B. Ervik (Dr. Zon) & Oyvind Neuman (Twaddler).  */
/*     MultiTracker  : Daniel Goldstein.                                      */
/*                                                                            */
/* Converted to C++ by Thomas Neumann.                                        */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


// PolyKit headers
#include "PException.h"
#include "PString.h"
#include "PFile.h"
#include "PAlert.h"
#include "GlobalFuncs.h"

// Player headers
#include "ModTracker.h"
#include "ModTracker_Lang.h"
#include "ModTracker_LangStr.h"
#include "Tables.h"


/******************************************************************************/
/* Version                                                                    */
/******************************************************************************/
#define PlayerVersion		1.61



/******************************************************************************/
/* Constructor                                                                */
/******************************************************************************/
ModTracker::ModTracker(APGlobalData *global) : APAddOnPlayer(global)
{
	// Fill out the version we have compiled under
	addOnVersion  = AP_ADDON_VERSION;
	playerVersion = AP_PLAYER_VERSION;
	agentVersion  = AP_AGENT_VERSION;

	// Initialize member variables
	samples     = NULL;
	tracks      = NULL;
	sequences   = NULL;
	amData      = NULL;
	channels    = NULL;
	readyToPlay = NULL;

	// Register string resources
	strings = AddStringResource(modStrings);
}



/******************************************************************************/
/* Destructor                                                                 */
/******************************************************************************/
ModTracker::~ModTracker(void)
{
	// Unregister string resources
	RemoveStringResource(strings);
}



/******************************************************************************/
/* GetVersion() returns the version of the add-on.                            */
/*                                                                            */
/* Output: The add-on version.                                                */
/******************************************************************************/
float ModTracker::GetVersion(void)
{
	return (PlayerVersion);
}



/******************************************************************************/
/* GetCount() returns the number of add-ons in the add-on.                    */
/*                                                                            */
/* Output: Is the number of the add-ons.                                      */
/******************************************************************************/
uint32 ModTracker::GetCount(void)
{
	return (8);
}



/******************************************************************************/
/* GetSupportFlags() returns some flags telling what the add-on supports.     */
/*                                                                            */
/* Output: Is the flags.                                                      */
/******************************************************************************/
uint32 ModTracker::GetSupportFlags(void)
{
	return (appSetPosition);
}



/******************************************************************************/
/* GetName() returns the name of the current add-on.                          */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on name.                                                   */
/******************************************************************************/
PString ModTracker::GetName(uint32 index)
{
	PString name;

	name.LoadString(strings, IDS_MOD_NAME + index);
	return (name);
}



/******************************************************************************/
/* GetDescription() returns the description of the current add-on.            */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on description.                                            */
/******************************************************************************/
PString ModTracker::GetDescription(uint32 index)
{
	PString description;

	description.LoadString(strings, IDS_MOD_DESCRIPTION + index);
	return (description);
}



/******************************************************************************/
/* GetModTypeString() returns the module type string for the current player.  */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*                                                                            */
/* Output: The module type string.                                            */
/******************************************************************************/
PString ModTracker::GetModTypeString(uint32 index)
{
	PString type;

	type.LoadString(strings, IDS_MOD_MIME + index);
	return (type);
}



/******************************************************************************/
/* ModuleCheck() tests the module to see which type of module it is.          */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*         "file" is a pointer to a APFile object with the file to check.     */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result ModTracker::ModuleCheck(uint32 index, PFile *file)
{
	ModType checkType;

	// Check the module
	checkType = TestModule(file);

	if (checkType == (int32)index)
	{
		modType = checkType;
		return (AP_OK);
	}

	// We couldn't recognize it
	return (AP_UNKNOWN);
}



/******************************************************************************/
/* TestModule() tests the module to see which type of module it is.           */
/*                                                                            */
/* Input:  "file" is a pointer to a APFile object with the file to check.     */
/*                                                                            */
/* Output: modUnknown for unknown or the module type.                         */
/******************************************************************************/
ModType ModTracker::TestModule(PFile *file)
{
	uint32 mark;
	uint16 i;
	uint8 buf[4];
	bool ok;
	ModType retVal = modUnknown;

	// First check to see if it's a MultiTracker module
	if (file->GetLength() < (int32)sizeof(MultiModule))
		return (modUnknown);

	// Check the signature
	file->SeekToBegin();
	file->Read(buf, 4);

	if ((buf[0] == 'M') && (buf[1] == 'T') && (buf[2] == 'M') && (buf[3] == 0x10))
		retVal = modMultiTracker;
	else
	{
		// Now check to see if it's a Noise- or ProTracker module
		if (file->GetLength() < (int32)sizeof(ProModule))
			return (modUnknown);

		// Check mark
		file->Seek(1080, PFile::pSeekBegin);
		mark = file->Read_B_UINT32();

		if ((mark == 'M.K.') || (mark == 'M!K!') || (mark == 'M&K!'))
		{
			// Now we know it either a Noise- or ProTracker module, but we
			// need to know exactly which type it is
			retVal = modNoiseTracker;

			// Check the restart byte
			file->Seek(951, PFile::pSeekBegin);
			if (file->Read_UINT8() > 126)
				retVal = modProTracker;
			else
			{
				// Check the first playing pattern for any BPM speed effects or
				// ExtraEffect effects just to be sure it's not a NoiseTracker module.
				file->Seek(1084 + file->Read_UINT8() * 4 * 4 * 64, PFile::pSeekBegin);

				for (i = 0; i < 4 * 64; i++)
				{
					uint8 a, b, c, d;

					a = file->Read_UINT8();
					b = file->Read_UINT8();
					c = file->Read_UINT8();
					d = file->Read_UINT8();

					if ((c & 0x0f) == effSetSpeed)
					{
						if (d > 31)
						{
							retVal = modProTracker;
							break;
						}
					}

					if ((c & 0x0f) == effExtraEffect)
					{
						if (d > 1)
						{
							retVal = modProTracker;
							break;
						}
					}
				}

				if (retVal != modProTracker)
				{
					// Well, now we want to be really really sure it's
					// not a NoiseTracker module, so we check the sample
					// information to see if some samples has a finetune.
					file->Seek(44, PFile::pSeekBegin);

					for (i = 0; i < 31; i++)
					{
						if ((file->Read_UINT8() & 0x0f) != 0)
						{
							retVal = modProTracker;
							break;
						}

						// Seek to the next sample
						file->Seek(30 - 1, PFile::pSeekCurrent);
					}
				}
			}
		}
		else
		{
			if (mark == 'FLT4')
				retVal = modStarTrekker;
			else
			{
				if ((mark == '6CHN') || (mark == '8CHN'))
					retVal = modFastTracker;
				else
				{
					if (((mark & 0x00ffffff) == '\0CHN') || ((mark & 0x0000ffff) == '\0\0CH') ||
						((mark & 0xffffff00) == 'TDZ\0'))
					{
						retVal = modTakeTracker;
					}
					else
					{
						// Check the mark to see if there is illegal character in it. There
						// has to be, else it isn't a SoundTracker module.
						ok = false;

						for (i = 0; i < 4; i++)
						{
							uint8 byte = mark & 0xff;
							if ((byte < 32) || (byte > 127))
							{
								ok = true;
								break;
							}

							mark = mark >> 8;
						}

						if (ok)
						{
							// Now we know it could be a SoundTracker module, but we have to
							// check the sample names, just to be sure.
							for (i = 0; i < 15; i++)
							{
								if (!(CheckSampleName(file, i)))
								{
									ok = false;
									break;
								}
							}

							if (ok)
							{
								// Check sample number 16, it has to be illegal if it's a
								// SoundTracker15 module
								if (!(CheckSampleName(file, 15)))
								{
									// And to be extra sure, the song length may not be 0
									file->Seek(470, PFile::pSeekBegin);

									if (file->Read_UINT8() != 0)
										retVal = modSoundTracker15;
								}
								else
								{
									// Well, we check the rest of the samples to see if it's
									// a SoundTracker31 module
									for (i = 15; i < 31; i++)
									{
										if (!(CheckSampleName(file, i)))
										{
											ok = false;
											break;
										}
									}

									// And to be extra sure, the song length may not be 0
									file->Seek(950, PFile::pSeekBegin);

									if (file->Read_UINT8() == 0)
										ok = false;

									if (ok)
										retVal = modSoundTracker31;
								}
							}
						}
					}
				}
			}
		}
	}

	return (retVal);
}



/******************************************************************************/
/* CheckSampleName() checks a sample name for illegal characters.             */
/*                                                                            */
/* Input:  "file" is a pointer to a file object with the file to check.       */
/*         "num" is the file to checksample number to check starting from 0.  */
/*                                                                            */
/* Output: True if the sample name is ok, else false.                         */
/******************************************************************************/
bool ModTracker::CheckSampleName(PFile *file, uint16 num)
{
	unsigned char sampName[22];
	bool result = true;

	// Seek to the right position and read the name
	file->Seek(20 + num * 30, PFile::pSeekBegin);
	file->Read(sampName, 22);

	// Now check the name
	for (uint16 i = 0; i < 22; i++)
	{
		if (sampName[i] != 0x00)
		{
			if ((sampName[i] < 32) || (sampName[i] > 127))
			{
				result = false;
				break;
			}
		}
	}

	return (result);
}



/******************************************************************************/
/* LoadModule() will load the module into the memory.                         */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*         "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result ModTracker::LoadModule(uint32 index, PFile *file)
{
	ap_result retVal;

	// Remember the module type
	modType = (ModType)index;

	if (modType == modMultiTracker)
		retVal = LoadMultiTracker(file);
	else
		retVal = LoadTracker(file);

	return (retVal);
}



/******************************************************************************/
/* LoadTracker() will load a tracker module into the memory.                  */
/*                                                                            */
/* Input:  "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result ModTracker::LoadTracker(PFile *file)
{
	char buf[21];
	int32 i, j, k, n;
	uint32 mark;
	ap_result retVal = AP_ERROR;

	try
	{
		// Find out the number of samples
		if (modType == modSoundTracker15)
			sampleNum = 15;
		else
			sampleNum = 31;

		// Read the songname
		buf[20] = 0x00;
		file->Read(buf, 20);
		songName = buf;

		// Allocate space to the samples
		samples = new Sample[sampleNum];
		if (samples == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		// Read the samples
		for (i = 0; i < sampleNum; i++)
		{
			SampleInfo sampInfo;
			Sample *samp = &samples[i];

			// Read the sample info
			sampInfo.sampleName[22] = 0x00;
			file->Read(sampInfo.sampleName, 22);

			sampInfo.length       = file->Read_B_UINT16();
			sampInfo.fineTune     = file->Read_UINT8();
			sampInfo.volume       = file->Read_UINT8();
			sampInfo.repeatStart  = file->Read_B_UINT16();
			sampInfo.repeatLength = file->Read_B_UINT16();

			// Correct "funny" modules
			if ((sampInfo.repeatStart + sampInfo.repeatLength) > sampInfo.length)
				sampInfo.repeatLength = sampInfo.length - sampInfo.repeatStart;

			// Do the recognized format support finetune?
			if ((modType == modSoundTracker15) || (modType == modSoundTracker31) ||
				(modType == modNoiseTracker) || (modType == modStarTrekker))
			{
				sampInfo.fineTune = 0;
			}

			if (file->IsEOF())
			{
				ShowError(IDS_MOD_ERR_LOADING_SAMPLEINFO);
				throw PUserException();
			}

			// Put the information into the 
			samp->sampleName = sampInfo.sampleName;
			samp->start      = NULL;
			samp->length     = sampInfo.length;
			samp->loopStart  = sampInfo.repeatStart;
			samp->loopLength = sampInfo.repeatLength;
			samp->fineTune   = sampInfo.fineTune & 0xf;
			samp->volume     = sampInfo.volume;
		}

		// Read more header information
		songLength = file->Read_UINT8();

		if ((modType == modNoiseTracker) || (modType == modStarTrekker))
			restartPos = file->Read_UINT8();
		else
		{
			file->Read_UINT8();
			restartPos = 0;
		}

		file->Read(positions, 128);

		if ((modType != modSoundTracker15) && (modType != modSoundTracker31))
			mark = file->Read_B_UINT32();
		else
			mark = 0;

		if (file->IsEOF())
		{
			ShowError(IDS_MOD_ERR_LOADING_HEADER);
			throw PUserException();
		}

		// Find the missing information
		patternLength = 64;

		// Find the number of channels used
		channelNum = 4;

		if ((mark & 0x00ffffff) == '\0CHN')
			channelNum = ((mark & 0xff000000) >> 24) - 0x30;
		else
		{
			if ((mark & 0x0000ffff) == '\0\0CH')
				channelNum = (((mark & 0xff000000) >> 24) - 0x30) * 10 + ((mark & 0x00ff0000) >> 16) - 0x30;
			else
			{
				if ((mark & 0xffffff00) == 'TDZ\0')
					channelNum = (mark & 0x000000ff) - 0x30;
			}
		}

		// Find heighest pattern number
		maxPattern = 0;

		for (i = 0; i < 128; i++)
		{
			if (positions[i] > maxPattern)
				maxPattern = positions[i];
		}
		maxPattern++;

		trackNum = maxPattern * channelNum;

		// Find the min and max periods
		if ((modType == modFastTracker) || (modType == modTakeTracker))
		{
			minPeriod = 28;
			maxPeriod = 3424;
		}
		else
		{
			minPeriod = 113;
			maxPeriod = 856;
		}

		// Allocate space for the patterns
		tracks = new TrackLine *[trackNum];
		if (tracks == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		memset(tracks, 0, trackNum * sizeof(TrackLine *));

		// Read the tracks
		TrackLine **line = new TrackLine *[channelNum];

		for (i = 0; i < trackNum / channelNum; i++)
		{

			// Allocate memory to hold 4 tracks
			for (j = 0; j < channelNum; j++)
			{
				tracks[i * channelNum + j] = new TrackLine[64];
				if (tracks[i * channelNum + j] == NULL)
				{
					delete[] line;

					ShowError(IDS_MOD_ERR_MEMORY);
					throw PUserException();
				}

				line[j] = tracks[i * channelNum + j];
			}

			// Now read the tracks
			for (k = 0; k < 64; k++)
			{
				for (j = 0; j < channelNum; j++)
				{
					TrackLine *workLine;
					uint8 a, b, c, d;
					uint16 note;

					workLine = line[j];

					a = file->Read_UINT8();
					b = file->Read_UINT8();
					c = file->Read_UINT8();
					d = file->Read_UINT8();

					note = ((a & 0x0f) << 8) | b;

					if (note)		// Is there any note?
					{
						for (n = 0; n < NumberOfNotes; n++)
						{
							if (note >= periods[0][n])
								break;			// Found the note number
						}

						workLine[k].note = n + 1;
					}
					else
						workLine[k].note = 0;

					workLine[k].sample    = (a & 0xf0) | ((c & 0xf0) >> 4);
					workLine[k].effect    = c & 0x0f;
					workLine[k].effectArg = d;
				}
			}

			if (file->IsEOF())
			{
				delete[] line;

				ShowError(IDS_MOD_ERR_LOADING_PATTERN);
				throw PUserException();
			}
		}

		delete[] line;

		// Allocate memory to hold the sequences
		sequences = new uint16[32 * maxPattern];
		if (sequences == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		// Calculate the sequence numbers
		for (i = 0; i < maxPattern; i++)
		{
			for (j = 0; j < channelNum; j++)
				sequences[i * 32 + j] = i * channelNum + j;
		}

		// Read the samples
		for (i = 0; i < sampleNum; i++)
		{
			int8 *sampData;
			int32 length;

			// Allocate space to hold the sample
			length = samples[i].length * 2;

			if (length != 0)
			{
				sampData = new int8[length];
				if (sampData == NULL)
				{
					ShowError(IDS_MOD_ERR_MEMORY);
					throw PUserException();
				}

				memset(sampData, 0, length);
				samples[i].start = sampData;

				// Check the see if we miss to much from the last sample
				if (file->GetLength() - file->GetPosition() < (length - 512))
				{
					ShowError(IDS_MOD_ERR_LOADING_SAMPLE);
					throw PUserException();
				}

				// Read the sample
				file->Read(sampData, length);
			}
		}

		// Ok, we're done
		retVal = AP_OK;
	}
	catch(PUserException e)
	{
		// Juse delete the exception and clean up
		Cleanup();
	}
	catch(...)
	{
		// Clean up
		Cleanup();
		throw;
	}

	return (retVal);
}



/******************************************************************************/
/* LoadMultiTracker() will load a MultiTracker module into the memory.        */
/*                                                                            */
/* Input:  "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result ModTracker::LoadMultiTracker(PFile *file)
{
	MultiModule mulMod;
	int32 i, j;
	ap_result retVal = AP_ERROR;

	try
	{
		// Read the header
		file->Read(mulMod.mark, 3);

		mulMod.version = file->Read_UINT8();

		mulMod.songName[20] = 0x00;
		file->Read(mulMod.songName, 20);

		mulMod.trackNum      = file->Read_L_UINT16();
		mulMod.patternNum    = file->Read_UINT8();
		mulMod.songLength    = file->Read_UINT8();
		mulMod.commentLength = file->Read_L_UINT16();
		mulMod.sampleNum     = file->Read_UINT8();
		mulMod.attributes    = file->Read_UINT8();
		mulMod.patternLength = file->Read_UINT8();
		mulMod.channels      = file->Read_UINT8();

		file->Read(panning, 32);

		if (file->IsEOF())
		{
			ShowError(IDS_MOD_ERR_LOADING_HEADER);
			throw PUserException();
		}

		// Remember some of the information
		songName      = mulMod.songName;
		maxPattern    = mulMod.patternNum + 1;
		channelNum    = mulMod.channels;
		sampleNum     = mulMod.sampleNum;
		songLength    = mulMod.songLength + 1;
		trackNum      = mulMod.trackNum;
		patternLength = mulMod.patternLength;

		// Allocate the samples
		samples = new Sample[sampleNum];
		if (samples == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		for (i = 0; i < sampleNum; i++)
		{
			MultiSampleInfo sampInfo;
			Sample *samp = &samples[i];

			// Read the sample information
			sampInfo.sampleName[22] = 0x00;

			file->Read(sampInfo.sampleName, 22);

			sampInfo.length      = file->Read_L_UINT32();
			sampInfo.repeatStart = file->Read_L_UINT32();
			sampInfo.repeatEnd   = file->Read_L_UINT32();
			sampInfo.fineTune    = file->Read_UINT8();
			sampInfo.volume      = file->Read_UINT8();
			sampInfo.attributes  = file->Read_UINT8();

			samp->sampleName = sampInfo.sampleName;
			samp->start      = NULL;
			samp->length     = sampInfo.length / 2;
			samp->loopStart  = sampInfo.repeatStart / 2;
			samp->loopLength = sampInfo.repeatEnd / 2 - samp->loopStart;
			samp->fineTune   = sampInfo.fineTune & 0xf;
			samp->volume     = sampInfo.volume;

			if (file->IsEOF())
			{
				ShowError(IDS_MOD_ERR_LOADING_SAMPLEINFO);
				throw PUserException();
			}
		}

		// Read the positions
		file->Read(positions, 128);

		// Allocate memory to hold all the tracks
		tracks = new TrackLine *[trackNum + 1];
		if (tracks == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		memset(tracks, 0, (trackNum + 1) * sizeof(TrackLine *));

		// Generate an empty track
		tracks[0] = new TrackLine[patternLength];
		if (tracks[0] == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		memset(tracks[0], 0, patternLength * sizeof(TrackLine));

		// Read the tracks
		for (i = 0; i < trackNum; i++)
		{
			TrackLine *line;

			// Allocate memory to hold the track
			line = new TrackLine[patternLength];
			if (line == NULL)
			{
				ShowError(IDS_MOD_ERR_MEMORY);
				throw PUserException();
			}

			tracks[i + 1] = line;

			// Now read the track
			for (j = 0; j < patternLength; j++)
			{
				uint8 a, b, c;

				a = file->Read_UINT8();
				b = file->Read_UINT8();
				c = file->Read_UINT8();

				line[j].note = a >> 2;
				if (line[j].note != 0)
					line[j].note += 13;

				line[j].sample    = (a & 0x03) << 4 | ((b & 0xf0) >> 4);
				line[j].effect    = b & 0x0f;
				line[j].effectArg = c;
			}

			if (file->IsEOF())
			{
				ShowError(IDS_MOD_ERR_LOADING_PATTERN);
				throw PUserException();
			}
		}

		// Allocate memory to hold the sequences
		sequences = new uint16[32 * maxPattern];
		if (sequences == NULL)
		{
			ShowError(IDS_MOD_ERR_MEMORY);
			throw PUserException();
		}

		// Read the sequence data
		file->ReadArray_L_UINT16s(sequences, 32 * maxPattern);

		// Skip the comment field
		file->Seek(mulMod.commentLength, PFile::pSeekCurrent);

		if (file->IsEOF())
		{
			ShowError(IDS_MOD_ERR_LOADING_PATTERN);
			throw PUserException();
		}

		// Read the samples
		for (i = 0; i < sampleNum; i++)
		{
			int8 *sampData;
			int32 length;

			// Allocate space to hold the sample
			length = samples[i].length * 2;

			if (length != 0)
			{
				sampData = new int8[length];
				if (sampData == NULL)
				{
					ShowError(IDS_MOD_ERR_MEMORY);
					throw PUserException();
				}

				samples[i].start = sampData;

				// Read the sample
				file->Read(sampData, length);

				if (file->IsEOF())
				{
					ShowError(IDS_MOD_ERR_LOADING_SAMPLE);
					throw PUserException();
				}

				// Convert the sample to signed
				for (j = 0; j < length; j++)
					*sampData++ = *sampData + 0x80;
			}
		}

		// Initialize the rest of the variables used
		minPeriod  = 45;
		maxPeriod  = 1616;
		restartPos = 0;

		// Ok, we're done
		retVal = AP_OK;
	}
	catch(PUserException e)
	{
		// Juse delete the exception and clean up
		Cleanup();
	}
	catch(...)
	{
		// Clean up
		Cleanup();
		throw;
	}

	return (retVal);
}



/******************************************************************************/
/* ExtraLoad() will load all extra files needed.                              */
/*                                                                            */
/* Input:  "fileName" is the filename of the module with full path.           */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result ModTracker::ExtraLoad(PString fileName)
{
	if (modType == modStarTrekker)
	{
		PFile *file;

		try
		{
			file = OpenExtraFile(fileName, "nt");

			try
			{
				// Allocate memory to the AM structures
				amData = new AMSample[31];
				if (amData == NULL)
					throw PMemoryException();

				memset(amData, 0, 31 * sizeof(AMSample));

				// Load the AM data
				file->Seek(24 + 120, PFile::pSeekBegin);

				for (int i = 0; i < 31; i++)
				{
					AMSample *samp = &amData[i];

					samp->mark         = file->Read_B_UINT16();
					samp->pad00        = file->Read_B_UINT32();
					samp->startAmp     = file->Read_B_UINT16();
					samp->attack1Level = file->Read_B_UINT16();
					samp->attack1Speed = file->Read_B_UINT16();
					samp->attack2Level = file->Read_B_UINT16();
					samp->attack2Speed = file->Read_B_UINT16();
					samp->sustainLevel = file->Read_B_UINT16();
					samp->decaySpeed   = file->Read_B_UINT16();
					samp->sustainTime  = file->Read_B_UINT16();
					samp->pad01        = file->Read_B_UINT16();
					samp->releaseSpeed = file->Read_B_UINT16();
					samp->waveform     = file->Read_B_UINT16();
					samp->pitchFall    = file->Read_B_UINT16();
					samp->vibAmp       = file->Read_B_UINT16();
					samp->vibSpeed     = file->Read_B_UINT16();
					samp->baseFreq     = file->Read_B_UINT16();

					file->Read(samp->reserved, 84);

					if (file->IsEOF())
						break;
				}
			}
			catch(...)
			{
				CloseExtraFile(file);
				throw;
			}

			CloseExtraFile(file);
		}
		catch(...)
		{
			;
		}
	}

	return (AP_OK);
}



/******************************************************************************/
/* ShowError() opens a message window and show the error.                     */
/*                                                                            */
/* Input:  "id" is the string id to show.                                     */
/******************************************************************************/
void ModTracker::ShowError(uint32 id)
{
	PString title, msg;

	title.LoadString(strings, IDS_MOD_WIN_TITLE);
	msg.LoadString(strings, id);

	PAlert alert(title, msg, PAlert::pStop, PAlert::pOk);
	alert.Show();
}



/******************************************************************************/
/* Cleanup() frees all the memory the player have allocated.                  */
/******************************************************************************/
void ModTracker::Cleanup(void)
{
	int32 i;

	delete[] amData;
	amData = NULL;

	delete channels;
	channels = NULL;

	delete[] sequences;
	sequences = NULL;

	if (tracks != NULL)
	{
		for (i = 0; i < trackNum; i++)
			delete[] tracks[i];
	}

	delete[] tracks;
	tracks = NULL;

	if (samples != NULL)
	{
		for (i = 0; i < sampleNum; i++)
			delete[] samples[i].start;
	}

	delete[] samples;
	samples = NULL;

	// Destroy the mutex
	delete readyToPlay;
	readyToPlay = NULL;
}



/******************************************************************************/
/* InitPlayer() initialize the player.                                        */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool ModTracker::InitPlayer(void)
{
	readyToPlay = new PMutex();
	if (readyToPlay == NULL)
		return (false);

	// Initialize channel structures
	channels = new Channel[channelNum];
	if (channels == NULL)
	{
		delete readyToPlay;
		readyToPlay = NULL;

		return (false);
	}

	return (true);
}



/******************************************************************************/
/* EndPlayer() ends the use of the player.                                    */
/******************************************************************************/
void ModTracker::EndPlayer(void)
{
	Cleanup();
}



/******************************************************************************/
/* InitSound() initialize the current song.                                   */
/*                                                                            */
/* Input:  "songNum" is the subsong to play.                                  */
/******************************************************************************/
void ModTracker::InitSound(uint16 songNum)
{
	uint16 i;
	Channel *chan;

	// Start to lock the mutex
	readyToPlay->Lock();

	// Initialize all the variables
	speed        = 6;
	tempo        = 125;
	patternPos   = 0;
	counter      = 0;
	songPos      = 0;
	pBreakPos    = 0;
	posJumpFlag  = false;
	pBreakFlag   = false;
	endFlag      = false;
	lowMask      = 0xff;
	pattDelTime  = 0;
	pattDelTime2 = 0;

	for (i = 0; i < channelNum; i++)
	{
		chan = &channels[i];

		chan->trackLine.note      = 0;
		chan->trackLine.sample    = 0;
		chan->trackLine.effect    = 0;
		chan->trackLine.effectArg = 0;
		chan->start               = NULL;
		chan->length              = 0;
		chan->loopStart           = NULL;
		chan->loopLength          = 0;
		chan->startOffset         = 0;
		chan->period              = 0;
		chan->fineTune            = 0;
		chan->volume              = 0;
		chan->tonePortDirec       = 0;
		chan->tonePortSpeed       = 0;
		chan->wantedPeriod        = 0;
		chan->vibratoCmd          = 0;
		chan->vibratoPos          = 0;
		chan->tremoloCmd          = 0;
		chan->tremoloPos          = 0;
		chan->waveControl         = 0;
		chan->glissFunk           = 0;
		chan->sampleOffset        = 0;
		chan->pattPos             = 0;
		chan->loopCount           = 0;
		chan->funkOffset          = 0;
		chan->waveStart           = 0;
		chan->realLength          = 0;
		chan->pick                = 0;
		chan->amSample            = false;
		chan->amTodo              = 0;
		chan->sampleNum           = 0;
		chan->curLevel            = 0;
		chan->vibDegree           = 0;
		chan->sustainCounter      = 0;

		if (modType == modMultiTracker)
			chan->panning = panning[i] * 16;
		else
			chan->panning = 0;
	}

	// Unlock the mutex again
	readyToPlay->Unlock();
}



/******************************************************************************/
/* GetModuleName() returns the name of the module.                            */
/*                                                                            */
/* Output: Is the module name.                                                */
/******************************************************************************/
PString ModTracker::GetModuleName(void)
{
	return (songName);
}



/******************************************************************************/
/* GetVirtualChannels() returns the number of channels the module use.        */
/*                                                                            */
/* Output: Is the number of channels.                                         */
/******************************************************************************/
uint16 ModTracker::GetVirtualChannels(void)
{
	return (channelNum);
}



/******************************************************************************/
/* GetSongLength() returns the length of the current song.                    */
/*                                                                            */
/* Output: Is the length of the current song.                                 */
/******************************************************************************/
int16 ModTracker::GetSongLength(void)
{
	return (songLength);
}



/******************************************************************************/
/* GetSongPosition() returns the current position of the playing song.        */
/*                                                                            */
/* Output: Is the current position.                                           */
/******************************************************************************/
int16 ModTracker::GetSongPosition(void)
{
	return (songPos);
}



/******************************************************************************/
/* SetSongPosition() sets the current position of the playing song.           */
/*                                                                            */
/* Input:  "pos" is the new position.                                         */
/******************************************************************************/
void ModTracker::SetSongPosition(int16 pos)
{
	// Start to lock the mutex
	readyToPlay->Lock();

	// Change the position
	songPos      = pos;
	patternPos   = 0;
	pattDelTime  = 0;
	pattDelTime2 = 0;

	// Unlock the mutex again
	readyToPlay->Unlock();
}



/******************************************************************************/
/* GetInfoCount() returns the number of module info lines the player have.    */
/*                                                                            */
/* Output: The number of extra module info lines.                             */
/******************************************************************************/
uint32 ModTracker::GetInfoCount(void)
{
	return (4);
}



/******************************************************************************/
/* GetInfoString() returns the description or value string on the line given. */
/*                                                                            */
/* Input:  "line" is the line starting from 0.                                */
/*         "type" is the type of string you want.                             */
/*                                                                            */
/* Output: The string you want.                                               */
/******************************************************************************/
PString ModTracker::GetInfoString(uint32 line, ap_infoType type)
{
	PString retStr;

	if (type == apDescription)
	{
		// Description strings
		switch (line)
		{
			// Song Length
			case 0:
				retStr.LoadString(strings, IDS_MOD_INFODESCLINE0);
				break;

			// Used Patterns
			case 1:
				retStr.LoadString(strings, IDS_MOD_INFODESCLINE1);
				break;

			// Supported/Used Samples
			case 2:
				retStr.LoadString(strings, IDS_MOD_INFODESCLINE2);
				break;

			// Actual Speed (BPM)
			case 3:
				retStr.LoadString(strings, IDS_MOD_INFODESCLINE3);
				break;
		}
	}
	else
	{
		// Value strings
		switch (line)
		{
			// Song Length
			case 0:
				retStr.Format("%u", songLength);
				break;

			// Used Patterns
			case 1:
				retStr.Format("%u", maxPattern);
				break;

			// Supported/Used Samples
			case 2:
				retStr.Format("%u", sampleNum);
				break;

			// Actual Speed (BPM)
			case 3:
				retStr.Format("%u", tempo);
				break;
		}
	}

	return (retStr);
}



/******************************************************************************/
/* GetSampleCount() returns the number of samples the module have.            */
/*                                                                            */
/* Output: The number of samples.                                             */
/******************************************************************************/
uint32 ModTracker::GetSampleCount(void)
{
	return (sampleNum);
}



/******************************************************************************/
/* GetSampleInfo() fills out the APInstInfo structure given with the sample   */
/*      information of the sample number given.                               */
/*                                                                            */
/* Input:  "num" is the sample number starting from 0.                        */
/*         "info" is a pointer to an APSampleInfo structure to fill out.      */
/******************************************************************************/
void ModTracker::GetSampleInfo(uint32 num, APSampleInfo *info)
{
	Sample *sample;

	// Get the pointer to the sample data
	sample = &samples[num];

	// Find out the type of the sample
	info->type = apSample;
	if (amData != NULL)
	{
		if (amData[num].mark == 'AM')
			info->type = apAM;
	}

	// Fill out the sample info structure
	info->name       = sample->sampleName;
	info->flags      = (sample->loopLength <= 1 ? 0 : APSAMP_LOOP);
	info->bitSize    = 8;
	info->middleC    = (uint32)(7093789.2 / (periods[sample->fineTune][3 * 12] * 2));
	info->volume     = sample->volume * 4;
	info->panning    = -1;
	info->address    = sample->start;
	info->length     = sample->length * 2;
	info->loopStart  = sample->loopStart * 2;
	info->loopLength = sample->loopLength * 2;
}



/******************************************************************************/
/* Play() is the main player function.                                        */
/******************************************************************************/
void ModTracker::Play(void)
{
	if (readyToPlay->Lock(0) == pSyncOk)
	{
		if (speed)					// Only play if speed <> 0
		{
			counter++;				// Count speed counter
			if (counter >= speed)	// Do we have to change pattern line?
			{
				counter = 0;
				if (pattDelTime2)	// Pattern delay active
					NoNewAllChan();
				else
					GetNewNote();

				// Get next pattern line
				patternPos++;
				if (pattDelTime)	// New pattern delay time
				{
					// Activate the pattern delay
					pattDelTime2 = pattDelTime;
					pattDelTime  = 0;
				}

				// Pattern delay routine, jump one line back again
				if (pattDelTime2)
				{
					if (--pattDelTime2)
						patternPos--;
				}

				// Pattern break
				if (pBreakFlag)
				{
					// Calculate new position in the next pattern
					pBreakFlag = false;
					patternPos = pBreakPos;
					pBreakPos  = 0;
				}

				// Has the module ended
				if (endFlag)
				{
					if (!posJumpFlag)
						endReached = true;

					endFlag = false;
				}

				// Have we played the whole pattern?
				if (patternPos >= patternLength)
					NextPos();
			}
			else
				NoNewAllChan();

			if (posJumpFlag)
				NextPos();
		}
		else
		{
			NoNewAllChan();

			if (posJumpFlag)
				NextPos();
		}

		if (modType == modStarTrekker)
			AMHandler();

		// Unlock the mutex
		readyToPlay->Unlock();
	}
}



/******************************************************************************/
/* NextPos() jumps to the next song postion.                                  */
/******************************************************************************/
void ModTracker::NextPos(void)
{
	// Initialize the position variables
	patternPos  = pBreakPos;
	pBreakPos   = 0;
	posJumpFlag = false;
	songPos    += 1;
	songPos    &= 0x7f;

	if (songPos >= songLength)
	{
		songPos = restartPos;

		// Position has changed
		ChangePosition();

		// And the module has repeated
		endReached = true;
	}
	else
	{
		// Position has changed
		ChangePosition();
	}
}



/******************************************************************************/
/* NoNewAllChan() checks all channels to see if any commands should run.      */
/******************************************************************************/
void ModTracker::NoNewAllChan(void)
{
	uint16 i;

	for (i = 0; i < channelNum; i++)
		CheckEFX(virtChannels[i], channels[i]);
}



/******************************************************************************/
/* GetNewNote() parses the next pattern line.                                 */
/******************************************************************************/
void ModTracker::GetNewNote(void)
{
	uint16 i;
	uint16 curSongPos, curPattPos;
	uint16 seqNum;
	TrackLine *trackData;

	// Get position information into temporary variables
	curSongPos = songPos;
	curPattPos = patternPos;

	for (i = 0; i < channelNum; i++)
	{
		// Find the track to use
		seqNum    = sequences[positions[curSongPos] * 32 + i];
		trackData = tracks[seqNum] + curPattPos;

		PlayVoice(trackData, virtChannels[i], channels[i]);
	}
}



/******************************************************************************/
/* PlayVoice() parses one pattern line for one channel.                       */
/*                                                                            */
/* Input:  "trackData" is a pointer to the pattern line.                      */
/*         "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::PlayVoice(TrackLine *trackData, APChannel *apChan, Channel &chan)
{
	Sample *sample;
	AMSample *amSamp;
	uint8 sampNum, cmd;

	// Check for any note or effect running
	if ((trackData->note == 0) && (trackData->sample == 0) && (trackData->effect == 0) && (trackData->effectArg == 0))
	{
		// Nothing runs, so set the period
		apChan->SetAmigaPeriod(chan.period);
	}

	// Copy pattern line to fields in our channel structure
	chan.trackLine = *trackData;

	sampNum = trackData->sample;
	if (sampNum)
	{
		// New sample
		sample = &samples[sampNum - 1];

		chan.sampleNum   = sampNum;
		chan.amSample    = false;

		if (modType == modStarTrekker)
		{
			if (amData != NULL)
			{
				amSamp = &amData[sampNum - 1];

				if (amSamp->mark == 'AM')
				{
					chan.volume   = amSamp->startAmp / 4;
					chan.amSample = true;
				}
			}
		}

		chan.start       = sample->start;
		chan.length      = sample->length;
		chan.realLength  = chan.length;
		chan.startOffset = 0;
		chan.fineTune    = sample->fineTune;

		if (!chan.amSample)
			chan.volume = sample->volume;

		// Check to see if we got a loop
		if ((sample->loopStart) && (sample->loopLength > 1))
		{
			// We have, now check the player mode
			if ((modType == modSoundTracker15) || (modType == modSoundTracker31))
			{
				chan.start     += sample->loopStart;
				chan.loopStart  = chan.start;
				chan.waveStart  = chan.start;
				chan.length     = sample->loopLength;
				chan.loopLength = chan.length;
			}
			else
			{
				chan.loopStart  = chan.start + sample->loopStart * 2;
				chan.waveStart  = chan.loopStart;

				chan.length     = sample->loopStart + sample->loopLength;
				chan.loopLength = sample->loopLength;
			}
		}
		else
		{
			// No loop
			chan.loopStart  = chan.start + sample->loopStart;
			chan.waveStart  = chan.loopStart;
			chan.loopLength = sample->loopLength;
		}

		// Set volume
		if (!chan.amSample)
			apChan->SetVolume(chan.volume * 4);

		// Set panning
		if (modType == modMultiTracker)
			apChan->SetPanning(chan.panning);
	}

	// Check for some commands
	if (chan.trackLine.note)
	{
		// There is a new note to play
		cmd = chan.trackLine.effect;

		if (!chan.amSample)
		{
			// Check for SetFineTune
			if ((cmd == effExtraEffect) && ((chan.trackLine.effectArg & 0xf0) == effSetFineTune))
				SetFineTune(chan);
			else
			{
				switch (cmd)
				{
					case effTonePortamento:
					case effTonePort_VolSlide:
						SetTonePorta(chan);
						CheckMoreEFX(apChan, chan);
						return;

					case effSampleOffset:
						CheckMoreEFX(apChan, chan);
						break;
				}
			}
		}

		// Set the period
		chan.period = periods[chan.fineTune][chan.trackLine.note - 1];

		if (!((cmd == effExtraEffect) && ((chan.trackLine.effectArg & 0xf0) == effNoteDelay)))
		{
			if (!(chan.waveControl & 4))
				chan.vibratoPos = 0;

			if (!(chan.waveControl & 64))
				chan.tremoloPos = 0;

			if (chan.amSample)
			{
				// Setup AM sample
				amSamp = &amData[chan.sampleNum - 1];

				chan.start       = &amWaveforms[amSamp->waveform][0];
				chan.startOffset = 0;
				chan.length      = 16;
				chan.loopStart   = chan.start;
				chan.loopLength  = 16;

				chan.amTodo      = 1;
				chan.curLevel    = amSamp->startAmp;
				chan.vibDegree   = 0;
				chan.period      = chan.period << amSamp->baseFreq;
			}

			// Fill out the Channel
			if (chan.length > 0)
			{
				apChan->PlaySample(chan.start, chan.startOffset * 2, chan.length * 2);
				apChan->SetAmigaPeriod(chan.period);

				// Setup loop
				if (chan.loopLength > 0)
					apChan->SetLoop(chan.loopStart - chan.start, chan.loopLength * 2);
			}
			else
				apChan->Mute();
		}
	}

	CheckMoreEFX(apChan, chan);
}



/******************************************************************************/
/* CheckEFX() check one channel to see if there are some commands to run.     */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::CheckEFX(APChannel *apChan, Channel &chan)
{
	uint8 cmd;

	UpdateFunk(chan);

	cmd = chan.trackLine.effect;
	if ((!cmd) && (!(chan.trackLine.effectArg)))
		apChan->SetAmigaPeriod(chan.period);
	else
	{
		switch (cmd)
		{
			case effArpeggio:
				Arpeggio(apChan, chan);			// Arpeggio or normal note
				break;

			case effSlideUp:
				PortUp(apChan, chan);
				break;

			case effSlideDown:
				PortDown(apChan, chan);
				break;

			case effTonePortamento:
				TonePortamento(apChan, chan);
				break;

			case effVibrato:
				Vibrato(apChan, chan);
				break;

			case effTonePort_VolSlide:
				TonePlusVol(apChan, chan);
				break;

			case effVibrato_VolSlide:
				VibratoPlusVol(apChan, chan);
				break;

			case effExtraEffect:
				ECommands(apChan, chan);
				break;

			default:
				apChan->SetAmigaPeriod(chan.period);

				if (cmd == effTremolo)
					Tremolo(apChan, chan);
				else
					if (cmd == effVolumeSlide)
						VolumeSlide(apChan, chan);
				break;
		}
	}
}



/******************************************************************************/
/* CheckMoreEFX() check one channel to see if there are some commands to run. */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::CheckMoreEFX(APChannel *apChan, Channel &chan)
{
	if (modType != modMultiTracker)
	{
		switch (chan.trackLine.effect)
		{
			case effSampleOffset:
				SampleOffset(chan);
				break;

			case effPosJump:
				PositionJump(chan);
				break;

			case effSetVolume:
				VolumeChange(apChan, chan);
				break;

			case effPatternBreak:
				PatternBreak(chan);
				break;

			case effExtraEffect:
				ECommands(apChan, chan);
				break;

			case effSetSpeed:
				SetSpeed(chan);
				break;

			default:
				apChan->SetAmigaPeriod(chan.period);
				break;
		}
	}
	else
	{
		switch (chan.trackLine.effect)
		{
			case effSampleOffset:
				SampleOffset(chan);
				break;

			case effPosJump:
				PositionJump(chan);
				break;

			case effSetVolume:
				VolumeChange(apChan, chan);
				break;

			case effPatternBreak:
				PatternBreak(chan);
				break;

			case effExtraEffect:
				ECommands(apChan, chan);
				break;

			case effSetSpeed:
				SetSpeed(chan);
				break;

			case effSlideUp:
				PortUp(apChan, chan);
				break;

			case effSlideDown:
				PortDown(apChan, chan);
				break;

			case effTonePortamento:
				TonePortamento(apChan, chan);
				break;

			case effVibrato:
				Vibrato(apChan, chan);
				break;

			case effTonePort_VolSlide:
				TonePlusVol(apChan, chan);
				break;

			case effVibrato_VolSlide:
				VibratoPlusVol(apChan, chan);
				break;

			default:
				apChan->SetAmigaPeriod(chan.period);
				break;
		}
	}
}



/******************************************************************************/
/* ECommands() check one channel to see if there are some of the extra        */
/*         commands to run.                                                   */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::ECommands(APChannel *apChan, Channel &chan)
{
	if ((modType == modSoundTracker15) || (modType == modSoundTracker31) || (modType == modNoiseTracker) || (modType == modStarTrekker))
		Filter(chan);
	else
	{
		switch (chan.trackLine.effectArg & 0xf0)
		{
			case effSetFilter:
				Filter(chan);
				break;

			case effFineSlideUp:
				FinePortUp(apChan, chan);
				break;

			case effFineSlideDown:
				FinePortDown(apChan, chan);
				break;

			case effGlissandoCtrl:
				SetGlissCon(chan);
				break;

			case effVibratoWaveform:
				SetVibCon(chan);
				break;

			case effSetFineTune:
				SetFineTune(chan);
				break;

			case effJumpToLoop:
				JumpLoop(chan);
				break;

			case effTremoloWaveform:
				SetTreCon(chan);
				break;

			case effRetrig:
				RetrigNote(apChan, chan);
				break;

			case effFineVolSlideUp:
				VolumeFineUp(apChan, chan);
				break;

			case effFineVolSlideDown:
				VolumeFineDown(apChan, chan);
				break;

			case effNoteCut:
				NoteCut(apChan, chan);
				break;

			case effNoteDelay:
				NoteDelay(apChan, chan);
				break;

			case effPatternDelay:
				PatternDelay(chan);
				break;

			case effInvertLoop:
				FunkIt(chan);
				break;
		}
	}
}



/******************************************************************************/
/* SetTonePorta() sets the portamento frequency.                              */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SetTonePorta(Channel &chan)
{
	uint16 period, i;

	period = periods[0][chan.trackLine.note - 1];

	for (i = 0; i < NumberOfNotes; i++)
		if (periods[chan.fineTune][i] <= period)
		{
			i++;
			break;
		}

	// Decrement counter so it have the right value.
	// This is because if the loop goes all the way through.
	i--;

	if ((chan.fineTune > 7) && (i != 0))
		i--;

	period = periods[chan.fineTune][i];

	chan.wantedPeriod  = period;
	chan.tonePortDirec = 0;

	if (chan.period == period)
		chan.wantedPeriod = 0;
	else
		if (chan.period > period)
			chan.tonePortDirec = 1;
}



/******************************************************************************/
/* UpdateFunk() updates funk?                                                 */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::UpdateFunk(Channel &chan)
{
	uint8 glissFunk;
	int8 *sample;

	glissFunk = chan.glissFunk >> 4;
	if (glissFunk)
	{
		chan.funkOffset += funkTable[glissFunk];
		if (chan.funkOffset >= 128)
		{
			chan.funkOffset = 0;

			sample = chan.waveStart + 1;
			if (sample >= (chan.loopStart + chan.loopLength * 2))
				sample = chan.loopStart;

			chan.waveStart = sample;

			// Invert the sample data
			*sample = ~*sample;
		}
	}
}



/******************************************************************************/
/*                                                                            */
/* Below are the functions to all the normal effects.                         */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/* Arpeggio() (0x00) plays arpeggio or normal note.                           */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::Arpeggio(APChannel *apChan, Channel &chan)
{
	uint16 period, i;
	uint8 modulus, arp;

	if (chan.trackLine.effectArg)
	{
		modulus = counter % 3;

		switch (modulus)
		{
			case 1:
				arp = chan.trackLine.effectArg >> 4;
				break;

			case 2:
				arp = chan.trackLine.effectArg & 0x0f;
				break;

			default:
				arp = 0;
		}

		// Find the index into the period tables
		for (i = 0; i < NumberOfNotes; i++)
			if (periods[chan.fineTune][i] <= chan.period)
				break;

		// Get the period
		period = periods[chan.fineTune][i + arp];
	}
	else
	{
		// Normal note
		period = chan.period;
	}

	// Setup the NotePlayer registers
	apChan->SetAmigaPeriod(period);
}



/******************************************************************************/
/* PortUp() (0x01) slides the frequency up.                                   */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::PortUp(APChannel *apChan, Channel &chan)
{
	chan.period -= (chan.trackLine.effectArg & lowMask);
	if (chan.period < minPeriod)
		chan.period = minPeriod;

	lowMask = 0xff;

	apChan->SetAmigaPeriod(chan.period);
}



/******************************************************************************/
/* PortDown() (0x02) slides the frequency down.                               */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::PortDown(APChannel *apChan, Channel &chan)
{
	chan.period += (chan.trackLine.effectArg & lowMask);
	if (chan.period > maxPeriod)
		chan.period = maxPeriod;

	lowMask = 0xff;

	apChan->SetAmigaPeriod(chan.period);
}



/******************************************************************************/
/* TonePortamento() (0x03) slides the frequency to the current note.          */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/*         "skip" is true if it has to skip the initializing.                 */
/******************************************************************************/
void ModTracker::TonePortamento(APChannel *apChan, Channel &chan, bool skip)
{
	int32 period;
	uint16 i;

	if (!skip)
	{
		if (chan.trackLine.effectArg)
		{
			// Set the slide speed
			chan.tonePortSpeed       = chan.trackLine.effectArg;
			chan.trackLine.effectArg = 0;
		}
	}

	// If slide mode enabled?
	if (chan.wantedPeriod)
	{
		if (chan.tonePortDirec)
		{
			// Slide up
			period = chan.period - chan.tonePortSpeed;
			if (chan.wantedPeriod >= period)
			{
				// Set to the final period and disable slide
				period            = chan.wantedPeriod;
				chan.wantedPeriod = 0;
			}

			chan.period = period;
		}
		else
		{
			// Slide down
			period = chan.period + chan.tonePortSpeed;
			if (chan.wantedPeriod <= period)
			{
				// Set to final period and disable slide
				period            = chan.wantedPeriod;
				chan.wantedPeriod = 0;
			}

			chan.period = period;
		}

		// Is glissando enabled?
		if (chan.glissFunk & 0x0f)
		{
			for (i = 0; i < NumberOfNotes; i++)
				if (periods[chan.fineTune][i] <= period)
				{
					i++;
					break;
				}

			period = periods[chan.fineTune][i - 1];
		}

		// Setup the NotePlayer structure
		apChan->SetAmigaPeriod(period);
	}
}



/******************************************************************************/
/* Vibrato() (0x04) vibrates the frequency.                                   */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/*         "skip" is true if it has to skip the initializing.                 */
/******************************************************************************/
void ModTracker::Vibrato(APChannel *apChan, Channel &chan, bool skip)
{
	uint8 effArg, vibCmd, vibPos, waveCtrl, addVal;
	uint16 period;

	// Get the effect argument
	effArg = chan.trackLine.effectArg;

	// Setup vibrato command
	if ((!skip) && (effArg))
	{
		vibCmd = chan.vibratoCmd;

		if (effArg & 0x0f)
			vibCmd = (vibCmd & 0xf0) | (effArg & 0x0f);

		if (effArg & 0xf0)
			vibCmd = (vibCmd & 0x0f) | (effArg & 0xf0);

		chan.vibratoCmd = vibCmd;
	}

	// Calculate new position
	vibPos   = (chan.vibratoPos / 4) & 0x1f;
	waveCtrl = chan.waveControl & 0x03;

	if (waveCtrl)
	{
		vibPos *= 8;
		if (waveCtrl != 1)
			addVal = 255;					// Square vibrato
		else
		{
			// Ramp down vibrato
			if (chan.vibratoPos < 0)
				addVal = 255 - vibPos;
			else
				addVal = vibPos;
		}
	}
	else
	{
		// Sine vibrato
		addVal = vibratoTable[vibPos];
	}

	// Set the vibrato
	addVal = addVal * (chan.vibratoCmd & 0x0f) / 128;
	period = chan.period;

	if (chan.vibratoPos < 0)
		period -= addVal;
	else
		period += addVal;

	// Put the new period into the NotePlayer structure
	apChan->SetAmigaPeriod(period);

	chan.vibratoPos += ((chan.vibratoCmd / 4) & 0x3c);
}



/******************************************************************************/
/* TonePlusVol() (0x05) is both effect 0x03 and 0x0a.                         */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::TonePlusVol(APChannel *apChan, Channel &chan)
{
	TonePortamento(apChan, chan, true);
	VolumeSlide(apChan, chan);
}



/******************************************************************************/
/* VibratoPlusVol() (0x06) is both effect 0x04 and 0x0a.                      */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::VibratoPlusVol(APChannel *apChan, Channel &chan)
{
	Vibrato(apChan, chan, true);
	VolumeSlide(apChan, chan);
}



/******************************************************************************/
/* Tremolo() (0x07) makes vibrato on the volume.                              */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::Tremolo(APChannel *apChan, Channel &chan)
{
	uint8 effArg, treCmd, trePos, waveCtrl, addVal;
	int16 volume;

	// Get the effect argument
	effArg = chan.trackLine.effectArg;

	// Setup tremolo command
	if (effArg)
	{
		treCmd = chan.tremoloCmd;

		if (effArg & 0x0f)
			treCmd = (treCmd & 0xf0) | (effArg & 0x0f);

		if (effArg & 0xf0)
			treCmd = (treCmd & 0x0f) | (effArg & 0xf0);

		chan.tremoloCmd = treCmd;
	}

	// Calculate new position
	trePos   = (chan.tremoloPos / 4) & 0x1f;
	waveCtrl = (chan.waveControl >> 4) & 0x03;

	if (waveCtrl)
	{
		trePos *= 8;
		if (waveCtrl != 1)
			addVal = 255;					// Square tremolo
		else
		{
			// Ramp down tremolo
			if (chan.tremoloPos < 0)
				addVal = 255 - trePos;
			else
				addVal = trePos;
		}
	}
	else
	{
		// Sine tremolo
		addVal = vibratoTable[trePos];
	}

	// Set the tremolo
	addVal = addVal * (chan.tremoloCmd & 0x0f) / 64;
	volume = chan.volume;

	if (chan.tremoloPos < 0)
	{
		volume -= addVal;
		if (volume < 0)
			volume = 0;
	}
	else
	{
		volume += addVal;
		if (volume > 64)
			volume = 64;
	}

	// Put the new volume into the NotePlayer structure
	apChan->SetVolume(volume * 4);

	chan.tremoloPos += ((chan.tremoloCmd / 4) & 0x3c);
}



/******************************************************************************/
/* SampleOffset() (0x09) starts the sample somewhere else, but the start.     */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SampleOffset(Channel &chan)
{
	uint16 offset;

	// Check for initialize value
	if (chan.trackLine.effectArg)
		chan.sampleOffset = chan.trackLine.effectArg;

	// Calculate the offset
	offset = chan.sampleOffset * 128;
	if (offset < chan.length)
		chan.startOffset = offset;
	else
	{
		chan.length      = chan.loopLength;
		chan.start       = chan.loopStart;
		chan.startOffset = 0;
	}
}



/******************************************************************************/
/* VolumeSlide() (0x0A) slides the volume.                                    */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::VolumeSlide(APChannel *apChan, Channel &chan)
{
	uint8 speed;

	speed = chan.trackLine.effectArg >> 4;
	if (speed)
	{
		// Slide up
		chan.volume += speed;
		if (chan.volume > 64)
			chan.volume = 64;
	}
	else
	{
		// Slide down
		chan.volume -= (chan.trackLine.effectArg & 0x0f);
		if (chan.volume < 0)
			chan.volume = 0;
	}

	// Set the volume in the NotePlayer structure
	apChan->SetVolume(chan.volume * 4);
}



/******************************************************************************/
/* PositionJump() (0x0B) jumps to another position.                           */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::PositionJump(Channel &chan)
{
	uint8 pos;

	pos = chan.trackLine.effectArg;
	if (pos < songPos)
		endReached = true;	// Module has repeated

	if (pos == songPos)
		endFlag = true;		// Module jump to the same position

	// Set the new position
	songPos     = pos - 1;
	pBreakPos   = 0;
	posJumpFlag = true;
}



/******************************************************************************/
/* VolumeChange() (0x0C) sets the sample volume.                              */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::VolumeChange(APChannel *apChan, Channel &chan)
{
	uint8 vol;

	vol = chan.trackLine.effectArg;
	if (vol > 64)
		vol = 64;

	chan.volume = vol;

	// Set the volume in the NotePlayer structure
	apChan->SetVolume(vol * 4);
}



/******************************************************************************/
/* PatternBreak() (0x0D) breaks the pattern and jump to the next position.    */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::PatternBreak(Channel &chan)
{
	uint8 arg;

	arg = chan.trackLine.effectArg;

	pBreakPos   = ((arg >> 4) & 0x0f) * 10 + (arg & 0x0f);
	posJumpFlag = true;
}



/******************************************************************************/
/* SetSpeed() (0x0F) changes the speed of the module.                         */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SetSpeed(Channel &chan)
{
	uint8 newSpeed;

	// Get the new speed
	newSpeed = chan.trackLine.effectArg;

	if ((modType == modSoundTracker15) || (modType == modSoundTracker31) || (modType == modNoiseTracker) || (modType == modStarTrekker))
	{
		// Old trackers
		if (newSpeed == 0)
			newSpeed = 1;

		if (newSpeed > 31)
			newSpeed = 31;

		// Set the new speed
		speed   = newSpeed;
		counter = 0;
	}
	else
	{
		// New trackers
		if (newSpeed == 0)
		{
			// Set the new speed
			speed         = 6;
			counter       = 0;
			pBreakPos     = 0;
			posJumpFlag   = true;

			// Module has stopped
			endReached = true;

			if (tempo != 125)
			{
				PString value;

				// BPM speed
				tempo = 125;
				SetBPMTempo(125);

				// Change the module info
				value.Format("%u", newSpeed);
				ChangeModuleInfo(3, apValue, value);
			}
		}
		else
		{
			if (newSpeed > 31)
			{
				if (newSpeed != tempo)
				{
					PString value;

					// BPM speed
					SetBPMTempo(newSpeed);

					// Change the module info
					value.Format("%u", newSpeed);
					ChangeModuleInfo(3, apValue, value);
				}
			}
			else
			{
				// Set the new speed
				speed   = newSpeed;
				counter = 0;
			}
		}
	}
}



/******************************************************************************/
/*                                                                            */
/* Below are the functions to all the extended effects.                       */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/* Filter() (0xE0) changes the filter.                                        */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::Filter(Channel &/*chan*/)
{
}



/******************************************************************************/
/* FinePortUp() (0xE1) fine slide the frequency up.                           */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::FinePortUp(APChannel *apChan, Channel &chan)
{
	if (counter == 0)
	{
		lowMask = 0x0f;
		PortUp(apChan, chan);
	}
}



/******************************************************************************/
/* FinePortDown() (0xE2) fine slide the frequency down.                       */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::FinePortDown(APChannel *apChan, Channel &chan)
{
	if (counter == 0)
	{
		lowMask = 0x0f;
		PortDown(apChan, chan);
	}
}



/******************************************************************************/
/* SetGlissCon() (0xE3) sets a new glissando control.                         */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SetGlissCon(Channel &chan)
{
	chan.glissFunk &= 0x0f;
	chan.glissFunk |= (chan.trackLine.effectArg & 0x0f);
}



/******************************************************************************/
/* SetVibCon() (0xE4) sets a new vibrato waveform.                            */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SetVibCon(Channel &chan)
{
	chan.waveControl &= 0xf0;
	chan.waveControl |= (chan.trackLine.effectArg & 0x0f);
}



/******************************************************************************/
/* SetFineTune() (0xE5) changes the finetune.                                 */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SetFineTune(Channel &chan)
{
	chan.fineTune = chan.trackLine.effectArg & 0x0f;
}



/******************************************************************************/
/* JumpLoop() (0xE6) jump to pattern loop position.                           */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::JumpLoop(Channel &chan)
{
	uint8 arg;

	if (counter == 0)
	{
		arg = chan.trackLine.effectArg & 0x0f;

		if (arg)
		{
			// Jump to the loop currently set
			if (chan.loopCount == 0)
				chan.loopCount = arg;
			else
				chan.loopCount--;

			if (chan.loopCount != 0)
			{
				pBreakPos  = chan.pattPos;
				pBreakFlag = true;
			}
		}
		else
		{
			// Set the loop start point
			chan.pattPos = patternPos;
		}
	}
}



/******************************************************************************/
/* SetTreCon() (0xE7) sets a new tremolo waveform.                            */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::SetTreCon(Channel &chan)
{
	chan.waveControl &= 0x0f;
	chan.waveControl |= ((chan.trackLine.effectArg & 0x0f) << 4);
}



/******************************************************************************/
/* RetrigNote() (0xE9) retrigs the current note.                              */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::RetrigNote(APChannel *apChan, Channel &chan)
{
	uint8 arg;

	arg = chan.trackLine.effectArg & 0x0f;

	if (arg)
	{
		if (!((counter == 0) && (chan.trackLine.note)))
		{
			if ((counter % arg) == 0)
			{
				// Retrig the sample
				if (chan.length)
				{
					apChan->PlaySample(chan.start, 0, chan.length * 2);

					if (chan.loopLength)
						apChan->SetLoop(chan.loopStart - chan.start, chan.loopLength * 2);

					apChan->SetAmigaPeriod(chan.period);
				}
				else
					apChan->Mute();
			}
		}
	}
}



/******************************************************************************/
/* VolumeFineUp() (0xEA) fine slide the volume up.                            */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::VolumeFineUp(APChannel *apChan, Channel &chan)
{
	if (counter == 0)
	{
		chan.volume += (chan.trackLine.effectArg & 0x0f);
		if (chan.volume > 64)
			chan.volume = 64;
		
		// Set the volume in the NotePlayer structure
		apChan->SetVolume(chan.volume * 4);
	}
}



/******************************************************************************/
/* VolumeFineDown() (0xEB) fine slide the volume down.                        */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::VolumeFineDown(APChannel *apChan, Channel &chan)
{
	if (counter == 0)
	{
		chan.volume -= (chan.trackLine.effectArg & 0x0f);
		if (chan.volume < 0)
			chan.volume = 0;
	}

	// Set the volume in the NotePlayer structure
	apChan->SetVolume(chan.volume * 4);
}



/******************************************************************************/
/* NoteCut() (0xEC) stops the current note for playing.                       */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::NoteCut(APChannel *apChan, Channel &chan)
{
	if ((chan.trackLine.effectArg & 0x0f) == counter)
	{
		chan.volume = 0;
		apChan->SetVolume(0);
	}
}



/******************************************************************************/
/* NoteDelay() (0xED) waits a little while before playing.                    */
/*                                                                            */
/* Input:  "apChan" is a pointer to the Channel object.                       */
/*         "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::NoteDelay(APChannel *apChan, Channel &chan)
{
	if (((chan.trackLine.effectArg & 0x0f) == counter) && (chan.trackLine.note))
	{
		// Retrig the sample
		if (chan.length)
		{
			apChan->PlaySample(chan.start, 0, chan.length * 2);

			if (chan.loopLength)
				apChan->SetLoop(chan.loopStart - chan.start, chan.loopLength * 2);

			apChan->SetAmigaPeriod(chan.period);
		}
		else
			apChan->Mute();
	}
}



/******************************************************************************/
/* PatternDelay() (0xEE) pauses the pattern for a little while.               */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::PatternDelay(Channel &chan)
{
	if ((counter == 0) && (pattDelTime2 == 0))
		pattDelTime = (chan.trackLine.effectArg & 0x0f) + 1;
}



/******************************************************************************/
/* FunkIt() (0xEF) inverts the loop.                                          */
/*                                                                            */
/* Input:  "chan" is a reference to the channel structure.                    */
/******************************************************************************/
void ModTracker::FunkIt(Channel &chan)
{
	if (counter == 0)
	{
		chan.glissFunk &= 0x0f;
		chan.glissFunk |= (chan.trackLine.effectArg & 0x0f) << 4;

		if (chan.glissFunk)
			UpdateFunk(chan);
	}
}



/******************************************************************************/
/* AMHandler() handles StarTrekker AM samples.                                */
/******************************************************************************/
void ModTracker::AMHandler(void)
{
	AMSample *amSamp;
 	Channel *chan;
 	uint16 i, degree;
 	int32 vibVal;
 	bool flag;

	for (i = 0; i < channelNum; i++)
	{
		chan = &channels[i];

		if (chan->amSample)
		{
			amSamp = &amData[chan->sampleNum - 1];

			switch (chan->amTodo)
			{
				// Do attack 1
				case 1:
					if (chan->curLevel == amSamp->attack1Level)
						chan->amTodo = 2;
					else
					{
						if (chan->curLevel < amSamp->attack1Level)
						{
							chan->curLevel += amSamp->attack1Speed;
							if (chan->curLevel >= amSamp->attack1Level)
							{
								chan->curLevel = amSamp->attack1Level;
								chan->amTodo   = 2;
							}
						}
						else
						{
							chan->curLevel -= amSamp->attack1Speed;
							if (chan->curLevel <= amSamp->attack1Level)
							{
								chan->curLevel = amSamp->attack1Level;
								chan->amTodo   = 2;
							}
						}
					}
					break;

				// Do attack 2
				case 2:
					if (chan->curLevel == amSamp->attack2Level)
						chan->amTodo = 3;
					else
					{
						if (chan->curLevel < amSamp->attack2Level)
						{
							chan->curLevel += amSamp->attack2Speed;
							if (chan->curLevel >= amSamp->attack2Level)
							{
								chan->curLevel = amSamp->attack2Level;
								chan->amTodo   = 3;
							}
						}
						else
						{
							chan->curLevel -= amSamp->attack2Speed;
							if (chan->curLevel <= amSamp->attack2Level)
							{
								chan->curLevel = amSamp->attack2Level;
								chan->amTodo   = 3;
							}
						}
					}
					break;

				// Do sustain
				case 3:
					if (chan->curLevel == amSamp->sustainLevel)
						chan->amTodo = 4;
					else
					{
						if (chan->curLevel < amSamp->sustainLevel)
						{
							chan->curLevel += amSamp->decaySpeed;
							if (chan->curLevel >= amSamp->sustainLevel)
							{
								chan->curLevel = amSamp->sustainLevel;
								chan->amTodo   = 4;
							}
						}
						else
						{
							chan->curLevel -= amSamp->decaySpeed;
							if (chan->curLevel <= amSamp->sustainLevel)
							{
								chan->curLevel = amSamp->sustainLevel;
								chan->amTodo   = 4;
							}
						}
					}
					break;

				// Do sustain delay
				case 4:
					chan->sustainCounter--;
					if (chan->sustainCounter < 0)
						chan->amTodo = 5;
					break;

				// Do release
				case 5:
					chan->curLevel -= amSamp->releaseSpeed;
					if (chan->curLevel <= 0)
					{
						chan->amTodo   = 0;
						chan->curLevel = 0;
						chan->amSample = false;
					}
					break;
			}

			// Set the volume
			virtChannels[i]->SetVolume(chan->curLevel);

			// Do pitch fall
			chan->period += amSamp->pitchFall;

			// Do vibrato
			vibVal = amSamp->vibAmp;
			if (vibVal)
			{
				flag   = false;
				degree = chan->vibDegree;
				if (degree >= 180)
				{
					degree -= 180;
					flag    = true;
				}

				vibVal = amSinus[degree] * amSamp->vibAmp / 128;
				if (flag)
					vibVal = -vibVal;
			}

			// Set new frequency
			virtChannels[i]->SetAmigaPeriod(chan->period + vibVal);

			chan->vibDegree += amSamp->vibSpeed;
			if (chan->vibDegree >= 360)
				chan->vibDegree -= 360;
		}
	}

	// Generate noise waveform
	for (i = 0; i < 32; i++)
		amWaveforms[3][i] = Random(255);
}
