//
// Implementation of ControlWindow class
//
// this belongs to the 'Tower of Benoi' Project
// Copyright 1998
//
// Written by: Marco Zinn
//

#include "Towers.h" // we need some of the global stuff
#include "ControlWindow.h"
#include "TowerWindow.h" // we need to know, how the TowerWindow is defined

// The constructor sets up the Window
//  This includes the Slides and the Button
ControlWindow::ControlWindow(BRect frame, const char* name)
			: BWindow(frame, name,B_TITLED_WINDOW,B_NOT_ZOOMABLE|B_NOT_RESIZABLE|B_WILL_ACCEPT_FIRST_CLICK|B_NOT_CLOSABLE) {

	// Create Slider. Note that r.bottom is set equal to r.top
	//  We can do this, because it's ignored for BSliders.
	BRect r(10,20,Bounds().Width()-10,20);
	
	// Create View with Hashmarks and labels
	SpeedSlider=new BSlider(r,"speed","Solving Speed",new BMessage(),0,100);
	SpeedSlider->SetHashMarks(B_HASH_MARKS_BOTTOM);
	SpeedSlider->SetLimitLabels("Slow", "Fast");
	SpeedSlider->SetValue(50); // Set it to the middle position
	AddChild(SpeedSlider); // and add it to the window
	
	r.OffsetBy(0,60); // go some pixels lower
	AddChild(new BButton(r,"stop","Stop that!",new BMessage(STOP_PRESSED)));
		// Create and add button including the model message

	StopFlag=FALSE; // Don't stop the solving (hey, we didn't start yet!)
//	Show();
	// Note that the window is not shown by default!
}


// MoveTower
//  Moves a Tower with NumDiscs(N) from FromTower(F) to ToTower(T) using BufferTower(B)
//  This is done recursivly:
//   a) Move (N-1) Discs from F to B
//   b) Move the remaining Disc from F to T
//   c) Move (N-1) Discs from B to T (the discs moved in a) )
//  That's all. This solves every Tower from the starting position on.
//  Have a good look at the move history window (or use the printf's) to understand it.
void ControlWindow::MoveTower(DiscTowerView *FromTower, DiscTowerView *ToTower, DiscTowerView *BufferTower, int NumDiscs) {
	// printf("Entering MoveTower: I shall move %i Discs\n",NumDiscs); // Debug

	// First, check the condition for the recursion to stop:
	//  MoveTower has been called in order to move '0' Discs
	if (NumDiscs==0) { 
		// printf(" Nothing to do, back out.\n"); // Debug
		return;
		}
	

	if (FromTower->TowerEmpty()) { 
		printf("ERROR: Tower is empty!\n");
		return; // Error-Check
		 }

	// printf("-Calling MoveTower\n"); // Debug
	
	//   a) Move (N-1) Discs from F to B (using T as Buffer)
	MoveTower(FromTower, BufferTower, ToTower, NumDiscs-1);

	// Check if the solving must be stopped.
	//  Stop, when the StopFlag is on (true).
	if (StopFlag) { printf("!Stop requested.\n"); return;}

	//  Also stop, when the window has been closed (deleted) in the meantime.
		// printf("  Window: %p \n  Towers: %p,%p,%p\n",FromTower->Window(),FromTower, ToTower, BufferTower); // Debug
	if (FromTower->Window()==NULL)  { printf(" No Window, back out.\n"); return;}
	// Note: The Views continue to exist (and work) 'offscreen', while the window might be
	// deleted. They are detatched from the window. As you can't see a lot without a window
	// to display the views, the solving is stopped then.

	//   b) Move the remaining Disc from F to T
	// printf("-Moving the Disc\n"); // Debug
	BMessage messy(OFFER_DISC);
		// Moving the Disc is done via the messaging mechanism and protocol, that is
		// used during the drag-and-drop-operations. So generate a message.
	messy.AddPointer("DiscPtr",FromTower->FirstDiscPtr());
		// Add a Pointer to the uppermost Disc of the 'source' Tower to the message
	ToTower->Window()->PostMessage(&messy,(BHandler*)ToTower,(BHandler*)FromTower);
		// That's the clue:
		// This posts a message to 'ToTower' (the Destination Tower for the Disc)
		// The message offers the uppermost disc of the 'FromTower'
		// the 2nd argument '(BHandler*)ToTower' specifies the ToTower BHandler
		//   (that is the (DiscTower-)View) as the target handler (ie, that view gets the message)
		// the 3rd argument '(BHandler*)FromTower' specifies the FromTower BHandler
		//   (again the (DiscTower-)View) as the reply handler (ie, that view gets any replies to the message)
		// This means, the ToTower 'thinks', the 'FromTower' did send the message and replies to it.
		// That's great, as we are out of the whole stuff and can leave the Towers do the rest.
	
	// printf("-Delaying\n");  // Debug
	snooze((103-SpeedSlider->Value())*15000); // Delay (value in microseconds)
	// This sets our thread (the ControlWindow thread) to sleep for some time
	// NOTE: that there is some delay necessary for the messages to be acted upon.
	//       If there's no delay, the whole thing will lock up!
	//       Probably, that's because the messages are posted to fast, that they cannot
	//       be replied to in time and the towers get inconsistent. 
	//       You will see one disc on two towers, when this happens!


	//   c) Move (N-1) Discs from B to T (the discs moved in a) )
	// printf("-Calling MoveTower again\n"); // Debug
	MoveTower(BufferTower, ToTower, FromTower, NumDiscs-1);
	}


// SolveGame solves the game
//  It relies on somebody (the application in our case) to have reset the towers,
//  as it depends on that, but it can't reset them itself.
//  While the solving is beeing done, it shows the window
//  After finishing the solving (or, when it's stopped somehow), it hides the window again
void ControlWindow::SolveGame(DiscTowerView *FromTower, DiscTowerView *ToTower, DiscTowerView *BufferTower) {
	// printf("Starting: Auto Solving\n"); // Debug
	StopFlag=false; // well, nice idea, isn't it.
	Show();
	MoveTower(FromTower,ToTower,BufferTower,NumberOfDiscs); // Do the magic call!
	StopFlag=false;
	Hide();
}

// StopSolving simply sets the Flag true.
//  When this flag is true, the recursive function (MoveTower) will returm immediately
//  and thus the solving will stop nearly at once.
void ControlWindow::StopSolving() {
	StopFlag=true;
	// printf("*** STOPFLAG set TRUE (StopSolving) ***\n"); // Debug
}

// Message handling for the window only handels the stop-button beeing pressed
void ControlWindow::MessageReceived(BMessage *message){
	switch(message->what) {
		case STOP_PRESSED: // Stop-button has been pressed down
			StopSolving();
		break;
	default: // pass all other messages along
		BWindow::MessageReceived(message);
		break;
		}
}
