//Name:		TetrisView.cpp
//Author:	Brian Tietz
//Copyright 1999
//Conventions:
//	Global constants (declared with const) and #defines - begin with "c_" followed by lowercase
//		words separated by underscores.
//		(E.G., #define c_my_constant 5).
//		(E.G., const int c_my_constant = 5;).
//	Global variables - begin with "g_" followed by lowercase words separated by underscores.
//		(E.G., int g_my_global;).
//	New data types (classes, structs, typedefs, etc.) - begin with an uppercase letter followed by
//		lowercase words separated by uppercase letters.  Enumerated constants contain a prefix
//		associating them with a particular enumerated set.
//		(E.G., typedef int MyTypedef;).
//		(E.G., enum MyEnumConst {c_mec_one, c_mec_two};)
//	Private member variables - begin with "m_" followed by lowercase words separated by underscores.
//		(E.G., int m_my_member;).
//	Public or friend-accessible member variables - all lowercase words separated by underscores.
//		(E.G., int public_member;).
//	Argument and local variables - begin with a lowercase letter followed by
//		lowercase words separated by underscores.  If the name is already taken by a public member
//		variable, prefix with a_ or l_
//		(E.G., int my_local; int a_my_arg, int l_my_local).
//	Functions (member or global) - begin with an uppercase letter followed by lowercase words
//		separated by uppercase letters.
//		(E.G., void MyFunction(void);).
//License:
//  Obviously, the idea for Tetris isn't mine, so I hold no copyright on the idea.  I do, however,
//  reserve all rights with regard to the source code for this BeOS version of Tetris.  The executable,
//  is freely distributable.  The source code is likewise freely distributable, provided that the
//  license and copyright are retained in these source code files and in the about box of the
//  executable.


//******************************************************************************************************
//**** System header files
//******************************************************************************************************
#include <Bitmap.h>


//******************************************************************************************************
//**** Project header files
//******************************************************************************************************
#include "TetrisView.h"
#include "TetrisResources.h"
#include "Colors.h"


//******************************************************************************************************
//**** Constants
//******************************************************************************************************
#if defined(__POWERPC__)
	//PPC: Big endian
	const int c_rgb_black = 0x000000FF;		//Black, alpha=255
#elif defined(__INTEL__)
	//INTEL: Little endian
	const int c_rgb_black = 0xFF000000;		//Black, alpha=255
#endif


//******************************************************************************************************
//**** TetrisView
//******************************************************************************************************
TetrisView::TetrisView(BRect frame, int8 width, int8 height, const BFont* font)
: BView(frame,NULL,B_FOLLOW_LEFT|B_FOLLOW_TOP,B_WILL_DRAW)
{
	m_grid = new int8[width*height];
	m_grid_width = width;
	m_grid_height = height;
	m_grid_bitmap = new BBitmap(BRect(0,0,(float)((width*16)-1),(float)((height*16)-1)),B_RGB32);
	m_grid_bitmap_data = (uint32*)m_grid_bitmap->Bits();
	Reset();
	SetViewColor(Black);
	SetFont(font);
	SetHighColor(White);
	struct font_height font_ht;
	font->GetHeight(&font_ht);
	m_font_ascent = ceil(font_ht.ascent);
	float font_height = m_font_ascent + ceil(font_ht.descent);
	float game_over_width = font->StringWidth(g_game_over_string);
	float game_over_x = floor((frame.Width()-game_over_width)/2);
	float game_over_y = floor((frame.Height()-font_height)/2.5);
	m_game_over_rect.Set(game_over_x,game_over_y,game_over_x+game_over_width,game_over_y+font_height);
}


void TetrisView::Draw(BRect update_rect)
{
	DrawBitmap(m_grid_bitmap,Bounds().LeftTop());
	if(m_game_over && update_rect.Intersects(m_game_over_rect))
	{
		SetDrawingMode(B_OP_OVER);
		DrawString(g_game_over_string,BPoint(m_game_over_rect.left,m_game_over_rect.top+m_font_ascent));
		SetDrawingMode(B_OP_COPY);
	}
}


bool TetrisView::IsBlockOKAt(int8 type, int8 center_x, int8 center_y, int8 rotation)
{
	if(type == 0 || m_game_over)
		return false;
	for(int8 x=0; x<4; x++)
		for(int8 y=0; y<4; y++)
		{
			int8 grid_x = center_x-2+x;
			int8 grid_y = center_y-2+y;
			if(grid_x>=0 && grid_x<m_grid_width && grid_y<m_grid_height)
			{
				//This position is in or above the grid
				if(grid_y>=0 && GridBrickAt(grid_x,grid_y) != 0 &&
					g_blocks[type-1].data[rotation][x][y] != 0)
				return false;
			}
			else
			{
				//This position is out of bounds
				if(g_blocks[type-1].data[rotation][x][y] != 0)
					return false;
			}
		}
	return true;
}


void TetrisView::Reset()
{
	for(int i=0; i<m_grid_width*m_grid_height; i++)
		m_grid[i] = 0;
	for(int y=0; y<m_grid_height*16; y++)
		for(int x=0; x<m_grid_width*16; x++)
			m_grid_bitmap_data[y*m_grid_width*16+x] = c_rgb_black;
	m_falling_block_type = 0;
	m_falling_block_x = 0;
	m_falling_block_y = 0;
	m_falling_block_rotation = 0;
	m_game_over = false;
	Invalidate();
}


bool TetrisView::AddFallingBlock(int8 type)
{
	m_falling_block_type = type;
	m_falling_block_x = m_grid_width/2;
	m_falling_block_y = -1;
	m_falling_block_rotation = 0;
	bool added_ok = IsBlockOKAt(m_falling_block_type,m_falling_block_x,m_falling_block_y,
		m_falling_block_rotation);
	if(!added_ok)
		m_falling_block_type = 0;
	else
	{
		BRect invalid;
		invalid.left = (float)((m_falling_block_x-2)*16);
		invalid.right = (float)((m_falling_block_x+2)*16-1);
		invalid.top = (float)((m_falling_block_y-2)*16);
		invalid.bottom = (float)((m_falling_block_y+2)*16-1);
		AddFallingBlockToBitmap();
		Invalidate(invalid);
	}
	return added_ok;
}


bool TetrisView::MoveFallingBlockDown()
{
	bool ok = IsBlockOKAt(m_falling_block_type,m_falling_block_x,m_falling_block_y+1,
		m_falling_block_rotation);
	if(ok)
	{
		BRect invalid;
		invalid.left = (float)((m_falling_block_x-2)*16);
		invalid.right = (float)((m_falling_block_x+2)*16-1);
		invalid.top = (float)((m_falling_block_y-2)*16);
		invalid.bottom = (float)((m_falling_block_y+3)*16-1);
		RemoveFallingBlockFromBitmap();
		m_falling_block_y++;
		AddFallingBlockToBitmap();
		Invalidate(invalid);
	}
	return ok;
}


void TetrisView::MoveFallingBlockLeft()
{
	if(IsBlockOKAt(m_falling_block_type,m_falling_block_x-1,m_falling_block_y,m_falling_block_rotation))
	{
		BRect invalid;
		invalid.left = (float)((m_falling_block_x-3)*16);
		invalid.right = (float)((m_falling_block_x+2)*16-1);
		invalid.top = (float)((m_falling_block_y-2)*16);
		invalid.bottom = (float)((m_falling_block_y+2)*16-1);
		RemoveFallingBlockFromBitmap();
		m_falling_block_x--;
		AddFallingBlockToBitmap();
		Invalidate(invalid);
	}
}


void TetrisView::MoveFallingBlockRight()
{
	if(IsBlockOKAt(m_falling_block_type,m_falling_block_x+1,m_falling_block_y,m_falling_block_rotation))
	{
		BRect invalid;
		invalid.left = (float)((m_falling_block_x-2)*16);
		invalid.right = (float)((m_falling_block_x+3)*16-1);
		invalid.top = (float)((m_falling_block_y-2)*16);
		invalid.bottom = (float)((m_falling_block_y+2)*16-1);
		RemoveFallingBlockFromBitmap();
		m_falling_block_x++;
		AddFallingBlockToBitmap();
		Invalidate(invalid);
	}
}


void TetrisView::RotateFallingBlock()
{
	int8 new_rotation = m_falling_block_rotation+1;
	if(new_rotation >= 4)
		new_rotation -= 4;

	if(IsBlockOKAt(m_falling_block_type,m_falling_block_x,m_falling_block_y,new_rotation))
	{
		BRect invalid;
		invalid.left = (float)((m_falling_block_x-2)*16);
		invalid.right = (float)((m_falling_block_x+2)*16-1);
		invalid.top = (float)((m_falling_block_y-2)*16);
		invalid.bottom = (float)((m_falling_block_y+2)*16-1);
		RemoveFallingBlockFromBitmap();
		m_falling_block_rotation = new_rotation;
		AddFallingBlockToBitmap();
		Invalidate(invalid);
	}
}


int8 TetrisView::AddFallingBlockToGrid()
{
	if(m_falling_block_type == 0)
		return -1;
	int8 x,x2;
	int8 y,y2;
	int8 lowest_falling_brick_y = 0;
	for(y=0; y<4; y++)
		for(x=0; x<4; x++)
			if(g_blocks[m_falling_block_type-1].data[m_falling_block_rotation][x][y] != 0)
			{
				int8 grid_x = m_falling_block_x-2+x;
				int8 grid_y = m_falling_block_y-2+y;
				SetGridBrickAt(grid_x,grid_y,m_falling_block_type);
				lowest_falling_brick_y = grid_y;
			}
	int8 last_removed_row = -1;
	int8 removed_rows = 0;
	for(y=0; y<=lowest_falling_brick_y; y++)
	{
		bool row_full = true;
		for(x=0; x<m_grid_width; x++)
			if(GridBrickAt(x,y) == 0)
			{
				row_full = false;
				break;
			}
		if(row_full)
		{
			for(y2=y; y2>0; y2--)
				for(x=0; x<m_grid_width; x++)
					SetGridBrickAt(x,y2,GridBrickAt(x,y2-1));
			for(x=0; x<m_grid_width; x++)
				SetGridBrickAt(x,0,0);
			removed_rows++;
			last_removed_row = y;
		}
	}
	m_falling_block_type = 0;
	if(last_removed_row > -1)
	{
		//Redraw the moved sections of the bitmap
		uint32* data = m_grid_bitmap_data;
		for(y=0; y<=last_removed_row; y++)
			for(y2=0; y2<16; y2++)
				for(x=0; x<m_grid_width; x++)
				{
					int8 type = GridBrickAt(x,y);
					if(type > 0)
						for(x2=0; x2<16; x2++)
						{
							*data = g_blocks[type-1].bitmap_data[x2][y2];
							data++;
						}
					else
						for(x2=0; x2<16; x2++)
						{
							*data = c_rgb_black;
							data++;
						}
				}
		BRect invalid;
		invalid.left = 0;
		invalid.right = (float)(m_grid_width*16-1);
		invalid.top = 0;
		invalid.bottom = (float)(last_removed_row*16+15);
		Invalidate(invalid);
	}
	return removed_rows;
}


void TetrisView::SetGameOver()
{
	m_game_over = true;
	Invalidate(m_game_over_rect);
}


void TetrisView::AddFallingBlockToBitmap()
{
	int32 bitmap_width = m_grid_width*16;
	for(int8 y=0; y<4; y++)
		for(int8 y2=0; y2<16; y2++)
		{
			for(int8 x=0; x<4; x++)
			{
				if(g_blocks[m_falling_block_type-1].data[m_falling_block_rotation][x][y] != 0)
				{
					int8 grid_x = m_falling_block_x-2+x;
					int8 grid_y = m_falling_block_y-2+y;
					if(grid_y >= 0)
					{
						uint32* data = &m_grid_bitmap_data[((((int32)grid_y)*16+((int32)y2))*
							bitmap_width)+((int32)grid_x*16)];
						for(int8 x2=0; x2<16; x2++)
						{
							*data = g_blocks[m_falling_block_type-1].bitmap_data[x2][y2];
							data++;
						}
					}
				}
			}
		}
}


void TetrisView::RemoveFallingBlockFromBitmap()
{
	
	int32 bitmap_width = m_grid_width*16;
	for(int8 y=0; y<4; y++)
		for(int8 y2=0; y2<16; y2++)
		{
			for(int8 x=0; x<4; x++)
			{
				if(g_blocks[m_falling_block_type-1].data[m_falling_block_rotation][x][y] != 0)
				{
					int8 grid_x = m_falling_block_x-2+x;
					int8 grid_y = m_falling_block_y-2+y;
					if(grid_y >= 0)
					{
						uint32* data = &m_grid_bitmap_data[((((int32)grid_y)*16+((int32)y2))*bitmap_width)+
							((int32)grid_x*16)];
						for(int8 x2=0; x2<16; x2++)
						{
							*data = c_rgb_black;
							data++;
						}
					}
				}
			}
		}
}

