/*********************************************************************
HyperTTT - Tic tac toe in higher dimensions
Copyright (c) 1998 Brian Nenninger

This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with 
this program; if not, write to the Free Software Foundation, Inc., 675 Mass 
Ave., Cambridge, MA 02139, USA.

You may contact me at bwn@kreative.net
************************************************************************/

#include "BeTTTView.h"
#include "BeTTTWindow.h"
#include "BeTTTApp.h"
#include "BeTTTGame.h"
#include "4dBoard.h"

extern int N,D;

BeTTTView::BeTTTView(BRect rect, char *name)
	   	   : BView(rect, name, B_FOLLOW_ALL, B_WILL_DRAW | B_FRAME_EVENTS |
	   	                                     (0*B_FULL_UPDATE_ON_RESIZE) | (0*B_PULSE_NEEDED))
{
  CountCells();
  //SetViewColor(B_TRANSPARENT_32_BIT);
  SetViewColor(255, 255, 255);
  m_lastHScale = m_lastVScale = 0;
  m_cellStyle = kSolidStyle;
}

void BeTTTView::SetGame(BeTTTGame *game)
{
  this->m_game = game;
}


void BeTTTView::CountCells()
{
  // calculate number of horizontal and vertical cells needed
  int i;
  for(i=1, m_vCells=1; i<D; i+=2)
    if (i+2<D)
      m_vCells *= N+1;
    else
      m_vCells *= N;
  for(i=0, m_hCells=1; i<D; i+=2)
    if (i+2<D)
      m_hCells *= N+1;
    else
      m_hCells *= N;
}


bool BeTTTView::UpdateScale()
{
  BRect b=Bounds();
  int i, hNum, vNum;
  CountCells();
  // scale cells to fit as much of frame as possible
  m_hScale = (b.right - b.left - 20) / m_hCells; 
  m_vScale = (b.bottom - b.top - 20) / m_vCells;
  
  if (m_hScale!=m_lastHScale || m_vScale!=m_lastVScale)
  {
    m_lastHScale = m_hScale;
    m_lastVScale = m_vScale;
    return true;
  }
  return false;
}

void BeTTTView::SetStyle(int newStyle)
{
  if (m_cellStyle != newStyle)
  {
    m_cellStyle = newStyle;
    Invalidate();
  }
}

int BeTTTView::GetStyle()
{
  return m_cellStyle;
}


BRect BeTTTView::ComputeCellRect(Vector &v)
{
  int hPos=kLeftGap, vPos=kTopGap;
  int incr;
  int i;
  // get vertical position
  for(i=1, incr=m_vScale; i<D; i+=2, incr*=N+1)
  {
    vPos += incr*v[i];
  }
  // get horizontal position
  for(i=0, incr=m_hScale; i<D; i+=2, incr*=N+1)
  {
    hPos += incr*v[i];
  }

  BRect r(hPos, vPos, hPos+m_hScale, vPos+m_vScale);
  
  return r;
}


void BeTTTView::DrawCell(Vector &v, int val)
{
  BRect r = ComputeCellRect(v);
    
  //FocusDraw();
  //m_boardPieces->DrawCell(val, r, index, 1);
  DrawPiece(r, this, val, m_cellStyle);
  
}


void DrawPiece(BRect r, BView *view, int val, int style)
{
  view->SetHighColor(0,0,0);
  view->StrokeRect(r);
  r.top++; r.left++; r.bottom--; r.right--;
  view->SetHighColor(255,255,255);
  view->FillRect(r);
  switch (style)
  {
    case kSolidStyle:
		  if (val==1)
		  {
		    view->SetHighColor(255,0,0);
		    view->FillRect(r);
		  }
		  else if (val==2)
		  {
		    view->SetHighColor(0,0,255);
		    view->FillRect(r);
		  }
		  break;
		  
		case kXOStyle:
		  if (val==1)
		  {
		    view->SetHighColor(255,0,0);
		    view->StrokeLine(BPoint(r.left, r.bottom), BPoint(r.right, r.top));
		    view->StrokeLine(BPoint(r.left, r.top), BPoint(r.right, r.bottom));
		  }
		  else if (val==2)
		  {
		    view->SetHighColor(0,0,255);
		    view->StrokeEllipse(r);
		  }
		  break;
  
  }
}



void BeTTTView::MouseDown(BPoint pt)
{
  Vector v;
  short i, hPos, vPos, divVal;
  if (m_game->OKForHuman())
  {
	  // get vector coords
	  hPos = pt.x - 10;
	  vPos = pt.y - 10;
	  for(i=1, divVal=m_vScale; i<D; i+=2)
	  {
	    v.Set(i, (vPos / divVal) % (N+1));
	    divVal *= N+1;
	  }
	  for(i=0, divVal=m_hScale; i<D; i+=2)
	  {
	    v.Set(i, (hPos / divVal) % (N+1));
	    divVal *= N+1;
	  }
	  if (v.IsLegal())
	  { // make the move
      m_game->DoHumanMove(v);
	  }
	}
}

void BeTTTView::FrameResized(float w, float h)
{
	BRect r = Bounds();
  if (UpdateScale())
  {
    SetHighColor(255,255,255);
	  FillRect(r);
	  m_game->LockWindow();
	  Draw(r);
	  m_game->UnlockWindow();
	}
}

void BeTTTView::NDChanged()
{
  UpdateScale();
  m_game->LockWindow();
  SetHighColor(255,255,255);
  FillRect(Bounds());
  m_game->UnlockWindow();
  Invalidate();
}

void BeTTTView::Draw(BRect)
{
  m_game->AcquireBoardSem();
  
  if (m_lastHScale==0)
  {
    UpdateScale();
  }
	Vector v;
	Board *b = m_game->GetBoard();
  // draw cell for each position
  for(v.Set0(); v.IsLegal(); v++)
  {
    DrawCell(v, (*b)[v]); 
  }
  
  m_game->ReleaseBoardSem();
}
