/*********************************************************************
HyperTTT - Tic tac toe in higher dimensions
Copyright (c) 1998 Brian Nenninger

This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with 
this program; if not, write to the Free Software Foundation, Inc., 675 Mass 
Ave., Cambridge, MA 02139, USA.

You may contact me at bwn@kreative.net
************************************************************************/

#include "BeTTTGame.h"
#include "BeTTTApp.h"
#include "BeTTTWindow.h"
#include "BeTTTView.h"
#include "4dBoard.h"

int N,D;

BeTTTGame::BeTTTGame(BeTTTApp *app, BeTTTWindow *wind, BeTTTView *view,
                     int startN=4, int startD=4)
{
  m_app  = app;
  m_wind = wind;
  m_view = view;
  
  m_playerMove = new Vector;
  m_flashLine = new ThreatLine;

  // set up game variables
  N = startN;
  D = startD;
  
  boardPtr = new Board;
  m_numPlayers= 2;
  players = new Player[m_numPlayers];
  players[0].strategy = HUMAN;
  int i;
  for(i=1; i<m_numPlayers; i++)
    players[i].strategy = BALANCED;
  boardPtr->Set0();
  m_InGame    = 0;
  m_numMoves  = 0;
  m_curPlayer = 0; 
  m_fastplay  = 0;
  // initialize wins+ties
  m_pWinList = new short[m_numPlayers+1];
  for(i=0; i<=m_numPlayers; i++)
    m_pWinList[i] = 0;

  m_okForHuman = 0;
  m_flashOn = 0;
  
  m_boardSem = create_sem(1, "Board Semaphore");
}

void BeTTTGame::LockWindow()
{
  m_wind->Lock();
}

void BeTTTGame::UnlockWindow()
{
  m_wind->Unlock();
}

void BeTTTGame::AcquireBoardSem()
{
  acquire_sem(m_boardSem);
}

void BeTTTGame::ReleaseBoardSem()
{
  release_sem(m_boardSem);
}

inline Player* BeTTTGame::GetPlayer(int i)
{
  return &(players[i]);
}

int BeTTTGame::GetPlayerWins(int i)
{
  return m_pWinList[i];
}

void BeTTTGame::ResetScore()
{
  for(int i=0; i<=m_numPlayers; i++)
    m_pWinList[i] = 0;
}

inline Player* BeTTTGame::GetCurrentPlayer()
{
  return &(players[m_curPlayer]);
}

void BeTTTGame::ResetGame()
{ // reinitializes game 
  int i, *stratList; 

  acquire_sem(m_boardSem);

  // stop flashing
  if (m_flashOn)
  {
    kill_thread(m_flashThread);
    m_flashOn = 0;
  }

  // save strategies
  stratList = new int[m_numPlayers];
  for(i=0; i<m_numPlayers; i++)
    stratList[i] = players[i].strategy;
  
  delete [] players;
  delete boardPtr;
  
  players = new Player[m_numPlayers];
  for(i=0; i<m_numPlayers; i++)
    players[i].strategy = stratList[i];
  boardPtr = new Board;
  boardPtr->Set0();
  m_numMoves  = 0;
  m_curPlayer = 0;
  /*
  Rect frame;
  m_mainWindow->CalcLocalFrameRect(frame);
  m_mainWindow->InvalPortRect(&frame);
  */
  delete [] stratList; 
  m_okForHuman = 0;

  release_sem(m_boardSem);
  
  //m_wind->SetTitle("In Game");
  snooze(20*1000.0);
}

void BeTTTGame::SetStrategy(int pNum, int newStrat)
{
  Player *p = &(players[pNum]);
  if (!m_InGame)
  {
    p->strategy = newStrat;
  }
  else
  {
    if (m_okForHuman && p==GetCurrentPlayer() && newStrat!=HUMAN)
    { // game thread was waiting on human, so resume
      p->strategy = newStrat;
      resume_thread(m_GameThread);
    }
    else
    { // it's ok to change strategies in mid-move
      p->strategy = newStrat;
    }
  }
}

void BeTTTGame::SetStyle(int newStyle)
{
  m_view->SetStyle(newStyle);
}

int BeTTTGame::GetStyle()
{
  return m_view->GetStyle();
}

void BeTTTGame::SetN(int newN)
{
  if (N != newN)
  {
    if (m_InGame)
    {
      long x;
      kill_thread(m_GameThread);
      wait_for_thread(m_GameThread, &x);
      m_InGame = 0;
    }
    else if (m_flashOn)
    {
      long x;
      kill_thread(m_flashThread);
      wait_for_thread(m_flashThread, &x);
      m_InGame = 0;
    }
    N = newN;
    ResetGame();
    m_view->NDChanged();
    m_wind->PostMessage(new BMessage(kNewGameMsg));
  }
}

void BeTTTGame::SetD(int newD)
{
  if (D != newD)
  {
    if (m_InGame)
    {
      long x;
      kill_thread(m_GameThread);
      wait_for_thread(m_GameThread, &x);
      m_InGame = 0;
    }
    else if (m_flashOn)
    {
      long x;
      kill_thread(m_flashThread);
      wait_for_thread(m_flashThread, &x);
      m_InGame = 0;
    }
    int oldD = D;
    D = newD;
    ResetGame();
    m_wind->DChanged(oldD, D, N);
    m_view->NDChanged();
    m_wind->PostMessage(new BMessage(kNewGameMsg));
  }
}

int BeTTTGame::DoHumanMove(Vector &v)
{ // returns 1 if move successful(into empty cell)
  if ((*boardPtr)[v]==EMPTY)
  { // make the move and wake up the game thread
    *m_playerMove = v;
    resume_thread(m_GameThread); 
    return 1;
  }
  else return 0;

}

void BeTTTGame::ToggleFastplay()
{
  m_fastplay = !m_fastplay;
}


void BeTTTGame::DoWin(int winner)
{
  m_pWinList[winner+1]++;
  m_wind->UpdateScore();
  if (FastplayOn())
  {
    ResetGame();
    m_wind->PostMessage(new BMessage(kNewGameMsg));
  }
  else
  {
    m_InGame = 0;
    if (winner!=kTie)
    {
      m_winnerNum = winner;
      *m_flashLine = GetPlayer(winner)->winLines.lineList[0];
      m_flashOn = 1;
      StartFlashing();
    }
  }
}

void BeTTTGame::StartFlashing()
{
  BeTTTGame *thisGame = this;
  m_flashThread = spawn_thread(BeTTTGame::StaticFlashEntryFunc, "Blinky", 
                               B_NORMAL_PRIORITY, (void *)thisGame);
  resume_thread(m_flashThread);
}

long BeTTTGame::StaticFlashEntryFunc(void *data)
{
  BeTTTGame *game = (BeTTTGame *)data;
  long result = game->FlashEntryFunc();
  
  return result;
}


long BeTTTGame::FlashEntryFunc()
{
  int i;
  Board *b = boardPtr;
  Vector v;
  int clearCells=0;
  while (1)
  {
    int newVal = (clearCells) ? EMPTY: m_winnerNum+1;
    v = m_flashLine->v;

    m_wind->Lock();
    for(i=0; i<N; i++)
    {
      b->Set(v, newVal);
      m_view->DrawCell(v, newVal);
      v += m_flashLine->dv;
    }
    m_view->Sync();
    m_wind->Unlock();

    clearCells = !clearCells;
    snooze(1000*1000.0);
  }
  return 0;
}

long BeTTTGame::GameEntryFunc()
{ // the game thread
  

  ResetGame();
  
  m_wind->Lock();
  m_view->Invalidate();
  m_wind->Unlock();

  Vector move;
  Player *p;
  int i, result;
  int done=0;
  int retVal;
  
  m_numMoves = 0;
  int numCells;
  for(numCells = N, i=1; i<D; i++)
  {
    numCells *= N;
  }
  
  Board *b = boardPtr;
  while (!done)
  {
    int humanMoved = 0;
    p = GetCurrentPlayer();  
      
    if (p->strategy == HUMAN)
    { // wait for the human's move
      m_okForHuman = 1;
      suspend_thread(find_thread(NULL));
      m_okForHuman = 0;
      // When resumed, we will have the player's move.
      // Or possibly the player changed to a computer setting, so
      // we'd better make sure
      
      if (p->strategy == HUMAN)
      {
        move = *m_playerMove;
        result = p->UpdateThreats(*b, m_curPlayer+1, &move);
        humanMoved = 1;
      }
      
    }
    
    if (!humanMoved)
    { // compute move for computer player
      result = p->MakeMove(*b, m_curPlayer+1, &move);
    }
    m_wind->Lock();
    m_view->DrawCell(move, m_curPlayer+1);
    //m_view->Invalidate();
    m_wind->Unlock();
 //    m_game->UpdateStatsWindow();
   
	  // have other players respond to move
	  for(i=0; i<m_numPlayers; i++)
	    if (i!=m_curPlayer)
	      GetPlayer(i)->RespondToMove(*b, m_curPlayer+1, &move);
	  if (result==WIN)
	  { // player wins
	    DoWin(m_curPlayer);
	    //EndGame(m_game->m_curPlayer);
	    retVal = m_curPlayer;
	    done = 1;
	    //m_wind->SetTitle("Win");
	  }
	  if (!done)
	  {
		  // update player counter and check for a tie game
		  m_curPlayer = (m_curPlayer+1) % m_numPlayers;
		  ++m_numMoves;
		  if (m_numMoves==numCells)
		  {
		    DoWin(kTie);
		    retVal = kTie;
		    done = 1;
		    //m_wind->SetTitle("Tie");
		  }
		}
  }
  
  m_InGame = 0;

  return retVal;
}



long BeTTTGame::StaticGameEntryFunc(void *data)
{
  BeTTTGame *game = (BeTTTGame *)data;
  long result = game->GameEntryFunc();
  
  return result;
}

void BeTTTGame::KillThread()
{
  if (m_InGame)
  {
    long x;
    kill_thread(m_GameThread);
    wait_for_thread(m_GameThread, &x);
    release_sem(m_boardSem);
    snooze(1000); // this seems to prevent a deadlock if a demented user holds down Cmd-N
  }
}

void BeTTTGame::StartGame()
{
  KillThread();
  ResetGame();
  m_view->Invalidate();
  m_InGame = 1;
  BeTTTGame *thisGame = this;
  m_GameThread = spawn_thread(BeTTTGame::StaticGameEntryFunc, "Game Thread", 
                              B_NORMAL_PRIORITY, (void *)thisGame);
  resume_thread(m_GameThread);
}