/***************************************************************************

1942 memory map (preliminary)

MAIN CPU:

0000-bfff ROM (8000-bfff banked)
cc00-cc7f Sprites
d000-d3ff Video RAM
d400-d7ff Color RAM
d800-dbff Background RAM (groups of 32 bytes, 16 code, 16 color/attribute)
e000-efff RAM

read:
c000      IN0
c001      IN1
c002      IN2
c003      DSW0
c004      DSW1
see the input_ports definition below for details on the input bits

write:
c800      command for the audio CPU
c802-c803 background scroll
c804      bit 7: flip screen
          bit 4: cpu B reset
		  bit 0: coin counter
c805      background palette bank selector
c806      bit 0-1 ROM bank selector 00=1-N5.BIN
                                    01=1-N6.BIN
                                    10=1-N7.BIN



SOUND CPU:

0000-3fff ROM
4000-47ff RAM
6000      command from the main CPU
8000      8910 #1 control
8001      8910 #1 write
c000      8910 #2 control
c001      8910 #2 write




The following doc was provided by Paul Leaman (paull@phonelink.com)


                                    1942

                            Hardware Description


                                Revision 0.4



INTRODUCTION
------------

This document describes the 1942 hardware. This will only be useful
to other emulator authors or for the curious.


LEGAL
-----

This document is freely distributable (with or without the emulator).
You may place it on a WEB page if you want.

You are free to use this information for whatever purpose you want providing
that:

* No profit is made
* You credit me somewhere in the documentation.
* The document is not changed




HARDWARE ARRANGEMENT
--------------------

1942 is a two board system. Board 1 contains 2 Z80A CPUs. One is used for
the sound, the other for the game.

The sound system uses 2 YM-2203 synth chips. These are compatible with the
AY-8910. Michael Cuddy has information (and source code) for these chips
on his Web page.

The second board contains the custom graphics hardware. There are three
graphics planes. The test screen refers to them as Scroll, Sprite and Tile.

The scroll and sprites are arranged in 16*16 blocks. The graphics ROMS
are not memory mapped. They are accessed directly by the hardware.

Early Capcom games seem to have similar hardware.


ROM descriptions:
=================

The generally available ROMs are as follows:

Board 1 - Code

Sound CPU:
1-C11.BIN       16K Sound ROM 0000-3fff

Main CPU:
1-N4.BIN        16K CODE ROM 0000-4000
1-N3.BIN        16K CODE ROM 4000-7fff
1-N5.BIN        16K CODE ROM 8000-bfff (paged)
1-N6.BIN         8K CODE ROM 8000-9fff (paged)
1-N7.BIN        16K CODE ROM 8000-bfff (paged)

1-F2.BIN         8K  Character ROM (Not mapped)

Board 2 - Graphics board
2-A5.BIN         8K TILE PLANE 1
2-A3.BIN         8K TILE PLANE 2
2-A1.BIN         8K TILE PLANE 3

2-A6.BIN         8K TILE PLANE 1
2-A4.BIN         8K TILE PLANE 2
2-A2.BIN         8K TILE PLANE 3

2-N1.BIN        16K OBJECT PLANE 1&2
2-L1.BIN        16K OBJECT PLANE 3&4

2-L2.BIN        16K OBJECT PLANE 1&2
2-N2.BIN        16K OBJECT PLANE 3&4


SOUND CPU MEMORY MAP
====================

0000-3fff Sound board CODE.
4000-47ff RAM data area and stack
6000      Sound input port. 0-0x1f
8000      PSG 1 Address
8001      PSG 1 Data
c000      PSG 2 Address
c001      PSG 2 Data

Runs in interrupt mode 1.

After initialization, Most of the sound code is driven by interrupt (0x38).
The code sits around waiting for the value in 0x6000 to change.

All Capcom games seem to share the same music hardware. The addresses of
the PSG chips and input vary according to the game.



MAIN CPU MEMORY MAP
===================

0000-bfff ROM Main code. Area (8000-bfff) is paged in

          Input ports
c000      Coin mech and start buttons
          0x10 Coin up
          0x08 Plater 4 start ????
          0x04 Player 3 start ????
          0x02 Player 2 start
          0x01 Player 1 start
c001      Joystick
c002      Joystick
c003      DIP switch 1 (1=off 0=on)
c004      DIP switch 2 (1=off 0=on)

          Output ports
c800      Sound output
c801      Unused
c802      Scroll register (lower 4 bits smooth, upper 4 bits rough)
c803      Scroll register MSB
c804      Watchdog circuit flip-flop ????
c805      Unknown
c806      Bits
            0-1 ROM paging   0=1-N5.BIN
                             1=1-N6.BIN
                             2=1-N7.BIN

          Video
cc00-cc7f Sprite RAM
          32 * 4 bytes

d000-d3ff Character RAM
d400-d7ff Character attribute
          Bits
             0x80 MSB character
             0x40
             0x20
             rest Attribute
d800-dbff Scroll RAM / attributes
             Alternating 16 byte rows of characters / attributes

             Attribute
                0x80 MSB tile
                0x40 Flip X
                0x20 Flip Y
                rest Attribute
e000-efff    RAM data / stack area
F000-FFFF    Unused

Game runs in interrupt mode 0 (the devices supply the interrupt number).

Two interrupts must be triggered per refresh for the game to function
correctly.

0x10 is the video retrace. This controls the speed of the game and generally
     drives the code. This must be triggerd for each video retrace.
0x08 is the sound card service interupt. The game uses this to throw sounds
     at the sound CPU.



Character RAM arrangement
-------------------------

The characters are rotated 90 degrees vertically. Each column is 32 bytes.

The attributes are arranged so that they correspond to each column.

Attribute
      0x80  Char MSB Set to get characters from 0x100 to 0x1ff
      0x40  Unknown
      rest  Character palette colour.


Tile system
-----------

Tiles are arranged in rotational buffer. Each line consists of 32 bytes.
The first 16 bytes are the tile values, the second 16 bytes contain the
attributes.

This arrangement may vary according to the machine. For example, Vulgus,
which is a horizontal/ vertical scroller uses 32 bytes per line. The
attributes are in a separate block of memory. This is probably done to
accommodate horizontal scrolling.

      0x80 Tile MSB (Set to obtain tiles 0x100-0x1ff.
      0x40 Tile flip X
      0x20 Tile Flip Y
      rest Palette colour scheme.

The scroll rough register determines the starting point for the bottom of the
screen. The buffer is circular. The bottom of the screen is at the start of
tile memory. To address the start of a line:

     lineaddress=(roughscroll * 0x20) & 0x3ff
     attributeaddress=lineaddress+0x10

Make sure to combine the MSB value to make up the rough scroll address.


Sprite arrangement
------------------

Sprites are 16*16 blocks. Attribute bits determine whether or not the
sprite is wider.

32 Sprites. 4 bytes for each sprite
    00 Sprite number 0-255
    01 Colour Code
         0x80 Sprite number MSB (256-512)
         0x40 Sprite size 16*64 (Very wide sprites)
         0x20 Sprite size 16*32 (Wide sprites)
         0x10 Sprite Y position MSB
         rest colour code
    02 Sprite X position
    03 Sprite Y position

The sprite sequence is slightly odd within the sprite data ROMS. It is
necessary to swap sprites 0x0080-0x00ff with sprites 0x0100-0x017f to get
the correct order. This is best done at load-up time.

If 0x40 is set, the next 4 sprite objects are combined into one from left
to right.
if 0x20 is set, the next 2 sprite objects are combined into one from left
to right.
if none of the above bits are set, the sprite is a simple 16*16 object.


Sprite clipping:
----------------

The title sprites are supposed to appear to move into each other. I
have not yet found the mechanism to do this.


Palette:
--------

Palette system is in hardware. There are 16 colours for each component (char,
scroll and object).

The .PAL file format used by the emulator is as follows:

Offset
0x0000-0x000f Char 16 colour palette colours
0x0010-0x001f Scroll 16 colour palette colours
0x0020-0x002f Object 16 colour palette colours
0x0030-0x00f0 Char colour schemes (3 bytes each). Values are 0-0xff
0x00f0-0x01ef Object colour scheme (16 bytes per scheme)
0x01f0-0x02ef Scroll colour scheme (8 bytes per scheme)

Note that only half the colour schemes are shown on the scroll palette
screen. The game uses colour values that are not shown. There must
be some mirroring here. Either that, or I have got the scheme wrong.



Interesting RAM locations:
--------------------------

0xE09B - Number of rolls



Graphics format:
----------------

Roberto Ventura has written a document, detailing the Ghosts 'n' Goblins
graphics layout. This can be found on the repository.


Schematics:
-----------

Schematics for Commando can be found at:

This game is fairly similar to 1942.



DIP Switch Settings
-------------------


WWW.SPIES.COM contains DIP switch settings.


***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "sndhrdw/generic.h"
#include "sndhrdw/8910intf.h"



void c1942_bankswitch_w(int offset,int data);
int c1942_bankedrom_r(int offset);
int c1942_interrupt(void);

extern unsigned char *c1942_backgroundram;
extern int c1942_backgroundram_size;
extern unsigned char *c1942_scroll;
extern unsigned char *c1942_palette_bank;
void c1942_background_w(int offset,int data);
void c1942_palette_bank_w(int offset,int data);
void c1942_flipscreen_w(int offset,int data);
int c1942_vh_start(void);
void c1942_vh_stop(void);
void c1942_vh_convert_color_prom(unsigned char *palette, unsigned char *colortable,const unsigned char *color_prom);
void c1942_vh_screenrefresh(struct osd_bitmap *bitmap);

int capcom_sh_start(void);
int capcom_sh_interrupt(void);



static struct MemoryReadAddress readmem[] =
{
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0xbfff, c1942_bankedrom_r },
	{ 0xc000, 0xc000, input_port_0_r },	/* IN0 */
	{ 0xc001, 0xc001, input_port_1_r },	/* IN1 */
	{ 0xc002, 0xc002, input_port_2_r },	/* IN2 */
	{ 0xc003, 0xc003, input_port_3_r },	/* DSW0 */
	{ 0xc004, 0xc004, input_port_4_r },	/* DSW1 */
	{ 0xd000, 0xdbff, MRA_RAM },
	{ 0xe000, 0xefff, MRA_RAM },
	{ -1 }	/* end of table */
};

static struct MemoryWriteAddress writemem[] =
{
	{ 0x0000, 0xbfff, MWA_ROM },
	{ 0xc800, 0xc800, soundlatch_w },
	{ 0xc802, 0xc803, MWA_RAM, &c1942_scroll },
	{ 0xc804, 0xc804, c1942_flipscreen_w },
	{ 0xc805, 0xc805, c1942_palette_bank_w, &c1942_palette_bank },
	{ 0xc806, 0xc806, c1942_bankswitch_w },
	{ 0xcc00, 0xcc7f, MWA_RAM, &spriteram, &spriteram_size },
	{ 0xd000, 0xd3ff, videoram_w, &videoram, &videoram_size },
	{ 0xd400, 0xd7ff, colorram_w, &colorram },
	{ 0xd800, 0xdbff, c1942_background_w, &c1942_backgroundram, &c1942_backgroundram_size },
	{ 0xe000, 0xefff, MWA_RAM },
	{ -1 }	/* end of table */
};



static struct MemoryReadAddress sound_readmem[] =
{
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x47ff, MRA_RAM },
	{ 0x6000, 0x6000, soundlatch_r },
	{ -1 }	/* end of table */
};

static struct MemoryWriteAddress sound_writemem[] =
{
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x47ff, MWA_RAM },
	{ 0x8000, 0x8000, AY8910_control_port_0_w },
	{ 0x8001, 0x8001, AY8910_write_port_0_w },
	{ 0xc000, 0xc000, AY8910_control_port_1_w },
	{ 0xc001, 0xc001, AY8910_write_port_1_w },
	{ -1 }	/* end of table */
};



INPUT_PORTS_START( input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */

	PORT_START	/* IN2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )	/* probably unused */

	PORT_START	/* DSW0 */
	PORT_DIPNAME( 0x07, 0x07, "Coin A", IP_KEY_NONE )
	PORT_DIPSETTING(    0x01, "4 Coins/1 Credit" )
	PORT_DIPSETTING(    0x02, "3 Coins/1 Credit" )
	PORT_DIPSETTING(    0x04, "2 Coins/1 Credit" )
	PORT_DIPSETTING(    0x07, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x03, "2 Coins/3 Credits" )
	PORT_DIPSETTING(    0x06, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0x05, "1 Coin/4 Credits" )
	PORT_DIPSETTING(    0x00, "Free Play" )
	PORT_DIPNAME( 0x08, 0x00, "Cabinet", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Upright" )
	PORT_DIPSETTING(    0x08, "Cocktail" )
	PORT_DIPNAME( 0x30, 0x30, "Bonus Life", IP_KEY_NONE )
	PORT_DIPSETTING(    0x30, "20000 80000" )
	PORT_DIPSETTING(    0x20, "20000 100000" )
	PORT_DIPSETTING(    0x10, "30000 80000" )
	PORT_DIPSETTING(    0x00, "30000 100000" )
	PORT_DIPNAME( 0xc0, 0xc0, "Lives", IP_KEY_NONE )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0x40, "2" )
	PORT_DIPSETTING(    0xc0, "3" )
	PORT_DIPSETTING(    0x00, "5" )

	PORT_START	/* DSW1 */
	PORT_DIPNAME( 0x07, 0x07, "Coin B", IP_KEY_NONE )
	PORT_DIPSETTING(    0x01, "4 Coins/1 Credit" )
	PORT_DIPSETTING(    0x02, "3 Coins/1 Credit" )
	PORT_DIPSETTING(    0x04, "2 Coins/1 Credit" )
	PORT_DIPSETTING(    0x07, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x03, "2 Coins/3 Credits" )
	PORT_DIPSETTING(    0x06, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0x05, "1 Coin/4 Credits" )
	PORT_DIPSETTING(    0x00, "Free Play" )
	PORT_BITX(    0x08, 0x08, IPT_DIPSWITCH_NAME | IPF_TOGGLE, "Service Mode", OSD_KEY_F2, IP_JOY_NONE, 0 )
	PORT_DIPSETTING(    0x08, "Off" )
	PORT_DIPSETTING(    0x00, "On" )
	PORT_DIPNAME( 0x10, 0x10, "Flip Screen", IP_KEY_NONE )
	PORT_DIPSETTING(    0x10, "Off" )
	PORT_DIPSETTING(    0x00, "On" )
	PORT_DIPNAME( 0x60, 0x60, "Difficulty", IP_KEY_NONE )
	PORT_DIPSETTING(    0x40, "Easy" )
	PORT_DIPSETTING(    0x60, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x00, "Hardest" )
	PORT_DIPNAME( 0x80, 0x80, "Freeze", IP_KEY_NONE )
	PORT_DIPSETTING(    0x80, "Off" )
	PORT_DIPSETTING(    0x00, "On" )
INPUT_PORTS_END



static struct GfxLayout charlayout =
{
	8,8,	/* 8*8 characters */
	512,	/* 512 characters */
	2,	/* 2 bits per pixel */
	{ 4, 0 },
	{ 0, 1, 2, 3, 8+0, 8+1, 8+2, 8+3 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16 },
	16*8	/* every char takes 16 consecutive bytes */
};
static struct GfxLayout tilelayout =
{
	16,16,	/* 16*16 tiles */
	512,	/* 512 tiles */
	3,	/* 3 bits per pixel */
	{ 0, 512*32*8, 2*512*32*8 },	/* the bitplanes are separated */
	{ 0, 1, 2, 3, 4, 5, 6, 7,
			16*8+0, 16*8+1, 16*8+2, 16*8+3, 16*8+4, 16*8+5, 16*8+6, 16*8+7 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8,
			8*8, 9*8, 10*8, 11*8, 12*8, 13*8, 14*8, 15*8 },
	32*8	/* every tile takes 32 consecutive bytes */
};
static struct GfxLayout spritelayout =
{
	16,16,	/* 16*16 sprites */
	512,	/* 512 sprites */
	4,	/* 4 bits per pixel */
	{ 512*64*8+4, 512*64*8+0, 4, 0 },
	{ 0, 1, 2, 3, 8+0, 8+1, 8+2, 8+3,
			32*8+0, 32*8+1, 32*8+2, 32*8+3, 33*8+0, 33*8+1, 33*8+2, 33*8+3 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16,
			8*16, 9*16, 10*16, 11*16, 12*16, 13*16, 14*16, 15*16 },
	64*8	/* every sprite takes 64 consecutive bytes */
};



static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ 1, 0x00000, &charlayout,             0, 64 },
	{ 1, 0x02000, &tilelayout,          64*4, 4*32 },
	{ 1, 0x0e000, &spritelayout, 64*4+4*32*8, 16 },
	{ -1 } /* end of array */
};



/* these are NOT the original color PROMs */
static unsigned char color_prom[] =
{
	/* 08E_SB-5: palette red component */
	0x00,0x00,0x00,0x04,0x06,0x07,0x00,0x08,0x05,0x00,0x08,0x00,0x07,0x00,0x00,0x00,
	0x00,0x00,0x00,0x04,0x06,0x07,0x00,0x08,0x05,0x06,0x08,0x07,0x00,0x05,0x00,0x00,
	0x00,0x09,0x08,0x06,0x05,0x04,0x07,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x07,0x06,0x07,0x00,0x08,0x05,0x06,0x08,0x00,0x07,0x00,0x00,0x00,
	0x00,0x0B,0x09,0x07,0x05,0x03,0x0C,0x0A,0x08,0x06,0x04,0x0E,0x0E,0x0D,0x07,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x08,0x05,0x0B,0x0A,0x0A,0x0B,0x0C,0x0C,0x0E,0x0C,0x00,0x03,0x04,0x06,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x08,0x05,0x0B,0x0A,0x0A,0x0B,0x0C,0x0C,0x0E,0x0C,0x00,0x03,0x04,0x06,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	/* 09E_SB-6: palette green component */
	0x00,0x05,0x08,0x06,0x05,0x06,0x07,0x08,0x04,0x05,0x07,0x07,0x07,0x06,0x05,0x06,
	0x00,0x05,0x08,0x06,0x05,0x06,0x07,0x08,0x05,0x06,0x07,0x07,0x04,0x06,0x05,0x06,
	0x00,0x09,0x08,0x07,0x06,0x05,0x06,0x07,0x05,0x06,0x04,0x00,0x00,0x00,0x05,0x06,
	0x00,0x05,0x08,0x06,0x05,0x06,0x07,0x08,0x04,0x05,0x07,0x07,0x07,0x06,0x05,0x06,
	0x00,0x0E,0x0C,0x0A,0x07,0x05,0x0C,0x0A,0x08,0x06,0x04,0x0C,0x09,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0A,0x09,0x0B,0x05,0x09,0x00,0x05,0x09,0x0E,0x0C,0x0A,0x0D,0x06,0x0A,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0A,0x09,0x0B,0x05,0x09,0x00,0x05,0x09,0x0E,0x0C,0x0A,0x0D,0x06,0x0A,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	/* 10E_SB-7: palette blue component */
	0x00,0x00,0x00,0x00,0x04,0x05,0x00,0x08,0x03,0x00,0x06,0x08,0x07,0x07,0x08,0x09,
	0x00,0x00,0x00,0x00,0x04,0x05,0x00,0x08,0x05,0x06,0x06,0x07,0x05,0x06,0x08,0x09,
	0x00,0x09,0x08,0x07,0x06,0x05,0x05,0x06,0x08,0x09,0x07,0x00,0x00,0x00,0x08,0x09,
	0x00,0x00,0x00,0x05,0x04,0x05,0x00,0x08,0x03,0x04,0x06,0x08,0x07,0x07,0x08,0x09,
	0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x07,0x05,0x03,0x01,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0D,0x0B,0x0B,0x08,0x0F,0x00,0x00,0x00,0x0E,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x0D,0x0B,0x0B,0x08,0x0F,0x00,0x00,0x00,0x0E,0x00,0x00,0x0B,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	/* F01_SB-0: char lookup table */
	0x0F,0x01,0x02,0x03,0x0F,0x02,0x03,0x04,0x0F,0x03,0x04,0x05,0x0F,0x04,0x05,0x06,
	0x0F,0x05,0x06,0x07,0x0F,0x06,0x07,0x08,0x0F,0x07,0x08,0x09,0x0F,0x08,0x09,0x0A,
	0x0F,0x09,0x0A,0x0B,0x0F,0x0A,0x0B,0x0C,0x0F,0x0B,0x0C,0x0D,0x0F,0x0C,0x0D,0x0E,
	0x0F,0x0D,0x0E,0x0F,0x0F,0x0E,0x0F,0x01,0x0F,0x0F,0x01,0x02,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x00,0x00,0x00,0x0F,0x01,0x01,0x01,0x0F,0x02,0x02,0x02,0x0F,0x03,0x03,0x03,
	0x0F,0x04,0x04,0x04,0x0F,0x05,0x05,0x05,0x0F,0x06,0x06,0x06,0x0F,0x07,0x07,0x07,
	0x0F,0x08,0x08,0x08,0x0F,0x09,0x09,0x09,0x0F,0x0A,0x0A,0x0A,0x0F,0x0B,0x0B,0x0B,
	0x0F,0x0C,0x0C,0x0C,0x0F,0x0D,0x0D,0x0D,0x0F,0x0E,0x0E,0x0E,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x03,0x01,0x04,0x0F,0x04,0x03,0x05,0x0F,0x05,0x04,0x06,0x0F,0x06,0x05,0x07,
	0x0F,0x07,0x06,0x08,0x0F,0x08,0x07,0x0A,0x0F,0x0A,0x08,0x0B,0x0F,0x0B,0x0A,0x0E,
	0x0F,0x0E,0x0B,0x0D,0x0F,0x0D,0x0E,0x0C,0x0F,0x0C,0x0D,0x09,0x0F,0x09,0x0C,0x02,
	0x0F,0x02,0x09,0x01,0x0F,0x01,0x02,0x03,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x09,0x0D,0x0E,0x0F,0x04,0x0D,0x0A,0x0F,0x01,0x02,0x09,0x0F,0x01,0x00,0x03,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x09,0x09,0x09,0x0F,0x0A,0x0A,0x0A,0x0F,0x0A,0x06,0x02,0x0F,0x0C,0x0F,0x0F,
	/* 06D_SB-4: tile lookup table */
	0x00,0x01,0x02,0x03,0x04,0x05,0x08,0x0A,0x00,0x01,0x07,0x0C,0x04,0x05,0x0E,0x0F,
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x0A,0x00,0x01,0x0D,0x03,0x04,0x0E,0x09,0x0F,
	0x00,0x01,0x0A,0x03,0x04,0x05,0x06,0x08,0x00,0x01,0x02,0x0E,0x0F,0x05,0x06,0x07,
	0x00,0x01,0x08,0x03,0x0B,0x0E,0x06,0x0F,0x00,0x01,0x07,0x0B,0x09,0x0E,0x0D,0x0F,
	0x00,0x01,0x0F,0x0C,0x04,0x05,0x06,0x07,0x00,0x01,0x0A,0x0F,0x04,0x0E,0x06,0x07,
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x09,0x00,0x0A,0x02,0x09,0x08,0x05,0x06,0x07,
	0x00,0x08,0x0A,0x03,0x04,0x05,0x0D,0x0B,0x00,0x01,0x02,0x03,0x0C,0x05,0x06,0x0D,
	0x00,0x01,0x02,0x03,0x05,0x08,0x09,0x0A,0x00,0x08,0x02,0x09,0x04,0x05,0x06,0x07,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x00,0x0D,0x08,0x09,0x0A,0x0B,0x0C,0x0E,
	/* 03K_SB-8: sprite lookup table */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x06,0x08,0x0D,0x0E,0x0F,
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x08,0x06,0x0D,0x0E,0x0F,
	0x00,0x06,0x07,0x08,0x09,0x0A,0x06,0x07,0x08,0x09,0x0A,0x06,0x08,0x0D,0x0E,0x0F,
	0x00,0x06,0x07,0x08,0x09,0x0A,0x06,0x07,0x08,0x09,0x0A,0x08,0x06,0x0D,0x0E,0x0F,
	0x00,0x0B,0x0C,0x0D,0x0E,0x0E,0x06,0x07,0x08,0x09,0x0A,0x06,0x08,0x0D,0x0E,0x0F,
	0x00,0x0B,0x0C,0x0D,0x0E,0x0E,0x06,0x07,0x08,0x09,0x0A,0x08,0x06,0x0D,0x0E,0x0F,
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
	0x00,0x06,0x07,0x08,0x09,0x0A,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,
	0x00,0x06,0x08,0x09,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
	0x00,0x08,0x06,0x09,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
};



static struct MachineDriver machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_Z80,
			4000000,	/* 4 Mhz (?) */
			0,
			readmem,writemem,0,0,
			c1942_interrupt,2
		},
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			3000000,	/* 3 Mhz ??? */
			2,	/* memory region #2 */
			sound_readmem,sound_writemem,0,0,
			capcom_sh_interrupt,12
		}
	},
	60,
	10,	/* 10 CPU slices per frame - enough for the sound CPU to read all commands */
	0,

	/* video hardware */
	32*8, 32*8, { 0*8, 32*8-1, 2*8, 30*8-1 },
	gfxdecodeinfo,
	256,64*4+4*32*8+16*16,
	c1942_vh_convert_color_prom,

	VIDEO_TYPE_RASTER,
	0,
	c1942_vh_start,
	c1942_vh_stop,
	c1942_vh_screenrefresh,

	/* sound hardware */
	0,
	0,
	capcom_sh_start,
	AY8910_sh_stop,
	AY8910_sh_update
};



/***************************************************************************

  Game driver(s)

***************************************************************************/

ROM_START( c1942_rom )
	ROM_REGION(0x1c000)	/* 64k for code + 3*16k for the banked ROMs images */
	ROM_LOAD( "1-n3.bin",  0x0000, 0x4000, 0x33ff9663 )
	ROM_LOAD( "1-n4.bin",  0x4000, 0x4000, 0xd492925e )
	ROM_LOAD( "1-n7.bin",  0x8000, 0x4000, 0xc9728db2 )	/* this is the only one of the banked ROMs containing code */
	ROM_LOAD( "1-n5.bin", 0x10000, 0x4000, 0xf2525fe2 )
	ROM_LOAD( "1-n6.bin", 0x14000, 0x2000, 0x659736b3 )
	ROM_LOAD( "1-n7.bin", 0x18000, 0x4000, 0xc9728db2 )

	ROM_REGION(0x1e000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "1-f2.bin", 0x00000, 0x2000, 0x90e7df87 )	/* characters */
	ROM_LOAD( "2-a1.bin", 0x02000, 0x2000, 0x144b5325 )	/* tiles */
	ROM_LOAD( "2-a2.bin", 0x04000, 0x2000, 0xbeecefb2 )
	ROM_LOAD( "2-a3.bin", 0x06000, 0x2000, 0x6649e7e3 )
	ROM_LOAD( "2-a4.bin", 0x08000, 0x2000, 0x33e7a52d )
	ROM_LOAD( "2-a5.bin", 0x0a000, 0x2000, 0x606bfaa1 )
	ROM_LOAD( "2-a6.bin", 0x0c000, 0x2000, 0xcb9343fd )
	ROM_LOAD( "2-l1.bin", 0x0e000, 0x4000, 0x3de5e617 )	/* sprites */
	ROM_LOAD( "2-l2.bin", 0x12000, 0x4000, 0x900c4178 )
	ROM_LOAD( "2-n1.bin", 0x16000, 0x4000, 0x8b02cefe )
	ROM_LOAD( "2-n2.bin", 0x1a000, 0x4000, 0x0fb73451 )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "1-c11.bin", 0x0000, 0x4000, 0xb16cd20e )
ROM_END



static int hiload(void)
{
	/* get RAM pointer (this game is multiCPU, we can't assume the global */
	/* RAM pointer is pointing to the right place) */
	unsigned char *RAM = Machine->memory_region[0];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0xe801],"\x00\x04\x00\x00",4) == 0 &&
			memcmp(&RAM[0xe981],"\x00\x00\x01\x00",4) == 0)
	{
		void *f;


		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			int i;


			osd_fread(f,&RAM[0xe800],16*25);
			osd_fread(f,&RAM[0xe9c0],1);
			/* find the high score */
			for (i = 0;i < 16*25;i += 16)
			{
				if (RAM[0xe800 + i] == 0x00)
				{
					RAM[0xe040] = RAM[0xe801 + i] >> 4;
					RAM[0xe041] = RAM[0xe801 + i] & 0x0f;
					RAM[0xe042] = RAM[0xe802 + i] >> 4;
					RAM[0xe043] = RAM[0xe802 + i] & 0x0f;
					RAM[0xe044] = RAM[0xe803 + i] >> 4;
					RAM[0xe045] = RAM[0xe803 + i] & 0x0f;
					RAM[0xe046] = RAM[0xe804 + i] >> 4;
					RAM[0xe047] = RAM[0xe804 + i] & 0x0f;

					break;
				}
			}
			osd_fclose(f);
		}

		return 1;
	}
	else return 0;	/* we can't load the hi scores yet */
}



static void hisave(void)
{
	void *f;
	/* get RAM pointer (this game is multiCPU, we can't assume the global */
	/* RAM pointer is pointing to the right place) */
	unsigned char *RAM = Machine->memory_region[0];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0xe800],16*25);
		osd_fwrite(f,&RAM[0xe9c0],1);
		osd_fclose(f);
	}
}



struct GameDriver c1942_driver =
{
	"1942",
	"1942",
	"Paul Leaman (hardware info)\nNicola Salmoria (MAME driver)",
	&machine_driver,

	c1942_rom,
	0, 0,
	0,

	0/*TBR*/,input_ports,0/*TBR*/,0/*TBR*/,0/*TBR*/,

	color_prom, 0, 0,
	ORIENTATION_ROTATE_270,

	hiload, hisave
};
